<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\ZatcaSettings;
use Carbon\Carbon;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Schema;
use Saleh7\Zatca\InvoiceSigner;
use Saleh7\Zatca\Helpers\Certificate;
use App\Services\ZatcaCertificateValidator;

/**
 * رقم 201061989778 واتس
 */
class ZatcaService
{
    protected $client;
    protected $settings;

    public function __construct(ZatcaSettings $settings)
    {
        $this->settings = $settings;
        $this->client = new Client([
            'base_uri' => $this->settings->getBaseUrl(),
            'timeout' => 30,
            'verify' => false, // للبيئة التجريبية
        ]);
    }

    /**
     * اختبار الاتصال مع ZATCA
     */
    public function testConnection(): array
    {
        try {
            Log::info('Testing ZATCA connection', [
                'environment' => $this->settings->environment,
                'base_url' => $this->settings->getBaseUrl(),
            ]);

            // اختبار الاتصال الأساسي أولاً
            $response = $this->client->get('/', [
                'timeout' => 10,
                'connect_timeout' => 5,
            ]);

            $this->settings->updateConnectionStatus('connected');

            return [
                'success' => true,
                'message' => 'تم الاتصال بنجاح مع خادم ZATCA',
                'data' => [
                    'status_code' => $response->getStatusCode(),
                    'environment' => $this->settings->environment,
                    'base_url' => $this->settings->getBaseUrl(),
                    'timestamp' => now()->toISOString()
                ]
            ];
        } catch (RequestException $e) {
            $statusCode = $e->getResponse() ? $e->getResponse()->getStatusCode() : null;
            $responseBody = $e->getResponse() ? $e->getResponse()->getBody()->getContents() : null;

            Log::warning('ZATCA connection test response', [
                'status_code' => $statusCode,
                'response_body' => $responseBody,
                'error' => $e->getMessage(),
            ]);

            // تحليل أنواع الأخطاء المختلفة
            if ($statusCode === 401) {
                $this->settings->updateConnectionStatus('connected', 'Server reachable but authentication required');
                return [
                    'success' => true,
                    'message' => 'الخادم متاح لكن يتطلب مصادقة (هذا طبيعي)',
                    'data' => [
                        'status_code' => $statusCode,
                        'note' => 'Server reachable, authentication required'
                    ]
                ];
            }

            if ($statusCode === 404) {
                $this->settings->updateConnectionStatus('connected', 'Server reachable but endpoint not found');
                return [
                    'success' => true,
                    'message' => 'الخادم متاح (404 هو استجابة طبيعية للجذر)',
                    'data' => [
                        'status_code' => $statusCode,
                        'note' => 'Server reachable, root endpoint returns 404 (normal)'
                    ]
                ];
            }

            if ($statusCode >= 500) {
                $this->settings->updateConnectionStatus('error', 'Server error: ' . $statusCode);
                return [
                    'success' => false,
                    'message' => 'خطأ في خادم ZATCA: ' . $statusCode,
                    'data' => [
                        'status_code' => $statusCode,
                        'error_type' => 'server_error'
                    ]
                ];
            }

            // أي استجابة أخرى تعني أن الخادم متاح
            if ($statusCode) {
                $this->settings->updateConnectionStatus('connected', 'Server responded with: ' . $statusCode);
                return [
                    'success' => true,
                    'message' => 'خادم ZATCA متاح ومتجاوب',
                    'data' => [
                        'status_code' => $statusCode,
                        'note' => 'Server is responding'
                    ]
                ];
            }

            // خطأ في الشبكة أو الاتصال
            $this->settings->updateConnectionStatus('error', $e->getMessage());
            return [
                'success' => false,
                'message' => 'فشل في الاتصال بخادم ZATCA: ' . $e->getMessage(),
                'data' => [
                    'error_type' => 'network_error',
                    'base_url' => $this->settings->getBaseUrl()
                ]
            ];
        } catch (Exception $e) {
            $this->settings->updateConnectionStatus('error', $e->getMessage());
            Log::error('ZATCA connection test failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => 'خطأ عام في اختبار الاتصال: ' . $e->getMessage(),
                'data' => [
                    'error_type' => 'general_error'
                ]
            ];
        }
    }

    /**
     * الحصول على رمز الوصول
     */
    public function getAccessToken(): ?string
    {
        if (!$this->settings->isTokenExpired() && $this->settings->access_token) {
            return $this->settings->access_token;
        }

        try {
            // استخدام Basic Authentication بدلاً من OAuth للبيئة التطويرية
            if ($this->settings->environment === 'sandbox') {
                // في البيئة التطويرية، قد لا نحتاج لـ OAuth token
                // يمكن استخدام Basic Auth أو شهادات مباشرة
                return 'sandbox_token'; // رمز وهمي للاختبار
            }

            $response = $this->client->post('/oauth2/token', [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ],
                'form_params' => [
                    'grant_type' => 'client_credentials',
                    'client_id' => $this->settings->client_id,
                    'client_secret' => $this->settings->client_secret,
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            // فحص أن البيانات صحيحة قبل محاولة الوصول لها
            if (is_array($data) && isset($data['access_token'])) {
                $this->settings->access_token = $data['access_token'];
                $this->settings->token_expires_at = now()->addSeconds($data['expires_in'] ?? 3600);
                $this->settings->save();

                return $this->settings->access_token;
            } else {
                Log::error('Invalid OAuth response format from ZATCA', ['response' => $data]);
                return null;
            }
        } catch (Exception $e) {
            Log::error('Failed to get ZATCA access token: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * إرسال فاتورة للتطهير (Clearance)
     */
    public function clearInvoice(Invoice $invoice, string $signedXmlContent): array
    {
        try {
            if (empty($invoice->invoice_hash)) {
                throw new \Exception('invoice_hash غير موجود. يجب توليده بعد توقيع XML.');
            }

            // البيئة والرابط
            $baseUrl = $this->settings->environment === 'sandbox'
                ? 'https://gw-fatoora.zatca.gov.sa/e-invoicing/simulation'
                : 'https://gw-apic-gov.gazt.gov.sa/e-invoicing/core';

            // إعداد المصادقة
            if ($this->settings->environment === 'sandbox') {
                $username = $this->settings->username;
                $password = $this->settings->password;

                if (empty($username) || empty($password)) {
                    throw new \Exception('بيانات دخول البيئة التجريبية ناقصة.');
                }

                $authHeader = 'Basic ' . base64_encode("{$username}:{$password}");
            } else {
                $token = $this->getAccessToken();
                if (!$token) {
                    throw new \Exception('فشل في الحصول على رمز الوصول من ZATCA.');
                }

                $authHeader = 'Bearer ' . $token;
            }

            // الرؤوس
            $headers = [
                'Accept'           => 'application/json',
                'Content-Type'     => 'application/json',
                'Accept-Version'   => 'V2',
                'Accept-Language'  => 'en',
                'Authorization'    => $authHeader,
            ];

            // بيانات الطلب
            $endpoint = $baseUrl . '/invoices/clearance/single';
            $payload = [
                'invoiceHash' => $invoice->invoice_hash,
                'uuid'        => $invoice->uuid,
                'invoice'     => base64_encode($signedXmlContent),
            ];

            // تنفيذ الطلب
            $response = $this->client->post($endpoint, [
                'headers' => $headers,
                'json'    => $payload,
                'timeout' => 60,
            ]);

            $status = $response->getStatusCode();
            $body   = (string) $response->getBody();
            $data   = json_decode($body, true);
            $settingsLink = url("/zatca-settings/{$this->settings->id}/edit");

            // التعامل مع الاستجابة
            switch ($status) {
                case 200:
                    if (
                        isset($data['clearanceStatus']) &&
                        $data['clearanceStatus'] === 'CLEARED' &&
                        !empty($data['clearedInvoice']['uuid'])
                    ) {
                        $invoice->status = 'cleared';
                        $invoice->irn = $data['clearedInvoice']['uuid'];
                        $invoice->zatca_response = $data;

                        if (!empty($data['clearedInvoice']['invoice'])) {
                            $zatcaXml = base64_decode($data['clearedInvoice']['invoice']);
                            $path = "invoices/zatca_signed/{$invoice->invoice_number}/zatca.xml";
                            Storage::put($path, $zatcaXml);
                            $invoice->signed_xml_path = $path;
                        }
                    } else {
                        $invoice->status = 'rejected';
                        $invoice->zatca_response = $data;
                    }

                    $invoice->sent_to_zatca_at = now();
                    $invoice->save();

                    $this->settings->logOperation('clearance', [
                        'invoice_id' => $invoice->id,
                        'status'     => $invoice->status,
                        'response'   => $data,
                    ]);

                    return $data;

                case 401:
                    throw new \Exception("⚠️ 401 Unauthorized: تحقق من بيانات الدخول. <a href=\"{$settingsLink}\" target=\"_blank\">تعديل الإعدادات</a>");

                case 400:
                    throw new \Exception("❌ 400 Bad Request: تحقق من صحة XML أو بيانات الفاتورة.");

                case 403:
                    throw new \Exception("🚫 403 Forbidden: ليس لديك صلاحية لإرسال هذه الفاتورة.");

                case 404:
                    throw new \Exception("🔍 404 Not Found: نقطة النهاية غير صحيحة.");

                case 415:
                    throw new \Exception("⚠️ 415 Unsupported Media Type: تأكد أن Content-Type هو application/json.");

                case 429:
                    throw new \Exception("⏱️ 429 Too Many Requests: الرجاء الانتظار ثم المحاولة.");

                case 500:
                    throw new \Exception("💥 500 Internal Server Error: خطأ داخلي من جهة ZATCA.");

                case 503:
                    throw new \Exception("🛠️ 503 Service Unavailable: الخدمة غير متوفرة مؤقتًا.");

                default:
                    throw new \Exception("❗ خطأ غير متوقع (Status Code: {$status})");
            }
        } catch (RequestException $e) {
            return $this->handleZatcaError($e, $invoice, 'clearance');
        } catch (\Exception $e) {
            Log::error('⛔ ZATCA clearance error', [
                'invoice_id' => $invoice->id ?? null,
                'error'      => $e->getMessage(),
            ]);

            throw new \Exception('فشل إرسال الفاتورة إلى ZATCA: ' . $e->getMessage());
        }
    }

    protected function handleZatcaError(RequestException $e, Invoice $invoice, string $operation = 'unknown'): array
    {
        $statusCode = $e->getResponse()?->getStatusCode();
        $errorResponse = $e->getResponse()?->getBody()?->getContents();

        Log::error("ZATCA {$operation} failed", [
            'invoice_id' => $invoice->id,
            'status_code' => $statusCode,
            'error' => $e->getMessage(),
            'response' => $errorResponse,
            'url' => $this->settings->getBaseUrl() . "/invoices/{$operation}/single",
        ]);

        $errorData = null;
        if ($errorResponse) {
            $errorData = json_decode($errorResponse, true);
        }

        $invoice->status = 'rejected';
        $invoice->zatca_response = $errorData ?: ['error' => $e->getMessage()];
        $invoice->sent_to_zatca_at = now();
        $invoice->save();

        throw new Exception('ZATCA ' . $operation . ' failed: ' . ($errorData['message'] ?? $e->getMessage()));
    }

    /**
     * إرسال فاتورة للإبلاغ (Reporting)
     */
    public function reportInvoice(Invoice $invoice): array
    {
        try {
            if (empty($invoice->invoice_hash)) {
                throw new \Exception('invoice_hash غير موجود. يجب توليده بعد توقيع XML.');
            }

            // 1. توليد XML الموقع للفاتورة
            $signedXml = $this->generateInvoiceXml($invoice);
            $encodedXml = base64_encode($signedXml);

            // 2. رابط البيئة
            $baseUrl = $this->settings->environment === 'sandbox'
                ? 'https://gw-fatoora.zatca.gov.sa/e-invoicing/simulation'
                : 'https://gw-apic-gov.gazt.gov.sa/e-invoicing/core';

            $endpoint = $baseUrl . '/invoices/reporting/single';

            // 3. المصادقة
            if ($this->settings->environment === 'sandbox') {
                $username = $this->settings->username;
                $password = $this->settings->password;

                if (empty($username) || empty($password)) {
                    throw new \Exception('بيانات الدخول غير مكتملة.');
                }

                $authHeader = 'Basic ' . base64_encode("{$username}:{$password}");
            } else {
                $token = $this->getAccessToken();
                if (!$token) {
                    throw new \Exception('فشل في الحصول على رمز الوصول.');
                }

                $authHeader = 'Bearer ' . $token;
            }

            // 4. الهيدر
            $headers = [
                'Accept'           => 'application/json',
                'Content-Type'     => 'application/json',
                'Accept-Version'   => 'V2',
                'Accept-Language'  => 'en',
                'Authorization'    => $authHeader,
            ];

            // 5. البيانات
            $payload = [
                'invoiceHash' => $invoice->invoice_hash,
                'uuid'        => $invoice->uuid,
                'invoice'     => $encodedXml,
            ];

            // 6. إرسال الطلب
            $response = $this->client->post($endpoint, [
                'headers' => $headers,
                'json'    => $payload,
                'timeout' => 60,
            ]);

            $status = $response->getStatusCode();
            $body = (string) $response->getBody();
            $data = json_decode($body, true);
            $settingsLink = url("/zatca-settings/{$this->settings->id}/edit");

            // 7. التعامل مع الاستجابة
            switch ($status) {
                case 200:
                    $invoice->status = 'reported';
                    $invoice->zatca_response = $data;

                    if (!empty($data['reportedInvoice']['invoice'])) {
                        $zatcaXml = base64_decode($data['reportedInvoice']['invoice']);
                        $path = "invoices/zatca_reported/{$invoice->invoice_number}/zatca.xml";
                        Storage::put($path, $zatcaXml);
                        $invoice->signed_xml_path = $path;
                    }

                    $invoice->sent_to_zatca_at = now();
                    $invoice->save();

                    $this->settings->logOperation('reporting', [
                        'invoice_id' => $invoice->id,
                        'status'     => 'reported',
                        'response'   => $data,
                    ]);

                    return $data;

                case 400:
                    throw new \Exception("❌ 400 Bad Request: تحقق من صحة XML أو بيانات الفاتورة.");

                case 401:
                    throw new \Exception("⚠️ 401 Unauthorized: تحقق من بيانات الدخول. <a href=\"{$settingsLink}\" target=\"_blank\">تعديل الإعدادات</a>");

                case 403:
                    throw new \Exception("🚫 403 Forbidden: لا تملك صلاحية للوصول إلى الخدمة.");

                case 500:
                    throw new \Exception("💥 500 Internal Server Error: مشكلة في خوادم ZATCA.");

                default:
                    throw new \Exception("❗ خطأ غير متوقع من ZATCA. الكود: {$status}");
            }
        } catch (RequestException $e) {
            return $this->handleZatcaError($e, $invoice, 'reporting');
        } catch (\Exception $e) {
            Log::error('⛔ ZATCA reporting error', [
                'invoice_id' => $invoice->id,
                'error'      => $e->getMessage(),
            ]);

            $invoice->zatca_response = ['error' => $e->getMessage()];
            $invoice->sent_to_zatca_at = now();
            $invoice->save();

            throw new \Exception('فشل إرسال الفاتورة المبسطة إلى ZATCA: ' . $e->getMessage());
        }
    }

    /**
     * توليد XML للفاتورة - يستخدم الآن الطريقة المحسنة
     */
    public function generateInvoiceXml(Invoice $invoice): string
    {
        // توليد XML خام (UBL) حسب النوع
        if ($invoice->type === 'simplified') {
            $xmlString = $this->generateSimplifiedUblInvoiceXml($invoice);
        } elseif ($invoice->type === 'standard') {
            $xmlString = $this->generateUblInvoiceXml($invoice);
        } else {
            throw new \InvalidArgumentException("نوع الفاتورة غير معروف: {$invoice->type}");
        }

        // إنشاء مجلد تخزين خاص بالفاتورة
        $tempDir = storage_path('app/invoices/' . $invoice->invoice_number);
        if (!is_dir($tempDir)) {
            mkdir($tempDir, 0777, true);
        }

        // حفظ XML الأصلي
        $originalXmlPath = $tempDir . '/original.xml';
        file_put_contents($originalXmlPath, $xmlString);

        // مسار XML الموقّع النهائي
        $signedXmlPath = $tempDir . '/signed.xml';

        // التوقيع
        $this->signInvoiceWithFatoora($originalXmlPath, $signedXmlPath);

        // قراءة XML الموقّع
        $signedXml = file_get_contents($signedXmlPath);

        // حساب الهاش وتحديث الفاتورة
        $invoice->invoice_hash = $this->calculateInvoiceHash($signedXml);
        $invoice->save();

        // (اختياري) حذف الملف المؤقت من أداة التوقيع إن كان مختلفًا
        // @unlink($signedXmlPath); ← لا يُفضل حذفه إلا لو لا تحتاج الأرشفة

        return $signedXml;
    }

    public function signInvoiceWithFatoora(string $inputPath, string $outputPath): void
    {
        try {
            // Enhanced certificate validation before signing
            if (empty($this->settings->certificate) || empty($this->settings->private_key)) {
                throw new \Exception('شهادة التوقيع أو المفتاح الخاص غير متوفر في إعدادات ZATCA');
            }

            // Validate certificate and private key compatibility
            $certificateValidator = new ZatcaCertificateValidator();
            $keyPairValidation = $certificateValidator->validateCertificateKeyPair(
                $this->settings->certificate,
                $this->settings->private_key,
                $this->settings->certificate_secret ?? null
            );

            if (!$keyPairValidation['compatible']) {
                Log::error('الشهادة والمفتاح الخاص غير متوافقين', [
                    'error' => $keyPairValidation['error']
                ]);
                throw new \Exception('Certificate and private key are not compatible: ' . $keyPairValidation['error']);
            }

            Log::info('تم التحقق من توافق الشهادة والمفتاح الخاص بنجاح');

            // Read the XML content
            if (!file_exists($inputPath)) {
                throw new \Exception("لم يتم العثور على ملف XML المدخل: {$inputPath}");
            }

            $xmlContent = file_get_contents($inputPath);
            if ($xmlContent === false) {
                throw new \Exception("فشل في قراءة ملف XML: {$inputPath}");
            }

            // Validate XML content before signing
            if (strlen($xmlContent) < 100) {
                throw new \Exception("محتوى XML قصير جداً للتوقيع");
            }

            // Create Certificate object with enhanced validation
            try {
                $certificate = new Certificate(
                    $this->settings->certificate,
                    $this->settings->private_key,
                    $this->settings->certificate_secret ?? ''
                );
            } catch (\Exception $e) {
                Log::error('فشل في إنشاء كائن الشهادة', [
                    'error' => $e->getMessage()
                ]);
                throw new \Exception('Failed to create certificate object: ' . $e->getMessage());
            }

            // Sign the invoice using PHP library
            $invoiceSigner = InvoiceSigner::signInvoice($xmlContent, $certificate);
            
            // Get the signed XML
            $signedXml = $invoiceSigner->getXML();

            // Validate signed XML before saving
            if (empty($signedXml) || strlen($signedXml) < strlen($xmlContent)) {
                throw new \Exception('الـ XML الموقع فارغ أو أقصر من الأصل');
            }

            // Save the signed XML
            $result = file_put_contents($outputPath, $signedXml);
            if ($result === false) {
                throw new \Exception("فشل في حفظ ملف XML الموقع: {$outputPath}");
            }

            // Verify the saved file
            if (!file_exists($outputPath) || filesize($outputPath) === 0) {
                throw new \Exception("الملف المحفوظ فارغ أو غير موجود: {$outputPath}");
            }

            Log::info('تم توقيع الفاتورة بنجاح باستخدام مكتبة PHP مع التحقق المحسن', [
                'input' => $inputPath,
                'output' => $outputPath,
                'input_size' => filesize($inputPath),
                'output_size' => filesize($outputPath),
                'hash' => $invoiceSigner->getHash(),
                'certificate_subject' => $keyPairValidation['certificate_info']['subject']['CN'] ?? 'Unknown'
            ]);

        } catch (\Exception $e) {
            Log::error('فشل توقيع الفاتورة باستخدام مكتبة PHP', [
                'input' => $inputPath,
                'output' => $outputPath,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Fall back to a basic signing approach if the main library fails
            $this->fallbackSignInvoice($inputPath, $outputPath);
        }
    }

    /**
     * Fallback signing method using basic XML manipulation
     */
    private function fallbackSignInvoice(string $inputPath, string $outputPath): void
    {
        try {
            $xmlContent = file_get_contents($inputPath);
            if ($xmlContent === false) {
                throw new \Exception("فشل في قراءة ملف XML: {$inputPath}");
            }

            // For now, just copy the file and add a basic signature placeholder
            // This is a temporary solution - in production, proper signing should be implemented
            $dom = new \DOMDocument('1.0', 'UTF-8');
            $dom->loadXML($xmlContent);
            $dom->formatOutput = true;

            // Add basic signature information if not present
            $signatureElements = $dom->getElementsByTagName('Signature');
            if ($signatureElements->length === 0) {
                // Add proper namespaces to root element if not present
                $root = $dom->documentElement;
                
                // Ensure the UBL namespaces are properly defined
                if (!$root->hasAttribute('xmlns:cac')) {
                    $root->setAttribute('xmlns:cac', 'urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2');
                }
                if (!$root->hasAttribute('xmlns:cbc')) {
                    $root->setAttribute('xmlns:cbc', 'urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2');
                }
                if (!$root->hasAttribute('xmlns:ds')) {
                    $root->setAttribute('xmlns:ds', 'http://www.w3.org/2000/09/xmldsig#');
                }
                
                // Add a basic signature block with proper namespace
                $signature = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2', 'cac:Signature');
                $signatureId = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2', 'cbc:ID', 'urn:oasis:names:specification:ubl:signature:Invoice');
                $signatureMethod = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2', 'cbc:SignatureMethod', 'urn:oasis:names:specification:ubl:dsig:enveloped:xades');
                
                $signature->appendChild($signatureId);
                $signature->appendChild($signatureMethod);
                $root->appendChild($signature);
            }

            $result = file_put_contents($outputPath, $dom->saveXML());
            if ($result === false) {
                throw new \Exception("فشل في حفظ ملف XML: {$outputPath}");
            }

            \Log::warning('تم استخدام الطريقة البديلة للتوقيع (بدون توقيع رقمي فعلي)', [
                'input' => $inputPath,
                'output' => $outputPath,
            ]);

        } catch (\Exception $e) {
            \Log::error('فشل في الطريقة البديلة للتوقيع', [
                'error' => $e->getMessage(),
            ]);
            throw new \Exception("فشل التوقيع عبر Fatoora: " . $e->getMessage());
        }
    }

    /**
     * توليد QR Code متوافق مع ZATCA Phase 2
     * يتضمن جميع العناصر المطلوبة للمرحلة الثانية: التوقيع الرقمي، المفتاح العام، والطابع الرقمي
     */
    public function generateQRCode(Invoice $invoice): string
    {
        try {
            $qrData = [];

            // Tag 1: Seller Name (مطلوب)
            $qrData[1] = $invoice->company->name ?? 'Unknown Seller';

            // Tag 2: VAT Number (مطلوب)
            $qrData[2] = $invoice->company->vat_number ?? '000000000000000';

            // Tag 3: Invoice Timestamp (مطلوب) - ISO 8601 format
            $qrData[3] = Carbon::parse($invoice->issue_date)->format('Y-m-d\TH:i:s\Z');

            // Tag 4: Invoice Total Amount (مطلوب)
            $qrData[4] = number_format($invoice->total_amount, 2, '.', '');

            // Tag 5: VAT Amount (مطلوب)
            $qrData[5] = number_format($invoice->tax_amount, 2, '.', '');

            // المرحلة الثانية - عناصر إضافية مطلوبة للتوافق الكامل

            // Tag 6: Invoice Hash (مطلوب للمرحلة الثانية)
            if (!empty($invoice->invoice_hash)) {
                $qrData[6] = $invoice->invoice_hash;
            } else {
                // إذا لم يكن موجود، سنحاول إنشاءه مؤقتاً
                Log::warning('QR Code: Invoice hash missing, generating temporary hash', [
                    'invoice_id' => $invoice->id
                ]);
                $qrData[6] = hash('sha256', $invoice->invoice_number . $invoice->issue_date->timestamp);
            }

            // Tag 7: Digital Signature (مطلوب للمرحلة الثانية)
            $digitalSignature = $this->generateDigitalSignatureForQR($invoice);
            if (!empty($digitalSignature)) {
                $qrData[7] = $digitalSignature;
            }

            // Tag 8: Public Key (مطلوب للمرحلة الثانية)
            $publicKey = $this->extractPublicKeyForQR();
            if (!empty($publicKey)) {
                $qrData[8] = $publicKey;
            }

            // Tag 9: Cryptographic Stamp (مطلوب للمرحلة الثانية)
            $cryptographicStamp = $this->generateCryptographicStampForQR($invoice);
            if (!empty($cryptographicStamp)) {
                $qrData[9] = $cryptographicStamp;
            }

            // بناء TLV (Tag-Length-Value) format
            $tlv = '';
            foreach ($qrData as $tag => $value) {
                if (empty($value)) {
                    Log::warning("QR Code: Empty value for tag {$tag}", [
                        'invoice_id' => $invoice->id,
                        'tag' => $tag
                    ]);
                    continue;
                }

                $encoded = mb_convert_encoding($value, 'UTF-8');
                $length = mb_strlen($encoded, '8bit');
                
                // التأكد من أن الطول لا يتجاوز 255 بايت
                if ($length > 255) {
                    Log::warning("QR Code: Value too long for tag {$tag}, truncating", [
                        'invoice_id' => $invoice->id,
                        'tag' => $tag,
                        'original_length' => $length
                    ]);
                    $encoded = mb_substr($encoded, 0, 255, '8bit');
                    $length = 255;
                }

                $tlv .= chr($tag) . chr($length) . $encoded;
            }

            // التحقق من حجم QR Code النهائي
            $finalSize = strlen($tlv);
            if ($finalSize > 1000) {
                Log::warning('QR Code exceeds recommended size limit', [
                    'invoice_id' => $invoice->id,
                    'size' => $finalSize,
                    'limit' => 1000
                ]);
            }

            $base64Result = base64_encode($tlv);
            
            Log::info('QR Code generated successfully for Phase 2', [
                'invoice_id' => $invoice->id,
                'tlv_size' => $finalSize,
                'base64_length' => strlen($base64Result),
                'tags_included' => array_keys($qrData)
            ]);

            return $base64Result;

        } catch (\Exception $e) {
            Log::error('Failed to generate Phase 2 QR Code', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // عودة إلى QR Code مبسط في حالة الفشل
            return $this->generateFallbackQRCode($invoice);
        }
    }

    /**
     * توليد التوقيع الرقمي لـ QR Code
     */
    private function generateDigitalSignatureForQR(Invoice $invoice): ?string
    {
        try {
            // محاولة الحصول على التوقيع من استجابة ZATCA أولاً
            if (!empty($invoice->zatca_response['digitalSignature'])) {
                return $invoice->zatca_response['digitalSignature'];
            }

            // إذا لم يكن موجود، إنشاء توقيع باستخدام المفتاح الخاص
            if (!empty($this->settings->private_key)) {
                $dataToSign = $invoice->invoice_hash ?? 
                             hash('sha256', $invoice->invoice_number . $invoice->issue_date->timestamp);
                
                $privateKey = openssl_pkey_get_private($this->settings->private_key);
                if ($privateKey === false) {
                    Log::warning('Invalid private key for QR digital signature');
                    return null;
                }

                $signature = '';
                $success = openssl_sign($dataToSign, $signature, $privateKey, OPENSSL_ALGO_SHA256);
                
                if ($success) {
                    return base64_encode($signature);
                }
            }

            Log::warning('Unable to generate digital signature for QR Code', [
                'invoice_id' => $invoice->id
            ]);
            
            return null;

        } catch (\Exception $e) {
            Log::error('Error generating digital signature for QR', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * استخراج المفتاح العام من الشهادة لـ QR Code
     */
    private function extractPublicKeyForQR(): ?string
    {
        try {
            // محاولة الحصول على المفتاح العام من الإعدادات أولاً
            if (!empty($this->settings->public_key)) {
                return $this->settings->public_key;
            }

            // استخراج المفتاح العام من الشهادة
            if (!empty($this->settings->certificate)) {
                $cert = openssl_x509_read($this->settings->certificate);
                if ($cert === false) {
                    Log::warning('Invalid certificate for extracting public key');
                    return null;
                }

                $publicKey = openssl_pkey_get_public($cert);
                if ($publicKey === false) {
                    Log::warning('Unable to extract public key from certificate');
                    return null;
                }

                $keyDetails = openssl_pkey_get_details($publicKey);
                if (isset($keyDetails['key'])) {
                    // إزالة headers وتنسيق المفتاح للـ QR Code
                    $publicKeyPem = $keyDetails['key'];
                    $publicKeyBase64 = preg_replace('/-----BEGIN PUBLIC KEY-----|-----END PUBLIC KEY-----|\s+/', '', $publicKeyPem);
                    return $publicKeyBase64;
                }
            }

            Log::warning('No certificate available for extracting public key');
            return null;

        } catch (\Exception $e) {
            Log::error('Error extracting public key for QR', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * توليد الطابع الرقمي (Cryptographic Stamp) لـ QR Code
     */
    private function generateCryptographicStampForQR(Invoice $invoice): string
    {
        try {
            // محاولة الحصول على الطابع من استجابة ZATCA أولاً
            if (!empty($invoice->zatca_response['cryptographicStamp'])) {
                return $invoice->zatca_response['cryptographicStamp'];
            }

            // استخدام UUID الفاتورة كطابع رقمي
            if (!empty($invoice->uuid)) {
                return $invoice->uuid;
            }

            // إنشاء طابع رقمي مؤقت باستخدام بيانات الفاتورة
            $stampData = $invoice->invoice_number . 
                        $invoice->company_id . 
                        $invoice->issue_date->timestamp . 
                        ($invoice->invoice_hash ?? '');
            
            return hash('sha256', $stampData);

        } catch (\Exception $e) {
            Log::error('Error generating cryptographic stamp for QR', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            
            // عودة إلى قيمة افتراضية
            return hash('sha256', $invoice->invoice_number . time());
        }
    }

    /**
     * QR Code مبسط كحل بديل في حالة فشل المرحلة الثانية
     */
    private function generateFallbackQRCode(Invoice $invoice): string
    {
        try {
            $qrData = [];

            // فقط العناصر الأساسية المطلوبة
            $qrData[1] = $invoice->company->name ?? 'Unknown Seller';
            $qrData[2] = $invoice->company->vat_number ?? '000000000000000';
            $qrData[3] = Carbon::parse($invoice->issue_date)->format('Y-m-d\TH:i:s\Z');
            $qrData[4] = number_format($invoice->total_amount, 2, '.', '');
            $qrData[5] = number_format($invoice->tax_amount, 2, '.', '');

            $tlv = '';
            foreach ($qrData as $tag => $value) {
                $encoded = mb_convert_encoding($value, 'UTF-8');
                $length = mb_strlen($encoded, '8bit');
                $tlv .= chr($tag) . chr($length) . $encoded;
            }

            Log::info('Fallback QR Code generated', [
                'invoice_id' => $invoice->id
            ]);

            return base64_encode($tlv);

        } catch (\Exception $e) {
            Log::error('Failed to generate fallback QR Code', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            
            // أبسط QR Code ممكن
            return base64_encode("Basic QR for invoice: " . $invoice->invoice_number);
        }
    }

    /**
     * اختبار توليد QR Code للمرحلة الثانية
     */
    public function testPhase2QRCodeGeneration(Invoice $invoice): array
    {
        try {
            Log::info('Testing Phase 2 QR Code generation', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number
            ]);

            // تحضير البيانات المطلوبة للاختبار
            $this->prepareInvoiceForQRTest($invoice);

            // توليد QR Code
            $qrCodeBase64 = $this->generateQRCode($invoice);

            if (empty($qrCodeBase64)) {
                throw new \Exception('فشل في توليد QR Code - البيانات فارغة');
            }

            // فك تشفير وتحليل محتوى QR Code
            $analysis = $this->analyzeQRCodeContent($qrCodeBase64);

            // التحقق من متطلبات المرحلة الثانية
            $phase2Compliance = $this->validatePhase2QRCompliance($analysis);

            $result = [
                'success' => true,
                'qr_code_base64' => $qrCodeBase64,
                'qr_code_length' => strlen($qrCodeBase64),
                'tlv_size' => strlen(base64_decode($qrCodeBase64)),
                'analysis' => $analysis,
                'phase2_compliance' => $phase2Compliance,
                'timestamp' => now()->toISOString()
            ];

            Log::info('Phase 2 QR Code test completed successfully', $result);

            return $result;

        } catch (\Exception $e) {
            $error = [
                'success' => false,
                'error' => $e->getMessage(),
                'invoice_id' => $invoice->id,
                'timestamp' => now()->toISOString()
            ];

            Log::error('Phase 2 QR Code test failed', $error);

            return $error;
        }
    }

    /**
     * تحضير الفاتورة لاختبار QR Code
     */
    private function prepareInvoiceForQRTest(Invoice $invoice): void
    {
        // التأكد من وجود UUID
        if (empty($invoice->uuid)) {
            $invoice->uuid = (string) Str::uuid();
        }

        // التأكد من وجود invoice_hash (مؤقت للاختبار)
        if (empty($invoice->invoice_hash)) {
            $invoice->invoice_hash = hash('sha256', $invoice->invoice_number . $invoice->issue_date->timestamp);
        }

        // محاكاة استجابة ZATCA للاختبار
        if (empty($invoice->zatca_response)) {
            $invoice->zatca_response = [
                'digitalSignature' => base64_encode('test_digital_signature_' . $invoice->id),
                'publicKey' => base64_encode('test_public_key_' . $invoice->id),
                'cryptographicStamp' => $invoice->uuid
            ];
        }

        $invoice->save();
    }

    /**
     * تحليل محتوى QR Code
     */
    private function analyzeQRCodeContent(string $qrCodeBase64): array
    {
        try {
            $tlvData = base64_decode($qrCodeBase64);
            $analysis = [
                'total_size' => strlen($tlvData),
                'tags_found' => [],
                'tags_data' => []
            ];

            $offset = 0;
            while ($offset < strlen($tlvData)) {
                if ($offset + 2 > strlen($tlvData)) {
                    break;
                }

                $tag = ord($tlvData[$offset]);
                $length = ord($tlvData[$offset + 1]);
                
                if ($offset + 2 + $length > strlen($tlvData)) {
                    break;
                }

                $value = substr($tlvData, $offset + 2, $length);
                
                $analysis['tags_found'][] = $tag;
                $analysis['tags_data'][$tag] = [
                    'length' => $length,
                    'value' => mb_convert_encoding($value, 'UTF-8', 'UTF-8'),
                    'value_preview' => mb_substr($value, 0, 50) . ($length > 50 ? '...' : '')
                ];

                $offset += 2 + $length;
            }

            return $analysis;

        } catch (\Exception $e) {
            return [
                'error' => 'فشل في تحليل محتوى QR Code: ' . $e->getMessage(),
                'total_size' => strlen(base64_decode($qrCodeBase64)),
                'tags_found' => [],
                'tags_data' => []
            ];
        }
    }

    /**
     * التحقق من التوافق مع متطلبات المرحلة الثانية
     */
    private function validatePhase2QRCompliance(array $analysis): array
    {
        $compliance = [
            'is_compliant' => true,
            'required_tags' => [1, 2, 3, 4, 5], // العناصر الأساسية
            'phase2_tags' => [6, 7, 8, 9], // عناصر المرحلة الثانية
            'missing_required' => [],
            'missing_phase2' => [],
            'size_compliance' => false,
            'messages' => []
        ];

        // فحص العناصر المطلوبة
        foreach ($compliance['required_tags'] as $tag) {
            if (!in_array($tag, $analysis['tags_found'] ?? [])) {
                $compliance['missing_required'][] = $tag;
                $compliance['is_compliant'] = false;
            }
        }

        // فحص عناصر المرحلة الثانية
        foreach ($compliance['phase2_tags'] as $tag) {
            if (!in_array($tag, $analysis['tags_found'] ?? [])) {
                $compliance['missing_phase2'][] = $tag;
            }
        }

        // فحص الحجم (يُنصح بألا يتجاوز 1000 بايت)
        $totalSize = $analysis['total_size'] ?? 0;
        $compliance['size_compliance'] = $totalSize <= 1000;
        $compliance['actual_size'] = $totalSize;

        if (!$compliance['size_compliance']) {
            $compliance['messages'][] = "حجم QR Code ({$totalSize} بايت) يتجاوز الحد المُوصى به (1000 بايت)";
        }

        // رسائل النتائج
        if (empty($compliance['missing_required'])) {
            $compliance['messages'][] = "✅ جميع العناصر الأساسية موجودة";
        } else {
            $compliance['messages'][] = "❌ عناصر أساسية مفقودة: " . implode(', ', $compliance['missing_required']);
        }

        if (empty($compliance['missing_phase2'])) {
            $compliance['messages'][] = "✅ جميع عناصر المرحلة الثانية موجودة";
        } else {
            $compliance['messages'][] = "⚠️ عناصر المرحلة الثانية المفقودة: " . implode(', ', $compliance['missing_phase2']);
        }

        return $compliance;
    }

    /**
     * تحديث إعدادات QR Code للمرحلة الثانية
     */
    public function updateQRSettingsForPhase2(array $settings): bool
    {
        try {
            // تحديث إعدادات ZATCA لدعم المرحلة الثانية
            $this->settings->qr_phase2_enabled = $settings['phase2_enabled'] ?? true;
            $this->settings->qr_include_signature = $settings['include_signature'] ?? true;
            $this->settings->qr_include_public_key = $settings['include_public_key'] ?? true;
            $this->settings->qr_include_stamp = $settings['include_stamp'] ?? true;
            
            if (isset($settings['max_qr_size'])) {
                $this->settings->qr_max_size = $settings['max_qr_size'];
            }

            $this->settings->save();

            Log::info('QR settings updated for Phase 2', $settings);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to update QR settings for Phase 2', [
                'error' => $e->getMessage(),
                'settings' => $settings
            ]);

            return false;
        }
    }

    /**
     * إنتاج تقرير شامل عن QR Code للمرحلة الثانية
     */
    public function generatePhase2QRReport(Invoice $invoice): array
    {
        try {
            $report = [
                'invoice_info' => [
                    'id' => $invoice->id,
                    'number' => $invoice->invoice_number,
                    'uuid' => $invoice->uuid,
                    'issue_date' => $invoice->issue_date->toISOString(),
                    'total_amount' => $invoice->total_amount,
                    'tax_amount' => $invoice->tax_amount
                ],
                'qr_test_result' => $this->testPhase2QRCodeGeneration($invoice),
                'settings_check' => [
                    'certificate_available' => !empty($this->settings->certificate),
                    'private_key_available' => !empty($this->settings->private_key),
                    'public_key_available' => !empty($this->settings->public_key)
                ],
                'recommendations' => []
            ];

            // توصيات للتحسين
            if (!$report['settings_check']['certificate_available']) {
                $report['recommendations'][] = "تحديد شهادة رقمية صالحة في إعدادات ZATCA";
            }

            if (!$report['settings_check']['private_key_available']) {
                $report['recommendations'][] = "تحديد المفتاح الخاص في إعدادات ZATCA";
            }

            if (!$report['qr_test_result']['success']) {
                $report['recommendations'][] = "إصلاح مشاكل توليد QR Code قبل الإنتاج";
            }

            if (($report['qr_test_result']['tlv_size'] ?? 0) > 1000) {
                $report['recommendations'][] = "تقليل حجم QR Code لتحسين الأداء";
            }

            return $report;

        } catch (\Exception $e) {
            return [
                'error' => 'فشل في إنتاج تقرير QR Code: ' . $e->getMessage(),
                'invoice_id' => $invoice->id
            ];
        }
    }

    /**
     * حساب Hash الفاتورة
     */
    public function calculateInvoiceHash(string $xmlContent): string
    {
        return hash('sha256', $xmlContent);
    }

    // باقي الدوال المطلوبة...
    /**
     * توليد XML للفاتورة المبسطة متوافق مع ZATCA v2.0
     */
    protected function generateSimplifiedUblInvoiceXml(Invoice $invoice): string
    {
        return $this->generateCompleteZatcaXml($invoice, 'simplified');
    }

    /**
     * توليد XML للفاتورة القياسية متوافق مع ZATCA v2.0
     */
    public function generateUblInvoiceXml(Invoice $invoice): string
    {
        return $this->generateCompleteZatcaXml($invoice, 'standard');
    }

    /**
     * توليد XML متكامل متوافق مع معيار ZATCA e-Invoicing XML v2.0
     */
    public function generateCompleteZatcaXml(Invoice $invoice, string $type = 'standard'): string
    {
        try {
            // إنشاء DOMDocument للتعامل مع XML بشكل صحيح
            $dom = new \DOMDocument('1.0', 'UTF-8');
            $dom->formatOutput = true;
            
            // إنشاء العنصر الجذر مع جميع namespaces المطلوبة
            $invoiceElement = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:Invoice-2', 'Invoice');
            $invoiceElement->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:cac', 'urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2');
            $invoiceElement->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:cbc', 'urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2');
            $invoiceElement->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:ds', 'http://www.w3.org/2000/09/xmldsig#');
            $invoiceElement->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:xades', 'http://uri.etsi.org/01903/v1.3.2#');
            $invoiceElement->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:ext', 'urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2');
            $dom->appendChild($invoiceElement);

            // 1. إضافة البيانات الأساسية (UBL Header) أولاً - مطلوب حسب معيار UBL
            $this->addUblHeader($dom, $invoiceElement, $invoice, $type);
            
            // 2. إضافة معرف التوقيع المطلوب لـ ZATCA بعد العناصر الأساسية
            $this->addSignatureElements($dom, $invoiceElement, $invoice);
            
            // 3. إضافة بيانات المورد (Seller/Supplier) - مطلوب أولاً
            $this->addSupplierParty($dom, $invoiceElement, $invoice);
            
            // 4. إضافة بيانات العميل (Buyer/Customer) - مطلوب قبل InvoiceLines حسب UBL 2.1
            // للفواتير القياسية: بيانات عميل كاملة
            // للفواتير المبسطة: بيانات عميل مبسطة (مطلوب حسب UBL 2.1)
            if ($type === 'standard' && $invoice->customer) {
                $this->addCustomerParty($dom, $invoiceElement, $invoice);
            } else {
                // إضافة بيانات عميل افتراضية للفواتير المبسطة (B2C) حسب UBL 2.1
                $this->addSimplifiedCustomerParty($dom, $invoiceElement, $invoice, $type);
            }
            
            // 5. إضافة عناصر الفاتورة (Invoice Lines) - يأتي بعد الأطراف
            $this->addInvoiceLines($dom, $invoiceElement, $invoice);
            
            // 5. إضافة إجمالي الضرائب (Tax Total)
            $this->addTaxTotal($dom, $invoiceElement, $invoice);
            
            // 6. إضافة المجاميع القانونية (Legal Monetary Total)
            $this->addLegalMonetaryTotal($dom, $invoiceElement, $invoice);
            
            // 7. إضافة QR Code
            $this->addQRCodeToXml($dom, $invoiceElement, $invoice);
            
            // 8. إضافة Extension للتوقيع الرقمي (إذا أمكن)
            $this->addDigitalSignatureExtension($dom, $invoiceElement, $invoice);
            
            $xmlContent = $dom->saveXML();
            
            // حفظ الملف
            $this->saveXmlFile($xmlContent, $invoice);
            
            Log::info('تم توليد XML متوافق مع ZATCA v2.0 بنجاح', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'type' => $type,
                'xml_size' => strlen($xmlContent)
            ]);
            
            return $xmlContent;
            
        } catch (\Exception $e) {
            Log::error('فشل في توليد XML متوافق مع ZATCA', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw new \Exception('فشل في توليد XML: ' . $e->getMessage());
        }
    }

    /**
     * إضافة البيانات الأساسية للفاتورة (UBL Header)
     */
    private function addUblHeader(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice, string $type): void
    {
        // UBL Version
        $invoiceElement->appendChild($dom->createElement('cbc:UBLVersionID', '2.1'));
        
        // Customization ID - ZATCA specific
        $invoiceElement->appendChild($dom->createElement('cbc:CustomizationID', 'BR-KSA-10'));
        
        // Profile ID - حسب نوع الفاتورة
        $profileId = $type === 'simplified' ? 'reporting:1.0' : 'clearance:1.0';
        $invoiceElement->appendChild($dom->createElement('cbc:ProfileID', $profileId));
        
        // Invoice ID (رقم الفاتورة)
        $invoiceElement->appendChild($dom->createElement('cbc:ID', htmlspecialchars($invoice->invoice_number)));
        
        // UUID
        if (empty($invoice->uuid)) {
            $invoice->uuid = (string) Str::uuid();
            $invoice->save();
        }
        $invoiceElement->appendChild($dom->createElement('cbc:UUID', $invoice->uuid));
        
        // تاريخ ووقت الإصدار
        $invoiceElement->appendChild($dom->createElement('cbc:IssueDate', $invoice->issue_date->format('Y-m-d')));
        $invoiceElement->appendChild($dom->createElement('cbc:IssueTime', $invoice->issue_date->format('H:i:s')));
        
        // نوع الفاتورة
        $invoiceTypeCode = $dom->createElement('cbc:InvoiceTypeCode', '388');
        $invoiceTypeCode->setAttribute('name', $type === 'simplified' ? '0200000' : '0100000');
        $invoiceElement->appendChild($invoiceTypeCode);
        
        // ملاحظات الفاتورة (إذا وجدت)
        if (!empty($invoice->notes)) {
            $invoiceElement->appendChild($dom->createElement('cbc:Note', htmlspecialchars($invoice->notes)));
        }
        
        // عملة الفاتورة
        $invoiceElement->appendChild($dom->createElement('cbc:DocumentCurrencyCode', 'SAR'));
        $invoiceElement->appendChild($dom->createElement('cbc:TaxCurrencyCode', 'SAR'));
        
        // إضافة نوع العملية التجارية (BR-KSA-EN16931-01) - مطلوب
        $this->addBusinessProcessType($dom, $invoiceElement, $invoice, $type);
        
        // إضافة عداد الفاتورة (KSA-16) - مطلوب لـ ZATCA
        $this->addInvoiceCounter($dom, $invoiceElement, $invoice);
        
        // إضافة رمز الفاتورة (KSA-2) - مطلوب (BR-KSA-06)
        $this->addInvoiceCodeKSA2($dom, $invoiceElement, $invoice, $type);
        
        // إضافة عداد الفاتورة (InstructionID) - مطلوب (BR-KSA-33)
        $this->addInstructionID($dom, $invoiceElement, $invoice);
        
        // إضافة مرجع الفاتورة السابقة إذا كان متاحاً
        $this->addPreviousInvoiceHash($dom, $invoiceElement, $invoice);
    }
    
    /**
     * إضافة عداد الفاتورة (KSA-16)
     */
    private function addInvoiceCounter(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        // إنشاء أو الحصول على عداد الفاتورة
        if (empty($invoice->invoice_counter)) {
            // البحث عن آخر فاتورة للشركة للحصول على العداد التالي
            $lastInvoice = \App\Models\Invoice::where('company_id', $invoice->company_id)
                ->where('id', '!=', $invoice->id)
                ->orderBy('created_at', 'desc')
                ->first();
            
            $invoice->invoice_counter = $lastInvoice ? ($lastInvoice->invoice_counter + 1) : 1;
            $invoice->save();
        }
        
        // إضافة عداد الفاتورة كـ Additional Document Reference
        $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
        $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'ICV'));
        $additionalDocRef->appendChild($dom->createElement('cbc:UUID', (string)$invoice->invoice_counter));
        $invoiceElement->appendChild($additionalDocRef);
    }
    
    /**
     * إضافة مرجع الفاتورة السابقة (KSA-13)
     */
    private function addPreviousInvoiceHash(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        // البحث عن الفاتورة السابقة
        $previousInvoice = \App\Models\Invoice::where('company_id', $invoice->company_id)
            ->where('id', '<', $invoice->id)
            ->whereNotNull('invoice_hash')
            ->orderBy('created_at', 'desc')
            ->first();
        
        if ($previousInvoice && !empty($previousInvoice->invoice_hash)) {
            $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
            $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'PIH'));
            $additionalDocRef->appendChild($dom->createElement('cbc:UUID', $previousInvoice->invoice_hash));
            $invoiceElement->appendChild($additionalDocRef);
        }
    }
    
    /**
     * إضافة نوع العملية التجارية (BR-KSA-EN16931-01) - مطلوب
     */
    private function addBusinessProcessType(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice, string $type): void
    {
        // إضافة BusinessProcessSpecificationID - مطلوب حسب BR-KSA-EN16931-01
        $businessProcessId = $type === 'simplified' ? 'KSA_B2C_Invoice' : 'KSA_B2B_Invoice';
        
        // إضافة كـ Additional Document Reference
        $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
        $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'BUSINESS_PROCESS'));
        $additionalDocRef->appendChild($dom->createElement('cbc:DocumentType', $businessProcessId));
        $additionalDocRef->appendChild($dom->createElement('cbc:DocumentDescription', 'Business Process Specification'));
        $invoiceElement->appendChild($additionalDocRef);
        
        Log::info('تم إضافة نوع العملية التجارية (BR-KSA-EN16931-01)', [
            'invoice_id' => $invoice->id,
            'business_process' => $businessProcessId,
            'invoice_type' => $type
        ]);
    }
    
    /**
     * إضافة رمز الفاتورة (KSA-2) - مطلوب (BR-KSA-06)
     */
    private function addInvoiceCodeKSA2(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice, string $type): void
    {
        // تحديد رمز الفاتورة حسب النوع
        // للفاتورة المبسطة: 02 = Tax Invoice for B2C
        // للفاتورة القياسية: 01 = Tax Invoice for B2B
        $invoiceCode = $type === 'simplified' ? '02' : '01';
        
        // إضافة رمز الفاتورة كـ Additional Document Reference
        $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
        $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'KSA-2'));
        $additionalDocRef->appendChild($dom->createElement('cbc:DocumentType', $invoiceCode));
        $additionalDocRef->appendChild($dom->createElement('cbc:DocumentDescription', 'Invoice Type Code'));
        $invoiceElement->appendChild($additionalDocRef);
        
        Log::info('تم إضافة رمز الفاتورة (KSA-2)', [
            'invoice_id' => $invoice->id,
            'invoice_code' => $invoiceCode,
            'invoice_type' => $type
        ]);
    }
    
    /**
     * إضافة عداد الفاتورة (InstructionID) - مطلوب (BR-KSA-33)
     */
    private function addInstructionID(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        // إنشاء أو الحصول على InstructionID
        if (empty($invoice->instruction_id)) {
            // إنشاء InstructionID فريد يحتوي على معرف الشركة ورقم الفاتورة والوقت
            $invoice->instruction_id = 'INS-' . $invoice->company_id . '-' . $invoice->id . '-' . time();
            $invoice->save();
        }
        
        // إضافة InstructionID كـ Additional Document Reference
        $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
        $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'InstructionID'));
        $additionalDocRef->appendChild($dom->createElement('cbc:UUID', $invoice->instruction_id));
        $additionalDocRef->appendChild($dom->createElement('cbc:DocumentDescription', 'Invoice Instruction Identifier'));
        $invoiceElement->appendChild($additionalDocRef);
        
        Log::info('تم إضافة عداد الفاتورة (InstructionID)', [
            'invoice_id' => $invoice->id,
            'instruction_id' => $invoice->instruction_id
        ]);
    }

    /**
     * إضافة بيانات المورد (Supplier/Seller) - محسن للتوافق مع UBL 2.1 Party structure
     */
    private function addSupplierParty(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        $supplierParty = $dom->createElement('cac:SellerSupplierParty');
        $party = $dom->createElement('cac:Party');
        
        // 1. إضافة معرفات الطرف أولاً (PartyIdentification) - يجب أن تكون أول عناصر Party
        // معرف الرقم الضريبي الأساسي
        $vatPartyIdentification = $dom->createElement('cac:PartyIdentification');
        $vatId = $dom->createElement('cbc:ID', $invoice->company->vat_number ?? '000000000000000');
        $vatId->setAttribute('schemeID', 'VAT');
        $vatPartyIdentification->appendChild($vatId);
        $party->appendChild($vatPartyIdentification);
        
        // إضافة معرفات إضافية للمورد (CRN, MOM, MLS, etc.) - يجب أن تأتي مباشرة بعد المعرف الأساسي
        $this->addSupplierAdditionalIdentifications($dom, $party, $invoice);
        
        // 2. اسم المورد (PartyName) - يأتي بعد جميع PartyIdentification elements
        $partyName = $dom->createElement('cac:PartyName');
        $partyName->appendChild($dom->createElement('cbc:Name', htmlspecialchars($invoice->company->name ?? 'Unknown Supplier')));
        $party->appendChild($partyName);
        
        // 3. عنوان المورد (PostalAddress) - يأتي بعد PartyName
        if ($invoice->company) {
            $address = $dom->createElement('cac:PostalAddress');
            
            // اسم الشارع (BT-35)
            $streetName = $invoice->company->address ?? 'Unknown Street';
            $address->appendChild($dom->createElement('cbc:StreetName', htmlspecialchars($streetName)));
            
            // رقم المبنى (KSA-17) - يجب أن يكون 4 أرقام
            $buildingNumber = $this->formatBuildingNumber($invoice->company->building_number ?? '1234');
            $address->appendChild($dom->createElement('cbc:BuildingNumber', $buildingNumber));
            
            // الرقم الإضافي
            if (!empty($invoice->company->additional_number)) {
                $address->appendChild($dom->createElement('cbc:PlotIdentification', $invoice->company->additional_number));
            }
            
            // الحي (KSA-3)
            $district = $invoice->company->district ?? $invoice->company->region ?? 'Unknown District';
            $address->appendChild($dom->createElement('cbc:CitySubdivisionName', htmlspecialchars($district)));
            
            // المدينة (BT-37)
            $address->appendChild($dom->createElement('cbc:CityName', htmlspecialchars($invoice->company->city ?? 'Unknown City')));
            
            // الرمز البريدي (BT-38)
            $address->appendChild($dom->createElement('cbc:PostalZone', $invoice->company->postal_code ?? '00000'));
            
            // رمز البلد (BT-40)
            $address->appendChild($dom->createElement('cbc:CountrySubentity', htmlspecialchars($invoice->company->region ?? 'Unknown Region')));
            
            $country = $dom->createElement('cac:Country');
            $country->appendChild($dom->createElement('cbc:IdentificationCode', 'SA'));
            $address->appendChild($country);
            
            $party->appendChild($address);
        }
        
        // 4. الرقم الضريبي للمورد (PartyTaxScheme) - يأتي بعد PostalAddress
        $partyTaxScheme = $dom->createElement('cac:PartyTaxScheme');
        $partyTaxScheme->appendChild($dom->createElement('cbc:CompanyID', $invoice->company->vat_number ?? '000000000000000'));
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $partyTaxScheme->appendChild($taxScheme);
        $party->appendChild($partyTaxScheme);
        
        // 5. معلومات قانونية إضافية (PartyLegalEntity) - يأتي في النهاية
        $partyLegalEntity = $dom->createElement('cac:PartyLegalEntity');
        $partyLegalEntity->appendChild($dom->createElement('cbc:RegistrationName', htmlspecialchars($invoice->company->name ?? 'Unknown Company')));
        
        // السجل التجاري (إذا توفر)
        if (!empty($invoice->company->commercial_registration)) {
            $partyLegalEntity->appendChild($dom->createElement('cbc:CompanyID', $invoice->company->commercial_registration));
        }
        
        $party->appendChild($partyLegalEntity);
        
        $supplierParty->appendChild($party);
        $invoiceElement->appendChild($supplierParty);
        
        Log::info('تم إضافة بيانات المورد بترتيب UBL 2.1 الصحيح', [
            'invoice_id' => $invoice->id,
            'party_elements_order' => [
                'PartyIdentification (VAT)',
                'PartyIdentification (Additional IDs)',
                'PartyName',
                'PostalAddress',
                'PartyTaxScheme',
                'PartyLegalEntity'
            ]
        ]);
    }
    
    /**
     * تنسيق رقم المبنى ليكون 4 أرقام
     */
    private function formatBuildingNumber(?string $buildingNumber): string
    {
        if (empty($buildingNumber)) {
            return '1234'; // رقم افتراضي
        }
        
        // إزالة الأحرف غير الرقمية
        $number = preg_replace('/[^0-9]/', '', $buildingNumber);
        
        // التأكد من أن الرقم مكون من 4 أرقام
        if (strlen($number) > 4) {
            return substr($number, 0, 4);
        } else {
            return str_pad($number, 4, '0', STR_PAD_LEFT);
        }
    }
    
    /**
     * إضافة معرفات إضافية للمورد
     */
    private function addSupplierAdditionalIdentifications(\DOMDocument $dom, \DOMElement $party, Invoice $invoice): void
    {
        // السجل التجاري (CRN)
        if (!empty($invoice->company->commercial_registration)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->commercial_registration);
            $idElement->setAttribute('schemeID', 'CRN');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
        
        // رخصة وزارة الشؤون البلدية (MOM)
        if (!empty($invoice->company->mom_license)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->mom_license);
            $idElement->setAttribute('schemeID', 'MOM');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
        
        // رخصة وزارة الموارد البشرية (MLS)
        if (!empty($invoice->company->mls_license)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->mls_license);
            $idElement->setAttribute('schemeID', 'MLS');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
        
        // رقم 700
        if (!empty($invoice->company->license_700)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->license_700);
            $idElement->setAttribute('schemeID', '700');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
        
        // رخصة وزارة الاستثمار (SAG)
        if (!empty($invoice->company->sag_license)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->sag_license);
            $idElement->setAttribute('schemeID', 'SAG');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
        
        // معرف آخر (OTH)
        if (!empty($invoice->company->other_id)) {
            $partyIdentification = $dom->createElement('cac:PartyIdentification');
            $idElement = $dom->createElement('cbc:ID', $invoice->company->other_id);
            $idElement->setAttribute('schemeID', 'OTH');
            $partyIdentification->appendChild($idElement);
            $party->appendChild($partyIdentification);
        }
    }

    /**
     * إضافة بيانات العميل (Customer/Buyer) - للفواتير القياسية
     */
    private function addCustomerParty(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        $customerParty = $dom->createElement('cac:BuyerCustomerParty');
        $party = $dom->createElement('cac:Party');
        
        // اسم العميل
        $partyName = $dom->createElement('cac:PartyName');
        $partyName->appendChild($dom->createElement('cbc:Name', htmlspecialchars($invoice->customer->name ?? 'Unknown Customer')));
        $party->appendChild($partyName);
        
        // عنوان العميل (إذا توفر)
        if ($invoice->customer && !empty($invoice->customer->address)) {
            $address = $dom->createElement('cac:PostalAddress');
            $address->appendChild($dom->createElement('cbc:StreetName', htmlspecialchars($invoice->customer->address)));
            $address->appendChild($dom->createElement('cbc:CityName', htmlspecialchars($invoice->customer->city ?? 'Unknown City')));
            
            $country = $dom->createElement('cac:Country');
            $country->appendChild($dom->createElement('cbc:IdentificationCode', 'SA'));
            $address->appendChild($country);
            
            $party->appendChild($address);
        }
        
        // الرقم الضريبي للعميل (إذا توفر)
        if ($invoice->customer && !empty($invoice->customer->vat_number)) {
            $partyTaxScheme = $dom->createElement('cac:PartyTaxScheme');
            $partyTaxScheme->appendChild($dom->createElement('cbc:CompanyID', $invoice->customer->vat_number));
            
            $taxScheme = $dom->createElement('cac:TaxScheme');
            $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
            $partyTaxScheme->appendChild($taxScheme);
            $party->appendChild($partyTaxScheme);
        }
        
        $customerParty->appendChild($party);
        $invoiceElement->appendChild($customerParty);
    }

    /**
     * إضافة بيانات عميل مبسطة للفواتير المبسطة (B2C) - متوافق مع UBL 2.1
     */
    private function addSimplifiedCustomerParty(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice, string $type): void
    {
        $customerParty = $dom->createElement('cac:BuyerCustomerParty');
        $party = $dom->createElement('cac:Party');
        
        // للفواتير المبسطة (B2C): إما بيانات العميل الفعلية أو بيانات افتراضية
        if ($invoice->customer && !empty($invoice->customer->name)) {
            // استخدام بيانات العميل الفعلية إذا كانت متوفرة
            $partyName = $dom->createElement('cac:PartyName');
            $partyName->appendChild($dom->createElement('cbc:Name', htmlspecialchars($invoice->customer->name)));
            $party->appendChild($partyName);
            
            // إضافة الرقم الضريبي للعميل إذا كان متوفراً (للشركات في فواتير B2C)
            if (!empty($invoice->customer->vat_number)) {
                $partyTaxScheme = $dom->createElement('cac:PartyTaxScheme');
                $partyTaxScheme->appendChild($dom->createElement('cbc:CompanyID', $invoice->customer->vat_number));
                
                $taxScheme = $dom->createElement('cac:TaxScheme');
                $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
                $partyTaxScheme->appendChild($taxScheme);
                $party->appendChild($partyTaxScheme);
            }
        } else {
            // بيانات عميل افتراضية للفواتير المبسطة بدون عميل محدد
            $partyName = $dom->createElement('cac:PartyName');
            $partyName->appendChild($dom->createElement('cbc:Name', 'Customer')); // اسم عام للعميل
            $party->appendChild($partyName);
        }
        
        // إضافة عنوان افتراضي للفواتير المبسطة (مطلوب حسب UBL)
        $address = $dom->createElement('cac:PostalAddress');
        
        if ($invoice->customer && !empty($invoice->customer->address)) {
            $address->appendChild($dom->createElement('cbc:StreetName', htmlspecialchars($invoice->customer->address)));
            $address->appendChild($dom->createElement('cbc:CityName', htmlspecialchars($invoice->customer->city ?? 'Unknown City')));
        } else {
            // عنوان افتراضي
            $address->appendChild($dom->createElement('cbc:StreetName', 'Customer Address'));
            $address->appendChild($dom->createElement('cbc:CityName', 'Customer City'));
        }
        
        $country = $dom->createElement('cac:Country');
        $country->appendChild($dom->createElement('cbc:IdentificationCode', 'SA'));
        $address->appendChild($country);
        $party->appendChild($address);
        
        $customerParty->appendChild($party);
        $invoiceElement->appendChild($customerParty);
        
        Log::info('تم إضافة بيانات عميل مبسطة للفاتورة المبسطة حسب UBL 2.1', [
            'invoice_id' => $invoice->id,
            'invoice_type' => $type,
            'customer_available' => !empty($invoice->customer),
            'customer_name' => $invoice->customer->name ?? 'Customer (default)'
        ]);
    }

    /**
     * إضافة عناصر الفاتورة (Invoice Lines)
     */
    private function addInvoiceLines(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        // إذا كانت هناك عناصر فاتورة مخزنة في علاقة
        if ($invoice->items && $invoice->items->count() > 0) {
            foreach ($invoice->items as $index => $item) {
                $this->addInvoiceLine($dom, $invoiceElement, $item, $index + 1);
            }
        } else {
            // إنشاء عنصر افتراضي إذا لم تكن هناك عناصر محددة
            $this->addDefaultInvoiceLine($dom, $invoiceElement, $invoice);
        }
    }

    /**
     * إضافة عنصر واحد من عناصر الفاتورة
     */
    private function addInvoiceLine(\DOMDocument $dom, \DOMElement $invoiceElement, $item, int $lineNumber): void
    {
        $invoiceLine = $dom->createElement('cac:InvoiceLine');
        
        // رقم السطر
        $invoiceLine->appendChild($dom->createElement('cbc:ID', $lineNumber));
        
        // الكمية مع وحدة قياس معيارية
        $quantity = $dom->createElement('cbc:InvoicedQuantity', number_format($item->quantity ?? 1, 3, '.', ''));
        $unitCode = $this->getStandardUnitCode($item->unit ?? 'قطعة');
        $quantity->setAttribute('unitCode', $unitCode);
        $invoiceLine->appendChild($quantity);
        
        // حساب المبلغ الصحيح للسطر (بدون ضريبة)
        $lineTotal = ($item->price ?? 0) * ($item->quantity ?? 1);
        $lineExtension = $dom->createElement('cbc:LineExtensionAmount', number_format($lineTotal, 2, '.', ''));
        $lineExtension->setAttribute('currencyID', 'SAR');
        $invoiceLine->appendChild($lineExtension);
        
        // تفاصيل الضريبة للعنصر
        $taxTotal = $dom->createElement('cac:TaxTotal');
        $taxRate = ($item->tax_rate ?? 15) / 100;
        $itemTaxAmount = $lineTotal * $taxRate;
        $taxAmount = $dom->createElement('cbc:TaxAmount', number_format($itemTaxAmount, 2, '.', ''));
        $taxAmount->setAttribute('currencyID', 'SAR');
        $taxTotal->appendChild($taxAmount);
        
        // إضافة تفصيل الضريبة
        $taxSubtotal = $dom->createElement('cac:TaxSubtotal');
        $taxableAmount = $dom->createElement('cbc:TaxableAmount', number_format($lineTotal, 2, '.', ''));
        $taxableAmount->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxableAmount);
        
        $taxAmountSub = $dom->createElement('cbc:TaxAmount', number_format($itemTaxAmount, 2, '.', ''));
        $taxAmountSub->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxAmountSub);
        
        $taxCategory = $dom->createElement('cac:TaxCategory');
        $taxCategory->appendChild($dom->createElement('cbc:ID', 'S'));
        $taxCategory->appendChild($dom->createElement('cbc:Percent', number_format($item->tax_rate ?? 15, 2, '.', '')));
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $taxCategory->appendChild($taxScheme);
        $taxSubtotal->appendChild($taxCategory);
        $taxTotal->appendChild($taxSubtotal);
        
        $invoiceLine->appendChild($taxTotal);
        
        // تفاصيل السلعة أو الخدمة
        $item_element = $dom->createElement('cac:Item');
        $item_element->appendChild($dom->createElement('cbc:Name', htmlspecialchars($item->name ?? 'Unknown Item')));
        
        // تصنيف الضريبة للسلعة
        $classifiedTaxCategory = $dom->createElement('cac:ClassifiedTaxCategory');
        $classifiedTaxCategory->appendChild($dom->createElement('cbc:ID', 'S')); // S = Standard rate
        $classifiedTaxCategory->appendChild($dom->createElement('cbc:Percent', number_format($item->tax_rate ?? 15, 2, '.', '')));
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $classifiedTaxCategory->appendChild($taxScheme);
        $item_element->appendChild($classifiedTaxCategory);
        
        $invoiceLine->appendChild($item_element);
        
        // السعر
        $price = $dom->createElement('cac:Price');
        $priceAmount = $dom->createElement('cbc:PriceAmount', number_format($item->price ?? 0, 2, '.', ''));
        $priceAmount->setAttribute('currencyID', 'SAR');
        $price->appendChild($priceAmount);
        
        // إضافة كمية القاعدة
        $baseQuantity = $dom->createElement('cbc:BaseQuantity', number_format($item->quantity ?? 1, 3, '.', ''));
        $baseQuantity->setAttribute('unitCode', $unitCode);
        $price->appendChild($baseQuantity);
        
        $invoiceLine->appendChild($price);
        
        $invoiceElement->appendChild($invoiceLine);
    }
    
    /**
     * تحويل وحدات القياس العربية إلى رموز UBL معيارية
     */
    private function getStandardUnitCode(string $arabicUnit): string
    {
        $unitMapping = [
            'قطعة' => 'PCE',
            'كيلو' => 'KGM',
            'جرام' => 'GRM',
            'لتر' => 'LTR',
            'متر' => 'MTR',
            'مربع' => 'MTK',
            'مكعب' => 'MTQ',
            'ساعة' => 'HUR',
            'يوم' => 'DAY',
            'شهر' => 'MON',
            'سنة' => 'ANN',
            'طن' => 'TNE',
            'صندوق' => 'BX',
            'كرتون' => 'CT',
            'عبوة' => 'PK',
            'زجاجة' => 'BO',
            'كيس' => 'BG',
            'دستة' => 'DZN'
        ];
        
        return $unitMapping[$arabicUnit] ?? 'PCE'; // افتراضي: قطعة
    }

    /**
     * إضافة عنصر فاتورة افتراضي
     */
    private function addDefaultInvoiceLine(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        $invoiceLine = $dom->createElement('cac:InvoiceLine');
        
        $invoiceLine->appendChild($dom->createElement('cbc:ID', '1'));
        
        $quantity = $dom->createElement('cbc:InvoicedQuantity', '1.000');
        $quantity->setAttribute('unitCode', 'PCE');
        $invoiceLine->appendChild($quantity);
        
        $lineExtension = $dom->createElement('cbc:LineExtensionAmount', number_format($invoice->subtotal, 2, '.', ''));
        $lineExtension->setAttribute('currencyID', 'SAR');
        $invoiceLine->appendChild($lineExtension);
        
        // تفاصيل الضريبة مع تفصيل كامل
        $taxTotal = $dom->createElement('cac:TaxTotal');
        $taxAmount = $dom->createElement('cbc:TaxAmount', number_format($invoice->tax_amount, 2, '.', ''));
        $taxAmount->setAttribute('currencyID', 'SAR');
        $taxTotal->appendChild($taxAmount);
        
        // إضافة تفصيل الضريبة
        $taxSubtotal = $dom->createElement('cac:TaxSubtotal');
        $taxableAmount = $dom->createElement('cbc:TaxableAmount', number_format($invoice->subtotal, 2, '.', ''));
        $taxableAmount->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxableAmount);
        
        $taxAmountSub = $dom->createElement('cbc:TaxAmount', number_format($invoice->tax_amount, 2, '.', ''));
        $taxAmountSub->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxAmountSub);
        
        $taxCategory = $dom->createElement('cac:TaxCategory');
        $taxCategory->appendChild($dom->createElement('cbc:ID', 'S'));
        $taxCategory->appendChild($dom->createElement('cbc:Percent', '15.00'));
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $taxCategory->appendChild($taxScheme);
        $taxSubtotal->appendChild($taxCategory);
        $taxTotal->appendChild($taxSubtotal);
        
        $invoiceLine->appendChild($taxTotal);
        
        // تفاصيل السلعة
        $item = $dom->createElement('cac:Item');
        $item->appendChild($dom->createElement('cbc:Name', 'Default Service/Product'));
        
        $classifiedTaxCategory = $dom->createElement('cac:ClassifiedTaxCategory');
        $classifiedTaxCategory->appendChild($dom->createElement('cbc:ID', 'S'));
        $classifiedTaxCategory->appendChild($dom->createElement('cbc:Percent', '15.00'));
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $classifiedTaxCategory->appendChild($taxScheme);
        $item->appendChild($classifiedTaxCategory);
        
        $invoiceLine->appendChild($item);
        
        // السعر
        $price = $dom->createElement('cac:Price');
        $priceAmount = $dom->createElement('cbc:PriceAmount', number_format($invoice->subtotal, 2, '.', ''));
        $priceAmount->setAttribute('currencyID', 'SAR');
        $price->appendChild($priceAmount);
        
        // إضافة كمية القاعدة
        $baseQuantity = $dom->createElement('cbc:BaseQuantity', '1.000');
        $baseQuantity->setAttribute('unitCode', 'PCE');
        $price->appendChild($baseQuantity);
        
        $invoiceLine->appendChild($price);
        
        $invoiceElement->appendChild($invoiceLine);
    }

    /**
     * إضافة إجمالي الضرائب (Tax Total) - محسن للتوافق مع ZATCA BR-S-08
     */
    private function addTaxTotal(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        $taxTotal = $dom->createElement('cac:TaxTotal');
        
        // حساب صحيح للضريبة بناءً على العناصر
        $netAmount = $this->calculateNetInvoiceAmount($invoice);
        $documentLevelDiscounts = $invoice->discount_amount ?? 0;
        $documentLevelCharges = $invoice->charges_amount ?? 0;
        
        // المبلغ الخاضع للضريبة = مجموع المبالغ الصافية - الخصومات + الرسوم
        $taxableBaseAmount = $netAmount - $documentLevelDiscounts + $documentLevelCharges;
        
        // حساب الضريبة الصحيح
        $calculatedTaxAmount = $taxableBaseAmount * 0.15; // 15% VAT
        
        // إجمالي مبلغ الضريبة
        $taxAmount = $dom->createElement('cbc:TaxAmount', number_format($calculatedTaxAmount, 2, '.', ''));
        $taxAmount->setAttribute('currencyID', 'SAR');
        $taxTotal->appendChild($taxAmount);
        
        // تفصيل الضريبة حسب النوع
        $taxSubtotal = $dom->createElement('cac:TaxSubtotal');
        
        // المبلغ الخاضع للضريبة (BR-S-08)
        $taxableAmount = $dom->createElement('cbc:TaxableAmount', number_format($taxableBaseAmount, 2, '.', ''));
        $taxableAmount->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxableAmount);
        
        // مبلغ الضريبة
        $taxAmountSub = $dom->createElement('cbc:TaxAmount', number_format($calculatedTaxAmount, 2, '.', ''));
        $taxAmountSub->setAttribute('currencyID', 'SAR');
        $taxSubtotal->appendChild($taxAmountSub);
        
        // فئة الضريبة
        $taxCategory = $dom->createElement('cac:TaxCategory');
        $taxCategory->appendChild($dom->createElement('cbc:ID', 'S')); // Standard rate
        $taxCategory->appendChild($dom->createElement('cbc:Percent', '15.00')); // نسبة الضريبة في السعودية
        
        $taxScheme = $dom->createElement('cac:TaxScheme');
        $taxScheme->appendChild($dom->createElement('cbc:ID', 'VAT'));
        $taxCategory->appendChild($taxScheme);
        $taxSubtotal->appendChild($taxCategory);
        
        $taxTotal->appendChild($taxSubtotal);
        $invoiceElement->appendChild($taxTotal);
        
        // تحديث قيم الفاتورة إذا كانت مختلفة
        if (abs($invoice->tax_amount - $calculatedTaxAmount) > 0.01) {
            Log::info('تم تصحيح مبلغ الضريبة للتوافق مع BR-S-08', [
                'invoice_id' => $invoice->id,
                'old_tax_amount' => $invoice->tax_amount,
                'new_tax_amount' => $calculatedTaxAmount,
                'taxable_base' => $taxableBaseAmount
            ]);
        }
    }

    /**
     * حساب المبلغ الصافي للفاتورة (BT-106)
     */
    private function calculateNetInvoiceAmount(Invoice $invoice): float
    {
        $netAmount = 0;
        
        // إذا كانت هناك عناصر فاتورة مخزنة
        if ($invoice->items && $invoice->items->count() > 0) {
            foreach ($invoice->items as $item) {
                $lineTotal = ($item->price ?? 0) * ($item->quantity ?? 1);
                $netAmount += $lineTotal;
            }
        } else {
            // استخدام subtotal إذا لم تكن هناك عناصر مفصلة
            $netAmount = $invoice->subtotal ?? 0;
        }
        
        return $netAmount;
    }

    /**
     * إضافة المجاميع القانونية (Legal Monetary Total) - محسن للتوافق مع ZATCA
     */
    private function addLegalMonetaryTotal(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        $legalMonetaryTotal = $dom->createElement('cac:LegalMonetaryTotal');
        
        // حساب المبالغ بشكل صحيح
        $netAmount = $this->calculateNetInvoiceAmount($invoice);
        $documentLevelDiscounts = $invoice->discount_amount ?? 0;
        $documentLevelCharges = $invoice->charges_amount ?? 0;
        
        // مجموع المبالغ الصافية على مستوى بند الفاتورة (BT-106)
        $lineExtensionAmount = $dom->createElement('cbc:LineExtensionAmount', number_format($netAmount, 2, '.', ''));
        $lineExtensionAmount->setAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->appendChild($lineExtensionAmount);
        
        // إضافة خصومات على مستوى المستند إن وجدت (BT-92)
        if ($documentLevelDiscounts > 0) {
            $allowanceTotalAmount = $dom->createElement('cbc:AllowanceTotalAmount', number_format($documentLevelDiscounts, 2, '.', ''));
            $allowanceTotalAmount->setAttribute('currencyID', 'SAR');
            $legalMonetaryTotal->appendChild($allowanceTotalAmount);
        }
        
        // إضافة رسوم على مستوى المستند إن وجدت (BT-99)
        if ($documentLevelCharges > 0) {
            $chargeTotalAmount = $dom->createElement('cbc:ChargeTotalAmount', number_format($documentLevelCharges, 2, '.', ''));
            $chargeTotalAmount->setAttribute('currencyID', 'SAR');
            $legalMonetaryTotal->appendChild($chargeTotalAmount);
        }
        
        // المبلغ الخاضع للضريبة (BT-116) = مجموع المبالغ الصافية - الخصومات + الرسوم
        $taxableBaseAmount = $netAmount - $documentLevelDiscounts + $documentLevelCharges;
        $taxExclusiveAmount = $dom->createElement('cbc:TaxExclusiveAmount', number_format($taxableBaseAmount, 2, '.', ''));
        $taxExclusiveAmount->setAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->appendChild($taxExclusiveAmount);
        
        // حساب الضريبة الصحيح
        $calculatedTaxAmount = $taxableBaseAmount * 0.15;
        
        // إجمالي المبلغ شامل الضريبة
        $totalWithTax = $taxableBaseAmount + $calculatedTaxAmount;
        $taxInclusiveAmount = $dom->createElement('cbc:TaxInclusiveAmount', number_format($totalWithTax, 2, '.', ''));
        $taxInclusiveAmount->setAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->appendChild($taxInclusiveAmount);
        
        // المبلغ المستحق الدفع
        $payableAmount = $dom->createElement('cbc:PayableAmount', number_format($totalWithTax, 2, '.', ''));
        $payableAmount->setAttribute('currencyID', 'SAR');
        $legalMonetaryTotal->appendChild($payableAmount);
        
        $invoiceElement->appendChild($legalMonetaryTotal);
        
        // تحديث قيم الفاتورة للتوافق مع الحسابات الصحيحة
        $invoice->tax_amount = $calculatedTaxAmount;
        $invoice->total_amount = $totalWithTax;
        $invoice->save();
        
        Log::info('تم حساب المجاميع القانونية للتوافق مع ZATCA', [
            'invoice_id' => $invoice->id,
            'net_amount' => $netAmount,
            'taxable_base' => $taxableBaseAmount,
            'tax_amount' => $calculatedTaxAmount,
            'total_amount' => $totalWithTax
        ]);
    }

    /**
     * إضافة QR Code إلى XML - محسن للتوافق مع ZATCA
     */
    private function addQRCodeToXml(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        try {
            // توليد QR Code للمرحلة الثانية
            $qrCodeData = $this->generateQRCode($invoice);
            
            if (!empty($qrCodeData)) {
                // إضافة QR Code كـ Additional Document Reference بطريقة متوافقة مع ZATCA
                $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
                $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'QR'));
                $additionalDocRef->appendChild($dom->createElement('cbc:UUID', $invoice->uuid));
                
                // إضافة معلومات إضافية للـ QR Code
                $additionalDocRef->appendChild($dom->createElement('cbc:DocumentType', 'QRCode'));
                $additionalDocRef->appendChild($dom->createElement('cbc:DocumentDescription', 'ZATCA Phase 2 QR Code'));
                
                // إضافة QR Code كـ Embedded Document
                $attachment = $dom->createElement('cac:Attachment');
                $embeddedDoc = $dom->createElement('cbc:EmbeddedDocumentBinaryObject', $qrCodeData);
                $embeddedDoc->setAttribute('mimeCode', 'text/plain');
                $embeddedDoc->setAttribute('filename', "qr_code_{$invoice->invoice_number}.txt");
                $attachment->appendChild($embeddedDoc);
                $additionalDocRef->appendChild($attachment);
                
                $invoiceElement->appendChild($additionalDocRef);
                
                Log::info('تم إضافة QR Code متوافق مع ZATCA إلى XML', [
                    'invoice_id' => $invoice->id,
                    'qr_length' => strlen($qrCodeData),
                    'qr_tlv_size' => strlen(base64_decode($qrCodeData)),
                    'includes_phase2_data' => true
                ]);
                
                // حفظ QR Code في قاعدة البيانات
                if (Schema::hasColumn('invoices', 'qr_code_data')) {
                    $invoice->qr_code_data = $qrCodeData;
                } else {
                    // استخدام الحقل القديم إذا لم يكن الجديد متوفراً
                    $invoice->qr_code = $qrCodeData;
                }
                $invoice->save();
                
            } else {
                Log::warning('QR Code فارغ - سيتم استخدام QR Code بديل', [
                    'invoice_id' => $invoice->id
                ]);
                
                // إنشاء QR Code بديل أساسي
                $this->addFallbackQRCodeToXml($dom, $invoiceElement, $invoice);
            }
        } catch (\Exception $e) {
            Log::error('فشل في إضافة QR Code إلى XML', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // محاولة إضافة QR Code بديل
            try {
                $this->addFallbackQRCodeToXml($dom, $invoiceElement, $invoice);
            } catch (\Exception $fallbackError) {
                Log::error('فشل في إضافة QR Code البديل أيضاً', [
                    'invoice_id' => $invoice->id,
                    'fallback_error' => $fallbackError->getMessage()
                ]);
            }
        }
    }
    
    /**
     * إضافة QR Code بديل بسيط في حالة فشل QR Code الرئيسي
     */
    private function addFallbackQRCodeToXml(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        try {
            // إنشاء QR Code بديل بسيط بالعناصر الأساسية فقط
            $fallbackQRData = $this->generateFallbackQRCode($invoice);
            
            if (!empty($fallbackQRData)) {
                $additionalDocRef = $dom->createElement('cac:AdditionalDocumentReference');
                $additionalDocRef->appendChild($dom->createElement('cbc:ID', 'QR'));
                $additionalDocRef->appendChild($dom->createElement('cbc:UUID', $invoice->uuid));
                $additionalDocRef->appendChild($dom->createElement('cbc:DocumentType', 'QRCode'));
                $additionalDocRef->appendChild($dom->createElement('cbc:DocumentDescription', 'Fallback QR Code'));
                
                $attachment = $dom->createElement('cac:Attachment');
                $embeddedDoc = $dom->createElement('cbc:EmbeddedDocumentBinaryObject', $fallbackQRData);
                $embeddedDoc->setAttribute('mimeCode', 'text/plain');
                $attachment->appendChild($embeddedDoc);
                $additionalDocRef->appendChild($attachment);
                
                $invoiceElement->appendChild($additionalDocRef);
                
                Log::info('تم إضافة QR Code بديل بنجاح', [
                    'invoice_id' => $invoice->id,
                    'qr_length' => strlen($fallbackQRData)
                ]);
            }
        } catch (\Exception $e) {
            Log::error('فشل في إضافة QR Code البديل', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * إضافة امتداد التوقيع الرقمي
     */
    private function addDigitalSignatureExtension(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        try {
            // إضافة UBL Extensions للتوقيع الرقمي
            $ublExtensions = $dom->createElement('ext:UBLExtensions');
            $ublExtension = $dom->createElement('ext:UBLExtension');
            $extensionURI = $dom->createElement('ext:ExtensionURI', 'urn:oasis:names:specification:ubl:dsig:enveloped:xades');
            $ublExtension->appendChild($extensionURI);
            
            $extensionContent = $dom->createElement('ext:ExtensionContent');
            
            // إضافة بلوك التوقيع الأساسي
            $signature = $dom->createElementNS('http://www.w3.org/2000/09/xmldsig#', 'ds:Signature');
            $signature->setAttribute('Id', 'signature');
            
            // Signed Info
            $signedInfo = $dom->createElement('ds:SignedInfo');
            $canonicalizationMethod = $dom->createElement('ds:CanonicalizationMethod');
            $canonicalizationMethod->setAttribute('Algorithm', 'http://www.w3.org/TR/2001/REC-xml-c14n-20010315');
            $signedInfo->appendChild($canonicalizationMethod);
            
            $signatureMethod = $dom->createElement('ds:SignatureMethod');
            $signatureMethod->setAttribute('Algorithm', 'http://www.w3.org/2001/04/xmldsig-more#ecdsa-sha256');
            $signedInfo->appendChild($signatureMethod);
            
            $reference = $dom->createElement('ds:Reference');
            $reference->setAttribute('Id', 'invoiceSignedData');
            $reference->setAttribute('URI', '');
            
            $transforms = $dom->createElement('ds:Transforms');
            $transform1 = $dom->createElement('ds:Transform');
            $transform1->setAttribute('Algorithm', 'http://www.w3.org/2000/09/xmldsig#enveloped-signature');
            $transforms->appendChild($transform1);
            
            $transform2 = $dom->createElement('ds:Transform');
            $transform2->setAttribute('Algorithm', 'http://www.w3.org/TR/2001/REC-xml-c14n-20010315');
            $transforms->appendChild($transform2);
            $reference->appendChild($transforms);
            
            $digestMethod = $dom->createElement('ds:DigestMethod');
            $digestMethod->setAttribute('Algorithm', 'http://www.w3.org/2001/04/xmlenc#sha256');
            $reference->appendChild($digestMethod);
            
            // توليد digest value حقيقي
            $digestValue = $dom->createElement('ds:DigestValue', $this->generateDigestValue($dom, $invoice));
            $reference->appendChild($digestValue);
            
            $signedInfo->appendChild($reference);
            $signature->appendChild($signedInfo);
            
            // Signature Value - توليد قيمة توقيع حقيقية
            $signatureValue = $dom->createElement('ds:SignatureValue', $this->generateSignatureValue($dom, $invoice));
            $signature->appendChild($signatureValue);
            
            // Key Info - استخدام الشهادة الحقيقية
            $keyInfo = $dom->createElement('ds:KeyInfo');
            $x509Data = $dom->createElement('ds:X509Data');
            $x509Certificate = $dom->createElement('ds:X509Certificate', $this->getCleanCertificate());
            $x509Data->appendChild($x509Certificate);
            $keyInfo->appendChild($x509Data);
            $signature->appendChild($keyInfo);
            
            $extensionContent->appendChild($signature);
            $ublExtension->appendChild($extensionContent);
            $ublExtensions->appendChild($ublExtension);
            
            // إدراج UBLExtensions في بداية الوثيقة
            $firstChild = $invoiceElement->firstChild;
            $invoiceElement->insertBefore($ublExtensions, $firstChild);
            
            Log::info('تم إضافة امتداد التوقيع الرقمي إلى XML', [
                'invoice_id' => $invoice->id
            ]);
            
        } catch (\Exception $e) {
            Log::warning('فشل في إضافة امتداد التوقيع الرقمي', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * توليد قيمة Digest للتوقيع الرقمي
     */
    private function generateDigestValue(\DOMDocument $dom, Invoice $invoice): string
    {
        try {
            // إنشاء نسخة من DOM بدون التوقيع لحساب الـ digest
            $tempDom = clone $dom;
            
            // إزالة عنصر التوقيع إذا كان موجوداً
            $signatures = $tempDom->getElementsByTagName('Signature');
            if ($signatures->length > 0) {
                for ($i = $signatures->length - 1; $i >= 0; $i--) {
                    $signatures->item($i)->parentNode->removeChild($signatures->item($i));
                }
            }
            
            // تطبيق canonical XML
            $canonicalXml = $tempDom->C14N();
            
            // حساب SHA256 hash
            $hash = hash('sha256', $canonicalXml, true);
            
            return base64_encode($hash);
            
        } catch (\Exception $e) {
            Log::warning('فشل في توليد digest value', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            
            // قيمة احتياطية
            return base64_encode(hash('sha256', $invoice->uuid . time(), true));
        }
    }
    
    /**
     * توليد قيمة التوقيع الرقمي
     */
    private function generateSignatureValue(\DOMDocument $dom, Invoice $invoice): string
    {
        try {
            if (empty($this->settings->private_key)) {
                Log::warning('المفتاح الخاص غير متوفر لتوليد التوقيع');
                return base64_encode('TEMP_SIGNATURE_' . hash('sha256', $invoice->uuid . time()));
            }
            
            // البيانات التي سيتم توقيعها
            $dataToSign = $this->generateDigestValue($dom, $invoice);
            
            // تحميل المفتاح الخاص
            $privateKey = openssl_pkey_get_private($this->settings->private_key, $this->settings->certificate_secret ?? '');
            
            if ($privateKey === false) {
                Log::warning('فشل في تحميل المفتاح الخاص');
                return base64_encode('TEMP_SIGNATURE_' . hash('sha256', $invoice->uuid . time()));
            }
            
            $signature = '';
            $success = openssl_sign($dataToSign, $signature, $privateKey, OPENSSL_ALGO_SHA256);
            
            if ($success) {
                return base64_encode($signature);
            } else {
                Log::warning('فشل في توقيع البيانات');
                return base64_encode('TEMP_SIGNATURE_' . hash('sha256', $invoice->uuid . time()));
            }
            
        } catch (\Exception $e) {
            Log::error('خطأ في توليد التوقيع الرقمي', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            
            return base64_encode('TEMP_SIGNATURE_' . hash('sha256', $invoice->uuid . time()));
        }
    }
    
    /**
     * الحصول على الشهادة بدون headers
     */
    private function getCleanCertificate(): string
    {
        try {
            if (empty($this->settings->certificate)) {
                Log::error('الشهادة غير متوفرة في إعدادات ZATCA');
                throw new \Exception('Certificate not available in ZATCA settings');
            }
            
            // استخدام خدمة التحقق من الشهادات الجديدة
            $certificateValidator = new ZatcaCertificateValidator();
            
            Log::info('بدء التحقق من صحة الشهادة باستخدام ZatcaCertificateValidator');
            
            $validationResult = $certificateValidator->validateAndCleanCertificate($this->settings->certificate);
            
            if (!$validationResult['success']) {
                $errors = implode(', ', $validationResult['errors']);
                Log::error('فشل التحقق من صحة الشهادة', [
                    'errors' => $errors,
                    'validation_checks' => $validationResult['validation_checks']
                ]);
                throw new \Exception('Certificate validation failed: ' . $errors);
            }
            
            $cleanedCertificate = $validationResult['cleaned_certificate'];
            
            if (empty($cleanedCertificate)) {
                Log::error('الشهادة المنظفة فارغة');
                throw new \Exception('Cleaned certificate is empty');
            }
            
            // تسجيل معلومات الشهادة للتتبع
            if (!empty($validationResult['certificate_info'])) {
                Log::info('معلومات الشهادة المعالجة', [
                    'subject' => $validationResult['certificate_info']['subject']['CN'] ?? 'Unknown',
                    'issuer' => $validationResult['certificate_info']['issuer']['CN'] ?? 'Unknown',
                    'valid_from' => isset($validationResult['certificate_info']['valid_from'])
                        ? date('Y-m-d', $validationResult['certificate_info']['valid_from']) : 'Unknown',
                    'valid_to' => isset($validationResult['certificate_info']['valid_to'])
                        ? date('Y-m-d', $validationResult['certificate_info']['valid_to']) : 'Unknown',
                    'is_expired' => $validationResult['validation_checks']['openssl']['is_expired'] ?? false
                ]);
            }
            
            // التحقق من التوافق مع ZATCA
            $zatcaCompliance = $validationResult['validation_checks']['zatca_compliance'] ?? [];
            if (!($zatcaCompliance['zatca_ready'] ?? true)) {
                $issues = implode(', ', $zatcaCompliance['issues'] ?? []);
                Log::warning('الشهادة قد لا تكون متوافقة تماماً مع ZATCA', [
                    'issues' => $issues,
                    'recommendations' => $zatcaCompliance['recommendations'] ?? []
                ]);
                // لا نفشل هنا، فقط تحذير
            }
            
            Log::info('تم التحقق من الشهادة وتنظيفها بنجاح', [
                'cleaned_certificate_length' => strlen($cleanedCertificate),
                'validation_passed' => true,
                'zatca_ready' => $zatcaCompliance['zatca_ready'] ?? false
            ]);
            
            return $cleanedCertificate;
            
        } catch (\Exception $e) {
            Log::error('فشل في معالجة الشهادة', [
                'error' => $e->getMessage(),
                'has_certificate' => !empty($this->settings->certificate),
                'certificate_length' => strlen($this->settings->certificate ?? ''),
                'trace' => $e->getTraceAsString()
            ]);
            
            throw new \Exception('Certificate processing failed: ' . $e->getMessage());
        }
    }

    /**
     * دالة محسنة لتوليد QR Code مع دعم مكتبة endroid/qr-code
     */
    public function generateEnhancedQRCode(Invoice $invoice): array
    {
        try {
            // توليد البيانات النصية للـ QR Code
            $qrTextData = $this->generateQRCode($invoice);
            
            $result = [
                'base64_data' => $qrTextData,
                'text_length' => strlen($qrTextData),
                'tlv_size' => strlen(base64_decode($qrTextData)),
                'generated_at' => now()->toISOString()
            ];

            // إذا كانت مكتبة endroid/qr-code متوفرة، إنشاء صورة QR Code
            if (class_exists('\Endroid\QrCode\QrCode')) {
                $result['image_generated'] = $this->generateQRCodeImage($qrTextData, $invoice);
            } else {
                $result['image_generated'] = false;
                $result['note'] = 'مكتبة endroid/qr-code غير مثبتة - يمكن تثبيتها بـ composer require endroid/qr-code';
            }

            Log::info('تم توليد QR Code محسن', [
                'invoice_id' => $invoice->id,
                'tlv_size' => $result['tlv_size'],
                'image_created' => $result['image_generated']
            ]);

            return $result;

        } catch (\Exception $e) {
            Log::error('فشل في توليد QR Code محسن', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);

            return [
                'base64_data' => null,
                'error' => $e->getMessage(),
                'generated_at' => now()->toISOString()
            ];
        }
    }

    /**
     * توليد صورة QR Code باستخدام مكتبة endroid/qr-code
     */
    private function generateQRCodeImage(string $qrData, Invoice $invoice): bool
    {
        try {
            if (!class_exists('\Endroid\QrCode\QrCode')) {
                return false;
            }

            $qrCode = new \Endroid\QrCode\QrCode($qrData);
            $qrCode->setSize(300);
            $qrCode->setMargin(10);

            // إنشاء مجلد التخزين
            $qrDir = storage_path('app/invoices/qr_codes/');
            if (!is_dir($qrDir)) {
                mkdir($qrDir, 0777, true);
            }

            // حفظ الصورة
            $filename = "qr_invoice_{$invoice->invoice_number}.png";
            $filepath = $qrDir . $filename;
            
            $qrCode->writeFile($filepath);

            // تحديث مسار الصورة في قاعدة البيانات
            if (Schema::hasColumn('invoices', 'qr_code_image_path')) {
                $invoice->qr_code_image_path = "invoices/qr_codes/{$filename}";
                $invoice->save();
            }

            return true;

        } catch (\Exception $e) {
            Log::warning('فشل في إنشاء صورة QR Code', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * دالة شاملة لتوليد ومعالجة الفاتورة
     */
    public function generateCompleteInvoicePackage(Invoice $invoice, array $options = []): array
    {
        try {
            $package = [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'generated_at' => now()->toISOString(),
                'success' => false,
                'files_created' => [],
                'errors' => []
            ];

            // 1. توليد XML
            try {
                $xmlContent = $this->generateCompleteZatcaXml($invoice, $invoice->type ?? 'standard');
                $package['xml_generated'] = true;
                $package['xml_size'] = strlen($xmlContent);
                $package['files_created'][] = "invoice_{$invoice->invoice_number}.xml";
            } catch (\Exception $e) {
                $package['xml_generated'] = false;
                $package['errors'][] = 'XML Generation: ' . $e->getMessage();
            }

            // 2. توليد QR Code
            try {
                $qrResult = $this->generateEnhancedQRCode($invoice);
                $package['qr_generated'] = !empty($qrResult['base64_data']);
                $package['qr_tlv_size'] = $qrResult['tlv_size'] ?? 0;
                
                if ($qrResult['image_generated'] ?? false) {
                    $package['files_created'][] = "qr_invoice_{$invoice->invoice_number}.png";
                }
            } catch (\Exception $e) {
                $package['qr_generated'] = false;
                $package['errors'][] = 'QR Code Generation: ' . $e->getMessage();
            }

            // 3. محاولة التوقيع الرقمي
            if ($options['sign_xml'] ?? true) {
                try {
                    $this->signInvoiceXmlIfPossible($invoice);
                    $package['xml_signed'] = true;
                } catch (\Exception $e) {
                    $package['xml_signed'] = false;
                    $package['errors'][] = 'Digital Signature: ' . $e->getMessage();
                }
            }

            // 4. التحقق من التوافق
            try {
                $complianceCheck = $this->performComplianceCheck($invoice);
                $package['compliance_check'] = $complianceCheck;
                $package['zatca_compliant'] = $complianceCheck['is_compliant'] ?? false;
            } catch (\Exception $e) {
                $package['compliance_check'] = ['error' => $e->getMessage()];
                $package['zatca_compliant'] = false;
            }

            // تحديد النجاح العام
            $package['success'] = $package['xml_generated'] && $package['qr_generated'];

            Log::info('تم إنشاء حزمة فاتورة متكاملة', [
                'invoice_id' => $invoice->id,
                'success' => $package['success'],
                'files_count' => count($package['files_created']),
                'errors_count' => count($package['errors'])
            ]);

            return $package;

        } catch (\Exception $e) {
            Log::error('فشل في إنشاء حزمة الفاتورة المتكاملة', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);

            return [
                'invoice_id' => $invoice->id,
                'success' => false,
                'error' => $e->getMessage(),
                'generated_at' => now()->toISOString()
            ];
        }
    }

    /**
     * محاولة توقيع XML إذا أمكن
     */
    private function signInvoiceXmlIfPossible(Invoice $invoice): void
    {
        if (empty($this->settings->certificate) || empty($this->settings->private_key)) {
            throw new \Exception('الشهادة الرقمية أو المفتاح الخاص غير متوفر');
        }

        $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
        $signedXmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}_signed.xml");

        if (!file_exists($xmlPath)) {
            throw new \Exception('ملف XML غير موجود للتوقيع');
        }

        $this->signInvoiceWithFatoora($xmlPath, $signedXmlPath);
    }

    /**
     * فحص التوافق مع معايير ZATCA
     */
    private function performComplianceCheck(Invoice $invoice): array
    {
        $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
        
        if (!file_exists($xmlPath)) {
            return ['error' => 'ملف XML غير موجود للفحص'];
        }

        $xmlContent = file_get_contents($xmlPath);
        
        $checks = [
            'xml_valid' => $this->validateXmlSyntax($xmlContent),
            'required_namespaces' => $this->checkRequiredNamespaces($xmlContent),
            'ubl_structure' => $this->checkUblStructure($xmlContent),
            'zatca_fields' => $this->checkZatcaRequiredFields($xmlContent),
            'tax_calculation' => $this->validateTaxCalculation($invoice),
            'qr_code_present' => strpos($xmlContent, 'QR') !== false
        ];

        $passed = array_filter($checks);
        $isCompliant = count($passed) === count($checks);

        return [
            'is_compliant' => $isCompliant,
            'passed_checks' => count($passed),
            'total_checks' => count($checks),
            'checks' => $checks,
            'compliance_percentage' => round((count($passed) / count($checks)) * 100, 2)
        ];
    }

    /**
     * التحقق من صحة XML من ناحية التركيب
     */
    private function validateXmlSyntax(string $xmlContent): bool
    {
        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $isValid = $dom->loadXML($xmlContent);
        libxml_clear_errors();
        return $isValid;
    }

    /**
     * فحص وجود namespaces المطلوبة
     */
    private function checkRequiredNamespaces(string $xmlContent): bool
    {
        $requiredNamespaces = [
            'xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2"',
            'xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2"',
            'xmlns:ds="http://www.w3.org/2000/09/xmldsig#"'
        ];

        foreach ($requiredNamespaces as $ns) {
            if (strpos($xmlContent, $ns) === false) {
                return false;
            }
        }

        return true;
    }

    /**
     * فحص بنية UBL المطلوبة
     */
    private function checkUblStructure(string $xmlContent): bool
    {
        $requiredElements = [
            'UBLVersionID',
            'CustomizationID',
            'ProfileID',
            'SellerSupplierParty',
            'InvoiceLine',
            'TaxTotal',
            'LegalMonetaryTotal'
        ];

        foreach ($requiredElements as $element) {
            if (strpos($xmlContent, "<cbc:{$element}>") === false &&
                strpos($xmlContent, "<cac:{$element}>") === false) {
                return false;
            }
        }

        return true;
    }

    /**
     * فحص الحقول المطلوبة لـ ZATCA
     */
    private function checkZatcaRequiredFields(string $xmlContent): bool
    {
        $zatcaFields = [
            'UUID',
            'IssueDate',
            'IssueTime',
            'InvoiceTypeCode',
            'DocumentCurrencyCode',
            'TaxCurrencyCode'
        ];

        foreach ($zatcaFields as $field) {
            if (strpos($xmlContent, "<cbc:{$field}>") === false) {
                return false;
            }
        }

        return true;
    }

    /**
     * التحقق من صحة حساب الضرائب
     */
    private function validateTaxCalculation(Invoice $invoice): bool
    {
        $expectedTax = $invoice->subtotal * 0.15; // 15% VAT
        $actualTax = $invoice->tax_amount;
        
        // السماح بفرق بسيط بسبب التقريب
        $tolerance = 0.01;
        
        return abs($expectedTax - $actualTax) <= $tolerance;
    }

    /**
     * إنشاء تقرير شامل للفاتورة
     */
    public function generateInvoiceReport(Invoice $invoice): array
    {
        try {
            $report = [
                'invoice_info' => [
                    'id' => $invoice->id,
                    'number' => $invoice->invoice_number,
                    'type' => $invoice->type,
                    'issue_date' => $invoice->issue_date->toISOString(),
                    'subtotal' => $invoice->subtotal,
                    'tax_amount' => $invoice->tax_amount,
                    'total_amount' => $invoice->total_amount,
                    'uuid' => $invoice->uuid
                ],
                'files_status' => [
                    'xml_file' => $this->checkFileExists($invoice->xml_file_path),
                    'qr_image' => $this->checkFileExists($invoice->qr_code_image_path),
                    'signed_xml' => $this->checkFileExists(str_replace('.xml', '_signed.xml', $invoice->xml_file_path ?? ''))
                ],
                'compliance' => $this->performComplianceCheck($invoice),
                'qr_analysis' => $this->analyzeInvoiceQRCode($invoice),
                'zatca_readiness' => null,
                'recommendations' => []
            ];

            // تقييم الجاهزية لـ ZATCA
            $report['zatca_readiness'] = $this->assessZatcaReadiness($report);
            
            // إضافة توصيات للتحسين
            $report['recommendations'] = $this->generateRecommendations($report);

            return $report;

        } catch (\Exception $e) {
            return [
                'error' => 'فشل في إنشاء التقرير: ' . $e->getMessage(),
                'invoice_id' => $invoice->id
            ];
        }
    }

    private function checkFileExists(?string $path): bool
    {
        return !empty($path) && file_exists(storage_path('app/' . $path));
    }

    private function analyzeInvoiceQRCode(Invoice $invoice): array
    {
        try {
            $qrData = $this->generateQRCode($invoice);
            return $this->analyzeQRCodeContent($qrData);
        } catch (\Exception $e) {
            return ['error' => $e->getMessage()];
        }
    }

    private function assessZatcaReadiness(array $report): string
    {
        $complianceScore = $report['compliance']['compliance_percentage'] ?? 0;
        $hasFiles = $report['files_status']['xml_file'] ?? false;
        $hasQR = !empty($report['qr_analysis']['tags_found']) ?? false;

        if ($complianceScore >= 90 && $hasFiles && $hasQR) {
            return 'جاهز للإنتاج';
        } elseif ($complianceScore >= 70) {
            return 'يحتاج تحسينات بسيطة';
        } elseif ($complianceScore >= 50) {
            return 'يحتاج تحسينات متوسطة';
        } else {
            return 'يحتاج عمل كبير';
        }
    }

    /**
     * اختبار شامل لجميع إصلاحات ZATCA Business Rules
     */
    public function testAllZatcaBusinessRuleFixes(Invoice $invoice): array
        {
            try {
                Log::info('بدء اختبار شامل لجميع إصلاحات ZATCA Business Rules', [
                    'invoice_id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number
                ]);
    
                $testResults = [
                    'invoice_id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'test_timestamp' => now()->toISOString(),
                    'overall_success' => false,
                    'tests' => [],
                    'business_rules_compliance' => [],
                    'recommendations' => []
                ];
    
                // 1. اختبار توليد XML مع جميع الإصلاحات
                $testResults['tests']['xml_generation'] = $this->testXmlGenerationWithFixes($invoice);
    
                // 2. اختبار Business Process Type (BR-KSA-EN16931-01)
                $testResults['business_rules_compliance']['BR-KSA-EN16931-01'] = $this->validateBusinessProcessType($invoice);
    
                // 3. اختبار Invoice Counter (BR-KSA-33)
                $testResults['business_rules_compliance']['BR-KSA-33'] = $this->validateInstructionID($invoice);
    
                // 4. اختبار Invoice Code (BR-KSA-06)
                $testResults['business_rules_compliance']['BR-KSA-06'] = $this->validateInvoiceCodeKSA2($invoice);
    
                // 5. اختبار QR Code Phase 2
                $testResults['tests']['qr_code_phase2'] = $this->testPhase2QRCodeGeneration($invoice);
    
                // 6. اختبار UBL Party Structure
                $testResults['tests']['ubl_party_structure'] = $this->validateUBLPartyStructure($invoice);
    
                // 7. اختبار حسابات الضرائب
                $testResults['tests']['tax_calculations'] = $this->validateTaxCalculationCompliance($invoice);
    
                // 8. فحص التوافق الشامل
                $testResults['tests']['complete_compliance'] = $this->performCompleteComplianceCheck($invoice);
    
                // تحديد النجاح العام
                $testResults['overall_success'] = $this->calculateOverallSuccess($testResults);
    
                // إضافة توصيات للتحسين
                $testResults['recommendations'] = $this->generateZatcaRecommendations($testResults);
    
                Log::info('اكتمل الاختبار الشامل لإصلاحات ZATCA', [
                    'invoice_id' => $invoice->id,
                    'overall_success' => $testResults['overall_success'],
                    'compliance_score' => $this->calculateComplianceScore($testResults)
                ]);
    
                return $testResults;
    
            } catch (\Exception $e) {
                Log::error('فشل في الاختبار الشامل لإصلاحات ZATCA', [
                    'invoice_id' => $invoice->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
    
                return [
                    'invoice_id' => $invoice->id,
                    'overall_success' => false,
                    'error' => $e->getMessage(),
                    'test_timestamp' => now()->toISOString()
                ];
            }
        }
    
        /**
         * اختبار توليد XML مع جميع الإصلاحات
         */
        private function testXmlGenerationWithFixes(Invoice $invoice): array
        {
            try {
                $xmlContent = $this->generateCompleteZatcaXml($invoice, $invoice->type ?? 'standard');
                
                return [
                    'success' => true,
                    'xml_size' => strlen($xmlContent),
                    'xml_valid' => $this->validateXmlSyntax($xmlContent),
                    'contains_business_process' => strpos($xmlContent, 'BUSINESS_PROCESS') !== false,
                    'contains_instruction_id' => strpos($xmlContent, 'InstructionID') !== false,
                    'contains_ksa2_code' => strpos($xmlContent, 'KSA-2') !== false,
                    'contains_qr_code' => strpos($xmlContent, 'QR') !== false,
                    'file_saved' => !empty($invoice->xml_file_path)
                ];
            } catch (\Exception $e) {
                return [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
    
        /**
         * التحقق من Business Process Type (BR-KSA-EN16931-01)
         */
        private function validateBusinessProcessType(Invoice $invoice): array
        {
            try {
                $expectedProcess = ($invoice->type === 'simplified') ? 'KSA_B2C_Invoice' : 'KSA_B2B_Invoice';
                
                // فحص وجود Business Process في XML المُولد
                $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
                if (file_exists($xmlPath)) {
                    $xmlContent = file_get_contents($xmlPath);
                    $hasBusinessProcess = strpos($xmlContent, 'BUSINESS_PROCESS') !== false;
                    $hasCorrectType = strpos($xmlContent, $expectedProcess) !== false;
                    
                    return [
                        'compliant' => $hasBusinessProcess && $hasCorrectType,
                        'has_business_process_element' => $hasBusinessProcess,
                        'has_correct_type' => $hasCorrectType,
                        'expected_type' => $expectedProcess,
                        'rule' => 'BR-KSA-EN16931-01',
                        'description' => 'Business Process Specification must be present'
                    ];
                }
                
                return [
                    'compliant' => false,
                    'error' => 'XML file not found for validation',
                    'rule' => 'BR-KSA-EN16931-01'
                ];
                
            } catch (\Exception $e) {
                return [
                    'compliant' => false,
                    'error' => $e->getMessage(),
                    'rule' => 'BR-KSA-EN16931-01'
                ];
            }
        }
    
        /**
         * التحقق من Instruction ID (BR-KSA-33)
         */
        private function validateInstructionID(Invoice $invoice): array
        {
            try {
                $hasInstructionId = !empty($invoice->instruction_id);
                
                if ($hasInstructionId) {
                    // فحص وجود InstructionID في XML
                    $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
                    if (file_exists($xmlPath)) {
                        $xmlContent = file_get_contents($xmlPath);
                        $hasInXml = strpos($xmlContent, 'InstructionID') !== false;
                        $hasCorrectValue = strpos($xmlContent, $invoice->instruction_id) !== false;
                        
                        return [
                            'compliant' => $hasInXml && $hasCorrectValue,
                            'has_instruction_id' => $hasInstructionId,
                            'has_in_xml' => $hasInXml,
                            'has_correct_value' => $hasCorrectValue,
                            'instruction_id_value' => $invoice->instruction_id,
                            'rule' => 'BR-KSA-33',
                            'description' => 'InstructionID must be present and unique'
                        ];
                    }
                }
                
                return [
                    'compliant' => false,
                    'has_instruction_id' => $hasInstructionId,
                    'rule' => 'BR-KSA-33',
                    'description' => 'InstructionID must be present and unique'
                ];
                
            } catch (\Exception $e) {
                return [
                    'compliant' => false,
                    'error' => $e->getMessage(),
                    'rule' => 'BR-KSA-33'
                ];
            }
        }
    
        /**
         * التحقق من Invoice Code KSA-2 (BR-KSA-06)
         */
        private function validateInvoiceCodeKSA2(Invoice $invoice): array
        {
            try {
                $expectedCode = ($invoice->type === 'simplified') ? '02' : '01';
                
                // فحص وجود KSA-2 في XML
                $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
                if (file_exists($xmlPath)) {
                    $xmlContent = file_get_contents($xmlPath);
                    $hasKSA2Element = strpos($xmlContent, 'KSA-2') !== false;
                    $hasCorrectCode = strpos($xmlContent, $expectedCode) !== false;
                    
                    return [
                        'compliant' => $hasKSA2Element && $hasCorrectCode,
                        'has_ksa2_element' => $hasKSA2Element,
                        'has_correct_code' => $hasCorrectCode,
                        'expected_code' => $expectedCode,
                        'invoice_type' => $invoice->type,
                        'rule' => 'BR-KSA-06',
                        'description' => 'Invoice type code (KSA-2) must be present with correct value'
                    ];
                }
                
                return [
                    'compliant' => false,
                    'error' => 'XML file not found for validation',
                    'rule' => 'BR-KSA-06'
                ];
                
            } catch (\Exception $e) {
                return [
                    'compliant' => false,
                    'error' => $e->getMessage(),
                    'rule' => 'BR-KSA-06'
                ];
            }
        }
    
        /**
         * التحقق من UBL Party Structure
         */
        private function validateUBLPartyStructure(Invoice $invoice): array
        {
            try {
                $xmlPath = storage_path("app/invoices/zatca_xml/invoice_{$invoice->invoice_number}.xml");
                if (!file_exists($xmlPath)) {
                    return [
                        'success' => false,
                        'error' => 'XML file not found'
                    ];
                }
                
                $xmlContent = file_get_contents($xmlPath);
                $dom = new \DOMDocument();
                $dom->loadXML($xmlContent);
                
                // فحص ترتيب عناصر Party
                $supplierParties = $dom->getElementsByTagName('SellerSupplierParty');
                if ($supplierParties->length > 0) {
                    $party = $supplierParties->item(0)->getElementsByTagName('Party')->item(0);
                    if ($party) {
                        $children = $party->childNodes;
                        $elementOrder = [];
                        
                        foreach ($children as $child) {
                            if ($child->nodeType === XML_ELEMENT_NODE) {
                                $elementOrder[] = $child->localName;
                            }
                        }
                        
                        // الترتيب المطلوب حسب UBL 2.1
                        $expectedOrder = [
                            'PartyIdentification',
                            'PartyName',
                            'PostalAddress',
                            'PartyTaxScheme',
                            'PartyLegalEntity'
                        ];
                        
                        $orderCompliant = $this->checkElementOrder($elementOrder, $expectedOrder);
                        
                        return [
                            'success' => true,
                            'order_compliant' => $orderCompliant,
                            'actual_order' => array_unique($elementOrder),
                            'expected_order' => $expectedOrder,
                            'has_party_identification' => in_array('PartyIdentification', $elementOrder),
                            'has_party_name' => in_array('PartyName', $elementOrder),
                            'has_postal_address' => in_array('PostalAddress', $elementOrder)
                        ];
                    }
                }
                
                return [
                    'success' => false,
                    'error' => 'Party structure not found in XML'
                ];
                
            } catch (\Exception $e) {
                return [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
    
        /**
         * فحص ترتيب العناصر
         */
        private function checkElementOrder(array $actual, array $expected): bool
        {
            $actualFiltered = array_values(array_intersect($actual, $expected));
            $expectedFiltered = array_values(array_intersect($expected, $actual));
            
            return $actualFiltered === $expectedFiltered;
        }
    
        /**
         * التحقق من حسابات الضرائب
         */
        private function validateTaxCalculationCompliance(Invoice $invoice): array
        {
            try {
                $netAmount = $this->calculateNetInvoiceAmount($invoice);
                $expectedTax = $netAmount * 0.15;
                $actualTax = $invoice->tax_amount;
                $tolerance = 0.01;
                
                $taxCalculationCorrect = abs($expectedTax - $actualTax) <= $tolerance;
                
                return [
                    'success' => true,
                    'tax_calculation_correct' => $taxCalculationCorrect,
                    'net_amount' => $netAmount,
                    'expected_tax' => $expectedTax,
                    'actual_tax' => $actualTax,
                    'difference' => abs($expectedTax - $actualTax),
                    'tolerance' => $tolerance,
                    'total_amount_correct' => abs(($netAmount + $expectedTax) - $invoice->total_amount) <= $tolerance
                ];
                
            } catch (\Exception $e) {
                return [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
    
        /**
         * فحص التوافق الشامل
         */
        private function performCompleteComplianceCheck(Invoice $invoice): array
        {
            try {
                return $this->performComplianceCheck($invoice);
            } catch (\Exception $e) {
                return [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }
    
        /**
         * حساب النجاح العام
         */
        private function calculateOverallSuccess(array $testResults): bool
        {
            $successCount = 0;
            $totalTests = 0;
            
            // فحص نتائج الاختبارات
            foreach ($testResults['tests'] as $test) {
                $totalTests++;
                if ($test['success'] ?? false) {
                    $successCount++;
                }
            }
            
            // فحص نتائج Business Rules
            foreach ($testResults['business_rules_compliance'] as $rule) {
                $totalTests++;
                if ($rule['compliant'] ?? false) {
                    $successCount++;
                }
            }
            
            return $totalTests > 0 && ($successCount / $totalTests) >= 0.8; // 80% نجاح مطلوب
        }
    
        /**
         * حساب نقاط التوافق
         */
        private function calculateComplianceScore(array $testResults): float
        {
            $successCount = 0;
            $totalTests = 0;
            
            foreach ($testResults['tests'] as $test) {
                $totalTests++;
                if ($test['success'] ?? false) {
                    $successCount++;
                }
            }
            
            foreach ($testResults['business_rules_compliance'] as $rule) {
                $totalTests++;
                if ($rule['compliant'] ?? false) {
                    $successCount++;
                }
            }
            
            return $totalTests > 0 ? round(($successCount / $totalTests) * 100, 2) : 0;
        }
    
        /**
         * توليد توصيات ZATCA
         */
        private function generateZatcaRecommendations(array $testResults): array
        {
            $recommendations = [];
            
            // فحص Business Rules
            foreach ($testResults['business_rules_compliance'] as $ruleId => $rule) {
                if (!($rule['compliant'] ?? false)) {
                    switch ($ruleId) {
                        case 'BR-KSA-EN16931-01':
                            $recommendations[] = 'إضافة Business Process Type المطلوب للفاتورة';
                            break;
                        case 'BR-KSA-33':
                            $recommendations[] = 'إضافة InstructionID فريد للفاتورة';
                            break;
                        case 'BR-KSA-06':
                            $recommendations[] = 'إضافة رمز نوع الفاتورة (KSA-2) الصحيح';
                            break;
                    }
                }
            }
            
            // فحص الاختبارات الأخرى
            if (!($testResults['tests']['qr_code_phase2']['success'] ?? false)) {
                $recommendations[] = 'تحسين QR Code للتوافق مع المرحلة الثانية';
            }
            
            if (!($testResults['tests']['ubl_party_structure']['order_compliant'] ?? false)) {
                $recommendations[] = 'إصلاح ترتيب عناصر Party حسب معيار UBL 2.1';
            }
            
            if (!($testResults['tests']['tax_calculations']['tax_calculation_correct'] ?? false)) {
                $recommendations[] = 'تصحيح حسابات الضرائب لتتوافق مع معايير ZATCA';
            }
            
            if (empty($recommendations)) {
                $recommendations[] = 'الفاتورة متوافقة مع جميع متطلبات ZATCA!';
            }
            
            return $recommendations;
        }

    private function generateRecommendations(array $report): array
    {
        $recommendations = [];

        if (!($report['files_status']['xml_file'] ?? false)) {
            $recommendations[] = 'إنشاء ملف XML متوافق مع ZATCA';
        }

        if (!($report['files_status']['signed_xml'] ?? false)) {
            $recommendations[] = 'توقيع XML رقمياً باستخدام شهادة معتمدة';
        }

        if (($report['compliance']['compliance_percentage'] ?? 0) < 90) {
            $recommendations[] = 'تحسين مستوى التوافق مع معايير ZATCA';
        }

        if (empty($report['qr_analysis']['tags_found'] ?? [])) {
            $recommendations[] = 'إضافة QR Code صحيح متوافق مع المرحلة الثانية';
        }

        if (empty($recommendations)) {
            $recommendations[] = 'الفاتورة جاهزة للاستخدام مع ZATCA';
        }

        return $recommendations;
    }

    /**
     * حفظ ملف XML باسم مناسب
     */
    private function saveXmlFile(string $xmlContent, Invoice $invoice): void
    {
        try {
            // إنشاء مجلد التخزين
            $invoiceDir = storage_path('app/invoices/zatca_xml/');
            if (!is_dir($invoiceDir)) {
                mkdir($invoiceDir, 0777, true);
            }
            
            // اسم الملف
            $filename = "invoice_{$invoice->invoice_number}.xml";
            $filepath = $invoiceDir . $filename;
            
            // حفظ الملف
            file_put_contents($filepath, $xmlContent);
            
            // تحديث مسار الملف في قاعدة البيانات
            $invoice->xml_file_path = "invoices/zatca_xml/{$filename}";
            $invoice->save();
            
            Log::info('تم حفظ ملف XML بنجاح', [
                'invoice_id' => $invoice->id,
                'filename' => $filename,
                'filepath' => $filepath,
                'file_size' => filesize($filepath)
            ]);
            
        } catch (\Exception $e) {
            Log::error('فشل في حفظ ملف XML', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * إضافة عناصر التوقيع المطلوبة لـ ZATCA
     */
    private function addSignatureElements(\DOMDocument $dom, \DOMElement $invoiceElement, Invoice $invoice): void
    {
        // إضافة Signature Information (KSA-15) - يجب أن يأتي بعد العناصر الأساسية
        $signature = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2', 'cac:Signature');
        
        // إضافة معرف التوقيع (BR-KSA-28)
        $signatureId = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2', 'cbc:ID', 'urn:oasis:names:specification:ubl:signature:Invoice');
        $signature->appendChild($signatureId);
        
        // إضافة طريقة التوقيع (BR-KSA-30)
        $signatureMethod = $dom->createElementNS('urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2', 'cbc:SignatureMethod', 'urn:oasis:names:specification:ubl:dsig:enveloped:xades');
        $signature->appendChild($signatureMethod);
        
        $invoiceElement->appendChild($signature);
        
        Log::info('تم إضافة عناصر التوقيع المطلوبة لـ ZATCA', [
            'invoice_id' => $invoice->id
        ]);
    }
}