<?php

namespace App\Services;

use App\Models\DeviceInitializationLog;
use App\Models\ZatcaSettings;
use App\Models\Company;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class ZatcaDeviceService
{
    protected $config;

    public function __construct()
    {
        $this->config = config('zatca');
    }

    /**
     * طلب رمز OTP للمرحلة الثانية من التكامل
     */
    public function requestOTP(Company $company, string $csidNumber): array
    {
        try {
            // التحقق من معدل الطلبات
            if (!$this->checkRateLimit($company->id)) {
                return [
                    'success' => false,
                    'message' => 'تم تجاوز الحد الأقصى للطلبات في الساعة الواحدة'
                ];
            }

            // الحصول على إعدادات ZATCA للشركة
            $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();
            
            if (!$zatcaSettings) {
                return [
                    'success' => false,
                    'message' => 'لا توجد إعدادات ZATCA للشركة المحددة'
                ];
            }

            // تحضير البيانات للطلب
            $requestData = [
                'csr' => $csidNumber,
                'vat_number' => $company->vat_number,
                'company_name' => $company->name,
            ];

            // إرسال الطلب إلى ZATCA API
            $response = $this->sendOTPRequest($zatcaSettings, $requestData);

            if ($response['success']) {
                // تسجيل الطلب الناجح
                $this->incrementRateLimit($company->id);
                
                return [
                    'success' => true,
                    'message' => 'تم إرسال رمز التفعيل بنجاح',
                    'otp_token' => $response['otp_token'],
                    'expires_in' => $this->config['device_initialization']['otp_expiry_minutes'] * 60
                ];
            }

            return $response;

        } catch (\Exception $e) {
            Log::error('ZATCA OTP Request Error: ' . $e->getMessage(), [
                'company_id' => $company->id,
                'csid_number' => $csidNumber,
            ]);

            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء طلب رمز التفعيل'
            ];
        }
    }

    /**
     * التحقق من رمز OTP وإكمال تهيئة الجهاز
     */
    public function verifyOTP(DeviceInitializationLog $log, string $otpCode): array
    {
        try {
            // التحقق من انتهاء صلاحية OTP
            if ($log->isOtpExpired()) {
                return [
                    'success' => false,
                    'message' => 'انتهت صلاحية رمز التفعيل'
                ];
            }

            $company = $log->company;
            $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();

            // إرسال طلب التحقق إلى ZATCA
            $verificationData = [
                'otp' => $otpCode,
                'otp_token' => $log->otp_token,
                'request_id' => $log->id,
            ];

            $response = $this->sendOTPVerification($zatcaSettings, $verificationData);

            if ($response['success']) {
                // تحديث إعدادات ZATCA بالشهادة الجديدة
                $this->updateZatcaSettings($zatcaSettings, $response['certificate_data']);

                return [
                    'success' => true,
                    'message' => 'تم التحقق من رمز التفعيل بنجاح',
                    'certificate' => $response['certificate_data']['certificate'],
                    'private_key' => $response['certificate_data']['private_key'],
                    'expires_at' => $response['certificate_data']['expires_at'],
                ];
            }

            return $response;

        } catch (\Exception $e) {
            Log::error('ZATCA OTP Verification Error: ' . $e->getMessage(), [
                'log_id' => $log->id,
                'company_id' => $log->company_id,
            ]);

            return [
                'success' => false,
                'message' => 'حدث خطأ أثناء التحقق من رمز التفعيل'
            ];
        }
    }

    /**
     * إرسال طلب OTP إلى ZATCA API
     */
    protected function sendOTPRequest(ZatcaSettings $settings, array $data): array
    {
        // في البيئة الحقيقية، سيتم إرسال الطلب إلى ZATCA API
        // هنا نقوم بمحاكاة الاستجابة للأغراض التوضيحية
        
        return $this->simulateOTPResponse();
    }

    /**
     * إرسال طلب التحقق من OTP إلى ZATCA API
     */
    protected function sendOTPVerification(ZatcaSettings $settings, array $data): array
    {
        // محاكاة التحقق من OTP
        return $this->simulateVerificationResponse($data['otp']);
    }

    /**
     * تحديث إعدادات ZATCA بالشهادة الجديدة
     */
    protected function updateZatcaSettings(ZatcaSettings $settings, array $certificateData): void
    {
        $settings->update([
            'certificate_content' => $certificateData['certificate'],
            'private_key_content' => $certificateData['private_key'],
            'certificate_expires_at' => $certificateData['expires_at'],
            'connection_status' => 'connected',
            'is_active' => true,
            'last_connection_test' => now(),
            'last_error' => null,
        ]);
    }

    /**
     * التحقق من معدل الطلبات
     */
    protected function checkRateLimit(int $companyId): bool
    {
        $key = "zatca_otp_rate_limit_{$companyId}";
        $requests = Cache::get($key, 0);
        
        return $requests < $this->config['device_initialization']['rate_limit_per_hour'];
    }

    /**
     * زيادة عداد معدل الطلبات
     */
    protected function incrementRateLimit(int $companyId): void
    {
        $key = "zatca_otp_rate_limit_{$companyId}";
        $requests = Cache::get($key, 0);
        Cache::put($key, $requests + 1, now()->addHour());
    }

    /**
     * محاكاة استجابة طلب OTP (للأغراض التوضيحية)
     */
    protected function simulateOTPResponse(): array
    {
        // محاكاة نجاح 90% من الوقت
        if (rand(1, 10) <= 9) {
            return [
                'success' => true,
                'otp_token' => 'otp_' . uniqid() . '_' . time(),
                'message' => 'تم إرسال رمز التفعيل بنجاح'
            ];
        }

        return [
            'success' => false,
            'message' => 'فشل في إرسال رمز التفعيل من ZATCA'
        ];
    }

    /**
     * محاكاة استجابة التحقق من OTP (للأغراض التوضيحية)
     */
    protected function simulateVerificationResponse(string $otpCode): array
    {
        // رمز تجريبي للنجاح
        if ($otpCode === '123456' || rand(1, 10) <= 8) {
            return [
                'success' => true,
                'certificate_data' => [
                    'certificate' => $this->generateMockCertificate(),
                    'private_key' => $this->generateMockPrivateKey(),
                    'expires_at' => now()->addYear(),
                ]
            ];
        }

        return [
            'success' => false,
            'message' => 'رمز التفعيل غير صحيح'
        ];
    }

    /**
     * إنشاء شهادة تجريبية
     */
    protected function generateMockCertificate(): string
    {
        return "-----BEGIN CERTIFICATE-----\n" .
               "MIICljCCAX4CAQAwDQYJKoZIhvcNAQENBQAwFzEVMBMGA1UEAwwMTW9jayBDQSBD\n" .
               "ZXJ0MB4XDTI0MDEwMTAwMDAwMFoXDTI1MDEwMTAwMDAwMFowFzEVMBMGA1UEAwwM\n" .
               "TW9jayBDZXJ0aWZpY2F0ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB\n" .
               "AL8xb9+YQHl2F3k9JH8dY2cP5x7J2kL9mN6oP8qR5sT3uV7wX9zA2bC4dE5fG6hI\n" .
               "-----END CERTIFICATE-----";
    }

    /**
     * إنشاء مفتاح خاص تجريبي
     */
    protected function generateMockPrivateKey(): string
    {
        return "-----BEGIN PRIVATE KEY-----\n" .
               "MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQC/MW/fmEB5dhd5\n" .
               "PSR/HWNnD+ceydpC/ZjeqD/KkebE97le8F/cwNmwuHROXxuoSIyu5TPJzvaUNKkd\n" .
               "-----END PRIVATE KEY-----";
    }
}