<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Exception;

/**
 * ZATCA Certificate Validator Service
 * 
 * This service handles certificate validation and verification for ZATCA compliance,
 * including certificate and private key compatibility checks, certificate parsing,
 * and ZATCA-specific validations.
 */
class ZatcaCertificateValidator
{
    /**
     * Validate certificate and private key pair compatibility
     *
     * @param string $certificate Certificate in PEM format
     * @param string $privateKey Private key in PEM format
     * @param string|null $passphrase Optional passphrase for the private key
     * @return array Validation result with compatibility status and details
     */
    public function validateCertificateKeyPair(string $certificate, string $privateKey, ?string $passphrase = null): array
    {
        try {
            Log::info('Starting certificate and private key compatibility validation');

            $result = [
                'compatible' => false,
                'error' => null,
                'certificate_info' => null,
                'private_key_info' => null,
                'validation_details' => []
            ];

            // Validate certificate format and parse it
            $certResource = openssl_x509_read($certificate);
            if ($certResource === false) {
                $error = 'Invalid certificate format or corrupted certificate';
                Log::error($error);
                return array_merge($result, ['error' => $error]);
            }

            // Parse certificate information
            $certInfo = openssl_x509_parse($certResource);
            if ($certInfo === false) {
                $error = 'Failed to parse certificate information';
                Log::error($error);
                return array_merge($result, ['error' => $error]);
            }

            $result['certificate_info'] = $certInfo;
            $result['validation_details']['certificate_parsed'] = true;

            // Validate private key format
            $privateKeyResource = openssl_pkey_get_private($privateKey, $passphrase ?? '');
            if ($privateKeyResource === false) {
                $error = 'Invalid private key format, corrupted key, or incorrect passphrase';
                Log::error($error, ['openssl_errors' => $this->getOpenSSLErrors()]);
                return array_merge($result, ['error' => $error]);
            }

            // Get private key details
            $privateKeyDetails = openssl_pkey_get_details($privateKeyResource);
            if ($privateKeyDetails === false) {
                $error = 'Failed to get private key details';
                Log::error($error);
                return array_merge($result, ['error' => $error]);
            }

            $result['private_key_info'] = [
                'bits' => $privateKeyDetails['bits'],
                'type' => $privateKeyDetails['type'],
                'key_type' => $this->getKeyTypeString($privateKeyDetails['type'])
            ];
            $result['validation_details']['private_key_parsed'] = true;

            // Extract public key from certificate
            $publicKeyFromCert = openssl_pkey_get_public($certResource);
            if ($publicKeyFromCert === false) {
                $error = 'Failed to extract public key from certificate';
                Log::error($error);
                return array_merge($result, ['error' => $error]);
            }

            $publicKeyFromCertDetails = openssl_pkey_get_details($publicKeyFromCert);
            if ($publicKeyFromCertDetails === false) {
                $error = 'Failed to get public key details from certificate';
                Log::error($error);
                return array_merge($result, ['error' => $error]);
            }

            $result['validation_details']['public_key_extracted'] = true;

            // Check if the private key corresponds to the certificate's public key
            $privateKeyPublic = $privateKeyDetails['key'];
            $certPublicKey = $publicKeyFromCertDetails['key'];

            // Compare the public keys
            if ($privateKeyPublic === $certPublicKey) {
                $result['compatible'] = true;
                $result['validation_details']['keys_match'] = true;
                Log::info('Certificate and private key are compatible');
            } else {
                // Additional check using signature verification
                $testData = 'test_compatibility_' . time();
                $signature = '';
                
                // Sign test data with private key
                $signResult = openssl_sign($testData, $signature, $privateKeyResource, OPENSSL_ALGO_SHA256);
                if ($signResult) {
                    // Verify signature with public key from certificate
                    $verifyResult = openssl_verify($testData, $signature, $publicKeyFromCert, OPENSSL_ALGO_SHA256);
                    if ($verifyResult === 1) {
                        $result['compatible'] = true;
                        $result['validation_details']['keys_match'] = true;
                        $result['validation_details']['signature_verification_passed'] = true;
                        Log::info('Certificate and private key are compatible (verified through signature test)');
                    } else {
                        $result['error'] = 'Private key does not correspond to the certificate public key';
                        $result['validation_details']['keys_match'] = false;
                        $result['validation_details']['signature_verification_passed'] = false;
                        Log::error('Certificate and private key are not compatible - signature verification failed');
                    }
                } else {
                    $result['error'] = 'Failed to create test signature for compatibility check';
                    Log::error('Failed to sign test data for compatibility verification');
                }
            }

            // Additional certificate validations
            $result['validation_details']['certificate_validity'] = $this->validateCertificateValidity($certInfo);
            $result['validation_details']['certificate_purpose'] = $this->validateCertificatePurpose($certInfo);

            Log::info('Certificate and private key validation completed', [
                'compatible' => $result['compatible'],
                'certificate_subject' => $certInfo['subject']['CN'] ?? 'Unknown',
                'certificate_issuer' => $certInfo['issuer']['CN'] ?? 'Unknown'
            ]);

            return $result;

        } catch (Exception $e) {
            $error = 'Exception during certificate validation: ' . $e->getMessage();
            Log::error($error, [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'compatible' => false,
                'error' => $error,
                'certificate_info' => null,
                'private_key_info' => null,
                'validation_details' => []
            ];
        }
    }

    /**
     * Validate and clean certificate for ZATCA compliance
     *
     * @param string $certificate Certificate content
     * @return array Validation result with cleaned certificate
     */
    public function validateAndCleanCertificate(string $certificate): array
    {
        try {
            Log::info('Starting ZATCA certificate validation and cleaning');

            $result = [
                'success' => false,
                'cleaned_certificate' => null,
                'certificate_info' => null,
                'validation_checks' => [],
                'errors' => [],
                'warnings' => []
            ];

            // Clean and normalize certificate format
            $cleanedCert = $this->cleanCertificateFormat($certificate);
            if (empty($cleanedCert)) {
                $result['errors'][] = 'Failed to clean certificate format';
                return $result;
            }

            // Validate certificate with OpenSSL
            $opensslValidation = $this->validateWithOpenSSL($cleanedCert);
            $result['validation_checks']['openssl'] = $opensslValidation;

            if (!$opensslValidation['valid']) {
                $result['errors'][] = 'Certificate failed OpenSSL validation: ' . $opensslValidation['error'];
                return $result;
            }

            // Parse certificate information
            $certInfo = $opensslValidation['certificate_info'];
            $result['certificate_info'] = $certInfo;

            // ZATCA-specific validations
            $zatcaValidation = $this->validateZatcaCompliance($certInfo);
            $result['validation_checks']['zatca_compliance'] = $zatcaValidation;

            if (!empty($zatcaValidation['errors'])) {
                $result['errors'] = array_merge($result['errors'], $zatcaValidation['errors']);
            }

            if (!empty($zatcaValidation['warnings'])) {
                $result['warnings'] = array_merge($result['warnings'], $zatcaValidation['warnings']);
            }

            // If we have a valid certificate with only warnings (no errors), consider it successful
            if (empty($result['errors'])) {
                $result['success'] = true;
                $result['cleaned_certificate'] = $this->extractCertificateContent($cleanedCert);
            }

            Log::info('ZATCA certificate validation completed', [
                'success' => $result['success'],
                'errors_count' => count($result['errors']),
                'warnings_count' => count($result['warnings'])
            ]);

            return $result;

        } catch (Exception $e) {
            Log::error('Exception during ZATCA certificate validation', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => 'Exception during validation: ' . $e->getMessage(),
                'cleaned_certificate' => null,
                'certificate_info' => null,
                'validation_checks' => [],
                'errors' => ['Exception: ' . $e->getMessage()],
                'warnings' => []
            ];
        }
    }

    /**
     * Clean and normalize certificate format
     *
     * @param string $certificate Raw certificate content
     * @return string Cleaned certificate in proper PEM format
     */
    private function cleanCertificateFormat(string $certificate): string
    {
        try {
            // Remove any extra whitespace and normalize line endings
            $certificate = trim($certificate);
            $certificate = str_replace(["\r\n", "\r"], "\n", $certificate);

            // If certificate doesn't have PEM headers, try to add them
            if (!str_contains($certificate, '-----BEGIN CERTIFICATE-----')) {
                // Remove any existing headers/footers that might be malformed
                $certificate = preg_replace('/-----[A-Z\s]+-----/', '', $certificate);
                
                // Remove all whitespace and line breaks to get pure base64
                $base64Content = preg_replace('/\s+/', '', $certificate);
                
                // Validate base64 content
                if (!$this->isValidBase64($base64Content)) {
                    Log::warning('Certificate does not appear to contain valid base64 content');
                    return '';
                }

                // Reformat with proper line breaks (64 characters per line)
                $formattedBase64 = chunk_split($base64Content, 64, "\n");
                $formattedBase64 = trim($formattedBase64);

                // Add proper PEM headers
                $certificate = "-----BEGIN CERTIFICATE-----\n" . $formattedBase64 . "\n-----END CERTIFICATE-----";
            } else {
                // Certificate already has headers, just clean up formatting
                $lines = explode("\n", $certificate);
                $cleanedLines = [];
                $inCertificate = false;

                foreach ($lines as $line) {
                    $line = trim($line);
                    
                    if (str_contains($line, '-----BEGIN CERTIFICATE-----')) {
                        $inCertificate = true;
                        $cleanedLines[] = '-----BEGIN CERTIFICATE-----';
                    } elseif (str_contains($line, '-----END CERTIFICATE-----')) {
                        $inCertificate = false;
                        $cleanedLines[] = '-----END CERTIFICATE-----';
                        break; // Stop after first certificate
                    } elseif ($inCertificate && !empty($line)) {
                        // Ensure line contains only valid base64 characters
                        if (preg_match('/^[A-Za-z0-9+\/=]+$/', $line)) {
                            $cleanedLines[] = $line;
                        }
                    }
                }

                $certificate = implode("\n", $cleanedLines);
            }

            return $certificate;

        } catch (Exception $e) {
            Log::error('Error cleaning certificate format', [
                'error' => $e->getMessage()
            ]);
            return '';
        }
    }

    /**
     * Validate certificate using OpenSSL
     *
     * @param string $certificate Cleaned certificate
     * @return array Validation result
     */
    private function validateWithOpenSSL(string $certificate): array
    {
        try {
            // Read certificate with OpenSSL
            $certResource = openssl_x509_read($certificate);
            if ($certResource === false) {
                return [
                    'valid' => false,
                    'error' => 'OpenSSL failed to read certificate: ' . $this->getLastOpenSSLError(),
                    'certificate_info' => null
                ];
            }

            // Parse certificate information
            $certInfo = openssl_x509_parse($certResource);
            if ($certInfo === false) {
                return [
                    'valid' => false,
                    'error' => 'Failed to parse certificate information',
                    'certificate_info' => null
                ];
            }

            // Check if certificate is expired
            $now = time();
            $isExpired = $now > $certInfo['validTo_time_t'];
            $isNotYetValid = $now < $certInfo['validFrom_time_t'];

            return [
                'valid' => true,
                'certificate_info' => $certInfo,
                'is_expired' => $isExpired,
                'is_not_yet_valid' => $isNotYetValid,
                'days_until_expiry' => round(($certInfo['validTo_time_t'] - $now) / 86400),
                'valid_from' => $certInfo['validFrom_time_t'],
                'valid_to' => $certInfo['validTo_time_t']
            ];

        } catch (Exception $e) {
            return [
                'valid' => false,
                'error' => 'Exception during OpenSSL validation: ' . $e->getMessage(),
                'certificate_info' => null
            ];
        }
    }

    /**
     * Validate ZATCA compliance requirements
     *
     * @param array $certInfo Certificate information from OpenSSL
     * @return array ZATCA validation result
     */
    private function validateZatcaCompliance(array $certInfo): array
    {
        $result = [
            'zatca_ready' => true,
            'errors' => [],
            'warnings' => [],
            'issues' => [],
            'recommendations' => []
        ];

        try {
            // Check certificate purpose and key usage
            if (isset($certInfo['purposes'])) {
                $hasDigitalSignature = false;
                foreach ($certInfo['purposes'] as $purpose) {
                    if ($purpose[2] && (str_contains(strtolower($purpose[0]), 'sign') || str_contains(strtolower($purpose[0]), 'ssl'))) {
                        $hasDigitalSignature = true;
                        break;
                    }
                }

                if (!$hasDigitalSignature) {
                    $result['warnings'][] = 'Certificate may not be suitable for digital signing';
                    $result['issues'][] = 'No digital signature purpose found';
                }
            }

            // Check subject information
            if (empty($certInfo['subject']['CN'])) {
                $result['errors'][] = 'Certificate missing Common Name (CN) in subject';
                $result['zatca_ready'] = false;
            }

            // Check if certificate is expired
            $now = time();
            if ($now > $certInfo['validTo_time_t']) {
                $result['errors'][] = 'Certificate has expired';
                $result['zatca_ready'] = false;
            }

            if ($now < $certInfo['validFrom_time_t']) {
                $result['errors'][] = 'Certificate is not yet valid';
                $result['zatca_ready'] = false;
            }

            // Check certificate validity period
            $daysUntilExpiry = round(($certInfo['validTo_time_t'] - $now) / 86400);
            if ($daysUntilExpiry < 30) {
                $result['warnings'][] = "Certificate expires in {$daysUntilExpiry} days";
                $result['recommendations'][] = 'Consider renewing certificate soon';
            }

            // Check key strength
            if (isset($certInfo['extensions']['keyUsage'])) {
                $keyUsage = $certInfo['extensions']['keyUsage'];
                if (!str_contains($keyUsage, 'Digital Signature')) {
                    $result['warnings'][] = 'Certificate may not have Digital Signature key usage';
                    $result['issues'][] = 'Missing Digital Signature in key usage';
                }
            }

            // Additional ZATCA-specific checks can be added here
            // For example: specific issuer requirements, extended key usage, etc.

            Log::info('ZATCA compliance validation completed', [
                'zatca_ready' => $result['zatca_ready'],
                'errors_count' => count($result['errors']),
                'warnings_count' => count($result['warnings'])
            ]);

        } catch (Exception $e) {
            $result['errors'][] = 'Exception during ZATCA compliance check: ' . $e->getMessage();
            $result['zatca_ready'] = false;
        }

        return $result;
    }

    /**
     * Extract clean certificate content without headers
     *
     * @param string $certificate Certificate with headers
     * @return string Certificate content without headers
     */
    private function extractCertificateContent(string $certificate): string
    {
        // Remove PEM headers and footers, and all whitespace
        $content = str_replace([
            '-----BEGIN CERTIFICATE-----',
            '-----END CERTIFICATE-----',
            "\r", "\n", " ", "\t"
        ], '', $certificate);

        return $content;
    }

    /**
     * Validate certificate validity period
     *
     * @param array $certInfo Certificate information
     * @return array Validity check result
     */
    private function validateCertificateValidity(array $certInfo): array
    {
        $now = time();
        $validFrom = $certInfo['validFrom_time_t'];
        $validTo = $certInfo['validTo_time_t'];

        return [
            'is_valid_period' => $now >= $validFrom && $now <= $validTo,
            'is_expired' => $now > $validTo,
            'is_not_yet_valid' => $now < $validFrom,
            'days_until_expiry' => round(($validTo - $now) / 86400),
            'valid_from_date' => date('Y-m-d H:i:s', $validFrom),
            'valid_to_date' => date('Y-m-d H:i:s', $validTo)
        ];
    }

    /**
     * Validate certificate purpose and key usage
     *
     * @param array $certInfo Certificate information
     * @return array Purpose validation result
     */
    private function validateCertificatePurpose(array $certInfo): array
    {
        $result = [
            'suitable_for_signing' => false,
            'purposes' => [],
            'key_usage' => [],
            'extended_key_usage' => []
        ];

        if (isset($certInfo['purposes'])) {
            foreach ($certInfo['purposes'] as $purpose) {
                $result['purposes'][] = [
                    'name' => $purpose[0],
                    'ca_allowed' => $purpose[1],
                    'general_allowed' => $purpose[2]
                ];

                if ($purpose[2] && str_contains(strtolower($purpose[0]), 'sign')) {
                    $result['suitable_for_signing'] = true;
                }
            }
        }

        if (isset($certInfo['extensions']['keyUsage'])) {
            $result['key_usage'] = explode(', ', $certInfo['extensions']['keyUsage']);
            if (in_array('Digital Signature', $result['key_usage'])) {
                $result['suitable_for_signing'] = true;
            }
        }

        if (isset($certInfo['extensions']['extendedKeyUsage'])) {
            $result['extended_key_usage'] = explode(', ', $certInfo['extensions']['extendedKeyUsage']);
        }

        return $result;
    }

    /**
     * Get human-readable key type string
     *
     * @param int $keyType OpenSSL key type constant
     * @return string Key type name
     */
    private function getKeyTypeString(int $keyType): string
    {
        switch ($keyType) {
            case OPENSSL_KEYTYPE_RSA:
                return 'RSA';
            case OPENSSL_KEYTYPE_DSA:
                return 'DSA';
            case OPENSSL_KEYTYPE_DH:
                return 'DH';
            case OPENSSL_KEYTYPE_EC:
                return 'EC';
            default:
                return 'Unknown';
        }
    }

    /**
     * Check if string is valid base64
     *
     * @param string $string String to check
     * @return bool Whether string is valid base64
     */
    private function isValidBase64(string $string): bool
    {
        // Base64 can only contain A-Z, a-z, 0-9, +, /, and = (for padding)
        if (!preg_match('/^[A-Za-z0-9+\/]*={0,2}$/', $string)) {
            return false;
        }

        // Try to decode and re-encode to check validity
        $decoded = base64_decode($string, true);
        if ($decoded === false) {
            return false;
        }

        // Re-encode and compare (removing any whitespace)
        $reencoded = base64_encode($decoded);
        return $reencoded === $string;
    }

    /**
     * Get OpenSSL errors as array
     *
     * @return array OpenSSL error messages
     */
    private function getOpenSSLErrors(): array
    {
        $errors = [];
        while (($error = openssl_error_string()) !== false) {
            $errors[] = $error;
        }
        return $errors;
    }

    /**
     * Get last OpenSSL error as string
     *
     * @return string Last OpenSSL error message
     */
    private function getLastOpenSSLError(): string
    {
        $errors = $this->getOpenSSLErrors();
        return !empty($errors) ? end($errors) : 'Unknown OpenSSL error';
    }
}