<?php

namespace App\Services;

use App\Models\Invoice;
use App\Models\CashTransaction;
use App\Models\Account;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class InvoiceCashFlowService
{
    /**
     * تحليل التدفق النقدي للفواتير في فترة محددة
     */
    public function analyzeCashFlow(string $startDate, string $endDate, array $filters = []): array
    {
        try {
            $invoicesQuery = Invoice::with(['customer', 'cashTransactions'])
                ->whereBetween('issue_date', [$startDate, $endDate]);

            // تطبيق الفلاتر
            if (!empty($filters['payment_method'])) {
                $invoicesQuery->where('payment_method', $filters['payment_method']);
            }

            if (!empty($filters['customer_id'])) {
                $invoicesQuery->where('customer_id', $filters['customer_id']);
            }

            if (!empty($filters['status'])) {
                $invoicesQuery->where('status', $filters['status']);
            }

            $invoices = $invoicesQuery->get();

            return [
                'summary' => $this->generateSummary($invoices),
                'daily_breakdown' => $this->getDailyBreakdown($invoices),
                'payment_methods' => $this->getPaymentMethodsBreakdown($invoices),
                'top_customers' => $this->getTopCashCustomers($invoices),
                'aging_analysis' => $this->getAgingAnalysis($invoices),
            ];
        } catch (\Exception $e) {
            \Log::error('Cash flow analysis failed: ' . $e->getMessage());
            throw new \Exception('فشل في تحليل التدفق النقدي: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء ملخص التدفق النقدي
     */
    private function generateSummary(Collection $invoices): array
    {
        return [
            'total_invoices' => $invoices->count(),
            'total_amount' => $invoices->sum('total_amount'),
            'total_paid' => $invoices->sum('paid_amount'),
            'total_remaining' => $invoices->sum('remaining_amount'),
            'cash_received' => $invoices->where('payment_method', 'cash')->sum('paid_amount'),
            'bank_transfers' => $invoices->where('payment_method', 'bank_transfer')->sum('paid_amount'),
            'card_payments' => $invoices->where('payment_method', 'credit_card')->sum('paid_amount'),
            'check_payments' => $invoices->where('payment_method', 'check')->sum('paid_amount'),
            'payment_percentage' => $invoices->sum('total_amount') > 0 
                ? round(($invoices->sum('paid_amount') / $invoices->sum('total_amount')) * 100, 2) 
                : 0,
        ];
    }

    /**
     * تحليل التدفق النقدي اليومي
     */
    private function getDailyBreakdown(Collection $invoices): Collection
    {
        return $invoices->groupBy(function($invoice) {
            return $invoice->issue_date->format('Y-m-d');
        })->map(function($dailyInvoices, $date) {
            return [
                'date' => $date,
                'invoices_count' => $dailyInvoices->count(),
                'total_amount' => $dailyInvoices->sum('total_amount'),
                'cash_received' => $dailyInvoices->where('payment_method', 'cash')->sum('paid_amount'),
                'other_received' => $dailyInvoices->whereNotIn('payment_method', ['cash', null])->sum('paid_amount'),
                'total_received' => $dailyInvoices->sum('paid_amount'),
                'pending_amount' => $dailyInvoices->sum('remaining_amount'),
            ];
        })->sortBy('date');
    }

    /**
     * تحليل طرق الدفع
     */
    private function getPaymentMethodsBreakdown(Collection $invoices): array
    {
        $totalReceived = $invoices->sum('paid_amount');
        $paymentMethods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'credit_card' => 'بطاقة ائتمان',
            'check' => 'شيك',
            'installment' => 'تقسيط'
        ];

        $breakdown = [];
        foreach ($paymentMethods as $method => $name) {
            $amount = $invoices->where('payment_method', $method)->sum('paid_amount');
            $breakdown[$method] = [
                'name' => $name,
                'amount' => $amount,
                'count' => $invoices->where('payment_method', $method)->count(),
                'percentage' => $totalReceived > 0 ? round(($amount / $totalReceived) * 100, 2) : 0,
            ];
        }

        return $breakdown;
    }

    /**
     * أفضل العملاء من ناحية المدفوعات النقدية
     */
    private function getTopCashCustomers(Collection $invoices): Collection
    {
        return $invoices->where('payment_method', 'cash')
            ->groupBy('customer_id')
            ->map(function($customerInvoices) {
                $customer = $customerInvoices->first()->customer;
                return [
                    'customer' => $customer,
                    'invoices_count' => $customerInvoices->count(),
                    'total_paid' => $customerInvoices->sum('paid_amount'),
                    'avg_payment' => $customerInvoices->avg('paid_amount'),
                ];
            })
            ->sortByDesc('total_paid')
            ->take(10);
    }

    /**
     * تحليل أعمار الديون
     */
    private function getAgingAnalysis(Collection $invoices): array
    {
        $unpaidInvoices = $invoices->where('payment_status', '!=', 'paid');
        $now = Carbon::now();

        $aging = [
            'current' => 0,      // 0-30 days
            'aging_30' => 0,     // 31-60 days
            'aging_60' => 0,     // 61-90 days
            'aging_90_plus' => 0 // 90+ days
        ];

        foreach ($unpaidInvoices as $invoice) {
            $daysPastDue = $now->diffInDays($invoice->issue_date);
            $remainingAmount = $invoice->remaining_amount;

            if ($daysPastDue <= 30) {
                $aging['current'] += $remainingAmount;
            } elseif ($daysPastDue <= 60) {
                $aging['aging_30'] += $remainingAmount;
            } elseif ($daysPastDue <= 90) {
                $aging['aging_60'] += $remainingAmount;
            } else {
                $aging['aging_90_plus'] += $remainingAmount;
            }
        }

        return $aging;
    }

    /**
     * ربط دفعة فاتورة بحركة نقدية
     */
    public function linkInvoicePaymentToCashTransaction(Invoice $invoice, array $paymentData): CashTransaction
    {
        try {
            DB::beginTransaction();

            // التحقق من صحة البيانات
            $this->validatePaymentData($invoice, $paymentData);

            // إنشاء الحركة النقدية
            $cashTransaction = CashTransaction::create([
                'transaction_number' => CashTransaction::generateTransactionNumber('receipt'),
                'transaction_date' => now(),
                'cash_account_id' => $paymentData['cash_account_id'],
                'type' => 'receipt',
                'amount' => $paymentData['amount'],
                'reference' => $paymentData['reference'] ?? $invoice->invoice_number,
                'description' => 'استلام دفعة للفاتورة ' . $invoice->invoice_number,
                'contra_account_id' => $this->getCustomerReceivableAccount($invoice->customer_id),
                'notes' => $paymentData['notes'] ?? null,
                'invoice_id' => $invoice->id,
                'received_from' => $invoice->customer ? $invoice->customer->name : null,
                'payment_method' => $paymentData['payment_method'],
                'created_by' => auth()->id(),
            ]);

            // إنشاء القيد المحاسبي
            $cashTransaction->createJournalEntry();

            // تحديث الفاتورة
            $invoice->addPayment(
                $paymentData['amount'], 
                $paymentData['payment_method'], 
                $paymentData['notes'] ?? null
            );

            DB::commit();

            return $cashTransaction;
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('Failed to link invoice payment to cash transaction: ' . $e->getMessage());
            throw new \Exception('فشل في ربط دفعة الفاتورة بحركة الصندوق: ' . $e->getMessage());
        }
    }

    /**
     * التحقق من صحة بيانات الدفعة
     */
    private function validatePaymentData(Invoice $invoice, array $paymentData): void
    {
        if ($paymentData['amount'] <= 0) {
            throw new \Exception('المبلغ يجب أن يكون أكبر من صفر');
        }

        if ($paymentData['amount'] > $invoice->remaining_amount) {
            throw new \Exception('المبلغ يتجاوز المبلغ المستحق للفاتورة');
        }

        if (!Account::find($paymentData['cash_account_id'])) {
            throw new \Exception('حساب النقدية غير موجود');
        }

        $validPaymentMethods = ['cash', 'bank_transfer', 'check', 'credit_card'];
        if (!in_array($paymentData['payment_method'], $validPaymentMethods)) {
            throw new \Exception('طريقة الدفع غير صحيحة');
        }
    }

    /**
     * الحصول على حساب المدينين للعميل
     */
    private function getCustomerReceivableAccount($customerId): int
    {
        $receivableAccount = Account::where('account_type', 'receivable')
            ->where('name', 'like', '%مدينين%')
            ->first();

        if (!$receivableAccount) {
            $receivableAccount = Account::create([
                'code' => '1210',
                'name' => 'حسابات مدينة - عملاء',
                'account_type' => 'receivable',
                'parent_id' => null,
                'is_active' => true
            ]);
        }

        return $receivableAccount->id;
    }

    /**
     * تصدير تقرير التدفق النقدي
     */
    public function exportCashFlowReport(array $data): array
    {
        $exportData = [
            ['تقرير حركة الصندوق للفواتير - ' . now()->format('Y-m-d H:i:s')],
            [''],
            ['الملخص العام'],
            ['إجمالي الفواتير', $data['summary']['total_invoices']],
            ['إجمالي المبلغ', number_format($data['summary']['total_amount'], 2) . ' ر.س'],
            ['إجمالي المحصل', number_format($data['summary']['total_paid'], 2) . ' ر.س'],
            ['إجمالي المتبقي', number_format($data['summary']['total_remaining'], 2) . ' ر.س'],
            ['نسبة التحصيل', $data['summary']['payment_percentage'] . '%'],
            [''],
            ['توزيع طرق الدفع'],
            ['الطريقة', 'المبلغ', 'النسبة المئوية'],
        ];

        foreach ($data['payment_methods'] as $method => $details) {
            $exportData[] = [
                $details['name'],
                number_format($details['amount'], 2) . ' ر.س',
                $details['percentage'] . '%'
            ];
        }

        return $exportData;
    }

    /**
     * الحصول على احصائيات التدفق النقدي للوحة التحكم
     */
    public function getDashboardStats(): array
    {
        $today = Carbon::today();
        $thisWeek = Carbon::now()->startOfWeek();
        $thisMonth = Carbon::now()->startOfMonth();

        return [
            'today' => $this->getStatsForPeriod($today, $today),
            'this_week' => $this->getStatsForPeriod($thisWeek, Carbon::now()),
            'this_month' => $this->getStatsForPeriod($thisMonth, Carbon::now()),
            'cash_accounts_balance' => $this->getCashAccountsBalance(),
        ];
    }

    /**
     * الحصول على احصائيات فترة محددة
     */
    private function getStatsForPeriod(Carbon $start, Carbon $end): array
    {
        $invoices = Invoice::whereBetween('issue_date', [$start, $end])->get();

        return [
            'invoices_count' => $invoices->count(),
            'total_amount' => $invoices->sum('total_amount'),
            'cash_received' => $invoices->where('payment_method', 'cash')->sum('paid_amount'),
            'total_received' => $invoices->sum('paid_amount'),
        ];
    }

    /**
     * الحصول على أرصدة الحسابات النقدية
     */
    private function getCashAccountsBalance(): array
    {
        $cashAccounts = Account::cashAccounts()->active()->get();
        $balances = [];

        foreach ($cashAccounts as $account) {
            $balances[] = [
                'account' => $account,
                'balance' => $account->current_balance ?? 0,
            ];
        }

        return $balances;
    }
}