<?php

namespace App\Services;

use App\Models\ZatcaSettings;
use Exception;

class CertificateHelper
{
    /**
     * الحصول على محتوى الشهادة من قاعدة البيانات
     */
    public static function getCertificateContent($companyId): ?string
    {
        $settings = ZatcaSettings::where('company_id', $companyId)->first();
        
        if (!$settings || !$settings->certificate) {
            return null;
        }
        
        return $settings->certificate;
    }
    
    /**
     * الحصول على محتوى المفتاح الخاص من قاعدة البيانات
     */
    public static function getPrivateKeyContent($companyId): ?string
    {
        $settings = ZatcaSettings::where('company_id', $companyId)->first();
        
        if (!$settings || !$settings->private_key) {
            return null;
        }
        
        return $settings->private_key;
    }
    
    /**
     * الحصول على محتوى المفتاح العام من قاعدة البيانات
     */
    public static function getPublicKeyContent($companyId): ?string
    {
        $settings = ZatcaSettings::where('company_id', $companyId)->first();
        
        if (!$settings || !$settings->public_key) {
            return null;
        }
        
        return $settings->public_key;
    }
    
    /**
     * قراءة الشهادة من ملف
     */
    public static function readCertificateFromFile(string $filePath): ?string
    {
        if (!file_exists($filePath)) {
            throw new Exception("ملف الشهادة غير موجود: {$filePath}");
        }
        
        $content = file_get_contents($filePath);
        
        if ($content === false) {
            throw new Exception("فشل في قراءة ملف الشهادة: {$filePath}");
        }
        
        return $content;
    }
    
    /**
     * قراءة المفتاح الخاص من ملف
     */
    public static function readPrivateKeyFromFile(string $filePath): ?string
    {
        if (!file_exists($filePath)) {
            throw new Exception("ملف المفتاح الخاص غير موجود: {$filePath}");
        }
        
        $content = file_get_contents($filePath);
        
        if ($content === false) {
            throw new Exception("فشل في قراءة ملف المفتاح الخاص: {$filePath}");
        }
        
        return $content;
    }
    
    /**
     * تحليل معلومات الشهادة
     */
    public static function parseCertificateInfo(string $certificateContent): array
    {
        $cert = openssl_x509_read($certificateContent);
        
        if (!$cert) {
            throw new Exception("فشل في تحليل الشهادة");
        }
        
        $certInfo = openssl_x509_parse($cert);
        
        return [
            'subject' => $certInfo['subject'] ?? [],
            'issuer' => $certInfo['issuer'] ?? [],
            'serial_number' => $certInfo['serialNumber'] ?? null,
            'valid_from' => date('Y-m-d H:i:s', $certInfo['validFrom_time_t'] ?? 0),
            'valid_to' => date('Y-m-d H:i:s', $certInfo['validTo_time_t'] ?? 0),
            'fingerprint' => openssl_x509_fingerprint($cert),
        ];
    }
    
    /**
     * التحقق من صحة المفتاح الخاص
     */
    public static function validatePrivateKey(string $privateKeyContent, string $passphrase = null): bool
    {
        $privateKey = openssl_pkey_get_private($privateKeyContent, $passphrase);
        
        if (!$privateKey) {
            return false;
        }
        
        return true;
    }
    
    /**
     * استخراج المفتاح العام من الشهادة
     */
    public static function extractPublicKeyFromCertificate(string $certificateContent): ?string
    {
        $cert = openssl_x509_read($certificateContent);
        
        if (!$cert) {
            throw new Exception("فشل في قراءة الشهادة");
        }
        
        $publicKey = openssl_pkey_get_public($cert);
        
        if (!$publicKey) {
            throw new Exception("فشل في استخراج المفتاح العام");
        }
        
        $keyDetails = openssl_pkey_get_details($publicKey);
        
        return $keyDetails['key'] ?? null;
    }
    
    /**
     * تحويل الشهادة إلى تنسيق Base64
     */
    public static function certificateToBase64(string $certificateContent): string
    {
        // إزالة headers و footers
        $cleanCert = str_replace([
            '-----BEGIN CERTIFICATE-----',
            '-----END CERTIFICATE-----',
            "\r",
            "\n",
            ' '
        ], '', $certificateContent);
        
        return $cleanCert;
    }
    
    /**
     * تحويل المفتاح الخاص إلى تنسيق Base64
     */
    public static function privateKeyToBase64(string $privateKeyContent): string
    {
        // إزالة headers و footers
        $cleanKey = str_replace([
            '-----BEGIN PRIVATE KEY-----',
            '-----END PRIVATE KEY-----',
            '-----BEGIN RSA PRIVATE KEY-----',
            '-----END RSA PRIVATE KEY-----',
            "\r",
            "\n",
            ' '
        ], '', $privateKeyContent);
        
        return $cleanKey;
    }
    
    /**
     * حفظ الشهادة والمفتاح في قاعدة البيانات
     */
    public static function saveCertificateAndKey($companyId, string $certificate, string $privateKey, string $publicKey = null): bool
    {
        try {
            $settings = ZatcaSettings::firstOrCreate(['company_id' => $companyId]);
            
            $settings->certificate = $certificate;
            $settings->private_key = $privateKey;
            
            if ($publicKey) {
                $settings->public_key = $publicKey;
            } else {
                // استخراج المفتاح العام من الشهادة
                $settings->public_key = self::extractPublicKeyFromCertificate($certificate);
            }
            
            // تحليل معلومات الشهادة لاستخراج تاريخ الانتهاء
            $certInfo = self::parseCertificateInfo($certificate);
            $settings->certificate_expires_at = $certInfo['valid_to'];
            $settings->certificate_serial = $certInfo['serial_number'];
            
            return $settings->save();
            
        } catch (Exception $e) {
            throw new Exception("فشل في حفظ الشهادة والمفتاح: " . $e->getMessage());
        }
    }
    
    /**
     * تصدير الشهادة والمفتاح إلى ملفات
     */
    public static function exportToFiles($companyId, string $certificatePath, string $privateKeyPath): bool
    {
        $certificate = self::getCertificateContent($companyId);
        $privateKey = self::getPrivateKeyContent($companyId);
        
        if (!$certificate || !$privateKey) {
            throw new Exception("الشهادة أو المفتاح الخاص غير موجود في قاعدة البيانات");
        }
        
        // حفظ الشهادة
        if (file_put_contents($certificatePath, $certificate) === false) {
            throw new Exception("فشل في حفظ ملف الشهادة");
        }
        
        // حفظ المفتاح الخاص
        if (file_put_contents($privateKeyPath, $privateKey) === false) {
            throw new Exception("فشل في حفظ ملف المفتاح الخاص");
        }
        
        return true;
    }
}