<?php

namespace App\Services;

use App\Models\Account;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use App\Models\CashTransaction;
use App\Models\CostCenter;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AccountingService
{
    /**
     * إنشاء قيد محاسبي تلقائياً من معاملة نقدية
     */
    public function createJournalEntryFromCashTransaction(CashTransaction $transaction)
    {
        DB::beginTransaction();
        
        try {
            // إنشاء القيد
            $journalEntry = JournalEntry::create([
                'reference' => $this->generateJournalEntryReference(),
                'date' => $transaction->date,
                'description' => $transaction->description . ' (من معاملة نقدية: ' . $transaction->reference . ')',
                'total_amount' => $transaction->amount,
                'is_approved' => false,
                'created_by' => auth()->id(),
            ]);

            // إضافة خطوط القيد
            if ($transaction->type === 'receipt') {
                // استلام نقدي: مدين الحساب النقدي، دائن الحساب المقابل
                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $transaction->account_id,
                    'debit_amount' => $transaction->amount,
                    'credit_amount' => 0,
                    'description' => 'استلام نقدي - ' . $transaction->account->name,
                    'cost_center_id' => $transaction->cost_center_id,
                ]);

                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $transaction->contra_account_id,
                    'debit_amount' => 0,
                    'credit_amount' => $transaction->amount,
                    'description' => 'مقابل استلام نقدي - ' . $transaction->contraAccount->name,
                    'cost_center_id' => $transaction->cost_center_id,
                ]);
            } else {
                // دفع نقدي: دائن الحساب النقدي، مدين الحساب المقابل
                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $transaction->contra_account_id,
                    'debit_amount' => $transaction->amount,
                    'credit_amount' => 0,
                    'description' => 'دفع نقدي - ' . $transaction->contraAccount->name,
                    'cost_center_id' => $transaction->cost_center_id,
                ]);

                JournalEntryLine::create([
                    'journal_entry_id' => $journalEntry->id,
                    'account_id' => $transaction->account_id,
                    'debit_amount' => 0,
                    'credit_amount' => $transaction->amount,
                    'description' => 'مقابل دفع نقدي - ' . $transaction->account->name,
                    'cost_center_id' => $transaction->cost_center_id,
                ]);
            }

            // ربط المعاملة النقدية بالقيد
            $transaction->update(['journal_entry_id' => $journalEntry->id]);

            DB::commit();
            return $journalEntry;

        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    /**
     * حساب أرصدة الحسابات
     */
    public function calculateAccountBalances($asOfDate = null)
    {
        $asOfDate = $asOfDate ? Carbon::parse($asOfDate) : now();

        $accounts = Account::all();
        
        foreach ($accounts as $account) {
            $balance = $this->getAccountBalance($account->id, $asOfDate);
            $account->update(['current_balance' => $balance]);
        }

        return true;
    }

    /**
     * حساب رصيد حساب محدد
     */
    public function getAccountBalance($accountId, $asOfDate = null)
    {
        $account = Account::find($accountId);
        if (!$account) {
            return 0;
        }

        $query = JournalEntryLine::where('account_id', $accountId)
            ->whereHas('journalEntry', function ($query) use ($asOfDate) {
                $query->where('is_approved', true);
                if ($asOfDate) {
                    $query->where('date', '<=', $asOfDate);
                }
            });

        $totalDebits = (clone $query)->sum('debit_amount');
        $totalCredits = (clone $query)->sum('credit_amount');

        // حساب الرصيد بناءً على نوع الحساب
        $balance = $account->opening_balance + $totalDebits - $totalCredits;

        return $balance;
    }

    /**
     * إنشاء ميزان المراجعة
     */
    public function generateTrialBalance($asOfDate = null)
    {
        $asOfDate = $asOfDate ? Carbon::parse($asOfDate) : now();
        
        $accounts = Account::where('is_active', true)
            ->orderBy('code')
            ->get();

        $trialBalance = [];
        $totalDebits = 0;
        $totalCredits = 0;

        foreach ($accounts as $account) {
            $balance = $this->getAccountBalance($account->id, $asOfDate);
            
            if ($balance != 0 || $account->is_control_account) {
                $debitBalance = $balance > 0 ? $balance : 0;
                $creditBalance = $balance < 0 ? abs($balance) : 0;

                $trialBalance[] = [
                    'account' => $account,
                    'debit_balance' => $debitBalance,
                    'credit_balance' => $creditBalance,
                ];

                $totalDebits += $debitBalance;
                $totalCredits += $creditBalance;
            }
        }

        return [
            'accounts' => $trialBalance,
            'total_debits' => $totalDebits,
            'total_credits' => $totalCredits,
            'is_balanced' => abs($totalDebits - $totalCredits) < 0.01, // تسامح في الفروقات الطفيفة
            'as_of_date' => $asOfDate,
        ];
    }

    /**
     * إنشاء الميزانية العمومية
     */
    public function generateBalanceSheet($asOfDate = null)
    {
        $asOfDate = $asOfDate ? Carbon::parse($asOfDate) : now();

        // الأصول
        $assets = $this->getAccountsByType('asset', $asOfDate);
        $currentAssets = $this->filterAccountsBySubtype($assets, ['current']);
        $fixedAssets = $this->filterAccountsBySubtype($assets, ['fixed']);
        
        // الخصوم
        $liabilities = $this->getAccountsByType('liability', $asOfDate);
        $currentLiabilities = $this->filterAccountsBySubtype($liabilities, ['current']);
        $longTermLiabilities = $this->filterAccountsBySubtype($liabilities, ['long_term']);
        
        // حقوق الملكية
        $equity = $this->getAccountsByType('equity', $asOfDate);
        
        // حساب صافي الدخل للفترة الحالية
        $retainedEarnings = $this->calculateRetainedEarnings($asOfDate);

        return [
            'current_assets' => $currentAssets,
            'fixed_assets' => $fixedAssets,
            'total_current_assets' => collect($currentAssets)->sum('balance'),
            'total_fixed_assets' => collect($fixedAssets)->sum('balance'),
            'total_assets' => collect($assets)->sum('balance'),
            
            'current_liabilities' => $currentLiabilities,
            'long_term_liabilities' => $longTermLiabilities,
            'total_current_liabilities' => abs(collect($currentLiabilities)->sum('balance')),
            'total_long_term_liabilities' => abs(collect($longTermLiabilities)->sum('balance')),
            'total_liabilities' => abs(collect($liabilities)->sum('balance')),
            
            'equity_accounts' => $equity,
            'retained_earnings' => $retainedEarnings,
            'total_equity' => abs(collect($equity)->sum('balance')) + abs($retainedEarnings),
            'total_liabilities_and_equity' => abs(collect($liabilities)->sum('balance')) + abs(collect($equity)->sum('balance')) + abs($retainedEarnings),
            
            'as_of_date' => $asOfDate,
        ];
    }

    /**
     * إنشاء قائمة الدخل
     */
    public function generateIncomeStatement($startDate, $endDate)
    {
        $startDate = Carbon::parse($startDate);
        $endDate = Carbon::parse($endDate);

        // الإيرادات
        $revenues = $this->getAccountActivityByType('revenue', $startDate, $endDate);
        $totalRevenue = abs(collect($revenues)->sum('amount'));

        // تكلفة البضاعة المباعة
        $cogs = $this->getCogsAccounts($startDate, $endDate);
        $totalCogs = collect($cogs)->sum('amount');

        // المصروفات
        $expenses = $this->getAccountActivityByType('expense', $startDate, $endDate);
        $totalExpenses = collect($expenses)->sum('amount');

        // حساب الأرباح
        $grossProfit = $totalRevenue - $totalCogs;
        $netIncome = $grossProfit - $totalExpenses;

        return [
            'revenue_accounts' => $revenues,
            'total_revenue' => $totalRevenue,
            
            'cogs_accounts' => $cogs,
            'total_cogs' => $totalCogs,
            'gross_profit' => $grossProfit,
            
            'expense_accounts' => $expenses,
            'total_expenses' => $totalExpenses,
            
            'net_income' => $netIncome,
            
            'start_date' => $startDate,
            'end_date' => $endDate,
        ];
    }

    /**
     * حساب الأرباح المحتجزة
     */
    private function calculateRetainedEarnings($asOfDate)
    {
        $yearStart = Carbon::parse($asOfDate)->startOfYear();
        $incomeStatement = $this->generateIncomeStatement($yearStart, $asOfDate);
        
        return $incomeStatement['net_income'];
    }

    /**
     * الحصول على الحسابات حسب النوع
     */
    private function getAccountsByType($type, $asOfDate)
    {
        $accounts = Account::where('type', $type)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        $result = [];
        foreach ($accounts as $account) {
            $balance = $this->getAccountBalance($account->id, $asOfDate);
            if ($balance != 0) {
                $result[] = [
                    'id' => $account->id,
                    'code' => $account->code,
                    'name' => $account->name,
                    'balance' => $balance,
                ];
            }
        }

        return $result;
    }

    /**
     * الحصول على نشاط الحسابات حسب النوع لفترة محددة
     */
    private function getAccountActivityByType($type, $startDate, $endDate)
    {
        $accounts = Account::where('type', $type)
            ->where('is_active', true)
            ->orderBy('code')
            ->get();

        $result = [];
        foreach ($accounts as $account) {
            $activity = $this->getAccountActivity($account->id, $startDate, $endDate);
            if ($activity != 0) {
                $result[] = [
                    'id' => $account->id,
                    'code' => $account->code,
                    'name' => $account->name,
                    'amount' => abs($activity),
                ];
            }
        }

        return $result;
    }

    /**
     * حساب نشاط حساب لفترة محددة
     */
    private function getAccountActivity($accountId, $startDate, $endDate)
    {
        $query = JournalEntryLine::where('account_id', $accountId)
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->where('is_approved', true)
                      ->whereBetween('date', [$startDate, $endDate]);
            });

        $totalDebits = (clone $query)->sum('debit_amount');
        $totalCredits = (clone $query)->sum('credit_amount');

        return $totalDebits - $totalCredits;
    }

    /**
     * الحصول على حسابات تكلفة البضاعة المباعة
     */
    private function getCogsAccounts($startDate, $endDate)
    {
        // البحث عن حسابات تكلفة البضاعة المباعة
        $cogsAccounts = Account::where(function ($query) {
            $query->where('name', 'like', '%تكلفة%')
                  ->orWhere('name', 'like', '%مباعة%')
                  ->orWhere('code', 'like', '5001%');
        })->where('type', 'expense')->get();

        $result = [];
        foreach ($cogsAccounts as $account) {
            $activity = $this->getAccountActivity($account->id, $startDate, $endDate);
            if ($activity > 0) {
                $result[] = [
                    'id' => $account->id,
                    'code' => $account->code,
                    'name' => $account->name,
                    'amount' => $activity,
                ];
            }
        }

        return $result;
    }

    /**
     * فلترة الحسابات حسب النوع الفرعي
     */
    private function filterAccountsBySubtype($accounts, $subtypes)
    {
        // يمكن تطوير هذا لاحقاً لدعم الأنواع الفرعية
        return $accounts;
    }

    /**
     * توليد رقم مرجع للقيد المحاسبي
     */
    private function generateJournalEntryReference()
    {
        $year = now()->year;
        $lastEntry = JournalEntry::whereYear('created_at', $year)
            ->orderBy('id', 'desc')
            ->first();

        $nextNumber = $lastEntry ? (intval(substr($lastEntry->reference, -4)) + 1) : 1;
        
        return 'JE' . $year . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * التحقق من توازن القيد المحاسبي
     */
    public function validateJournalEntryBalance($journalEntryId)
    {
        $totalDebits = JournalEntryLine::where('journal_entry_id', $journalEntryId)->sum('debit_amount');
        $totalCredits = JournalEntryLine::where('journal_entry_id', $journalEntryId)->sum('credit_amount');

        return abs($totalDebits - $totalCredits) < 0.01;
    }

    /**
     * اعتماد قيد محاسبي
     */
    public function approveJournalEntry($journalEntryId)
    {
        $journalEntry = JournalEntry::find($journalEntryId);
        
        if (!$journalEntry) {
            throw new \Exception('القيد المحاسبي غير موجود');
        }

        if ($journalEntry->is_approved) {
            throw new \Exception('القيد معتمد مسبقاً');
        }

        if (!$this->validateJournalEntryBalance($journalEntryId)) {
            throw new \Exception('القيد غير متوازن');
        }

        return $journalEntry->update([
            'is_approved' => true,
            'approved_by' => auth()->id(),
            'approved_at' => now(),
        ]);
    }
}