<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Blade;
use App\Services\ZatcaDeviceService;
use App\Models\ZatcaSettings;

class ZatcaServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        // تسجيل خدمة ZATCA Device
        $this->app->singleton(ZatcaDeviceService::class, function ($app) {
            return new ZatcaDeviceService();
        });

        // تسجيل ملف التكوين
        $this->mergeConfigFrom(
            __DIR__.'/../../config/zatca.php', 'zatca'
        );
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // نشر ملف التكوين
        $this->publishes([
            __DIR__.'/../../config/zatca.php' => config_path('zatca.php'),
        ], 'zatca-config');

        // إضافة متغيرات عامة للعروض
        View::composer('*', function ($view) {
            if (config('zatca.enabled', true)) {
                $view->with('zatcaEnabled', true);
            }
        });

        // تسجيل Blade Directives مخصصة لـ ZATCA
        $this->registerBladeDirectives();

        // تسجيل المؤشرات المخصصة
        $this->registerMacros();
    }

    /**
     * تسجيل Blade Directives مخصصة
     */
    protected function registerBladeDirectives(): void
    {
        // التحقق من تفعيل ZATCA
        Blade::if('zatcaEnabled', function () {
            return config('zatca.enabled', true);
        });

        // التحقق من البيئة
        Blade::if('zatcaSandbox', function () {
            return config('zatca.default_environment') === 'sandbox';
        });

        // عرض حالة اتصال ZATCA
        Blade::directive('zatcaStatus', function ($companyId) {
            return "<?php 
                \$zatcaSetting = \App\Models\ZatcaSettings::where('company_id', $companyId)->first();
                if (\$zatcaSetting) {
                    echo '<span class=\"badge bg-' . (\$zatcaSetting->connection_status === 'connected' ? 'success' : 'danger') . '\">';
                    echo \$zatcaSetting->connection_status === 'connected' ? 'متصل' : 'غير متصل';
                    echo '</span>';
                } else {
                    echo '<span class=\"badge bg-secondary\">غير مكون</span>';
                }
            ?>";
        });

        // عرض تاريخ انتهاء الشهادة
        Blade::directive('zatcaCertExpiry', function ($companyId) {
            return "<?php 
                \$zatcaSetting = \App\Models\ZatcaSettings::where('company_id', $companyId)->first();
                if (\$zatcaSetting && \$zatcaSetting->certificate_expires_at) {
                    \$daysToExpiry = now()->diffInDays(\$zatcaSetting->certificate_expires_at, false);
                    if (\$daysToExpiry <= 0) {
                        echo '<span class=\"text-danger\">منتهية الصلاحية</span>';
                    } elseif (\$daysToExpiry <= 30) {
                        echo '<span class=\"text-warning\">' . \$daysToExpiry . ' يوم متبقي</span>';
                    } else {
                        echo '<span class=\"text-success\">صالحة</span>';
                    }
                } else {
                    echo '<span class=\"text-muted\">غير محدد</span>';
                }
            ?>";
        });
    }

    /**
     * تسجيل Macros مخصصة
     */
    protected function registerMacros(): void
    {
        // إضافة macro للتحقق من حالة ZATCA
        \Illuminate\Database\Eloquent\Builder::macro('activeZatca', function () {
            return $this->where('is_active', true)
                        ->where('connection_status', 'connected');
        });

        // إضافة macro للشهادات المنتهية الصلاحية
        \Illuminate\Database\Eloquent\Builder::macro('expiredCertificates', function () {
            return $this->where('certificate_expires_at', '<', now());
        });

        // إضافة macro للشهادات التي ستنتهي قريباً
        \Illuminate\Database\Eloquent\Builder::macro('expiringSoon', function ($days = 30) {
            return $this->whereBetween('certificate_expires_at', [
                now(),
                now()->addDays($days)
            ]);
        });
    }
}