<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Blade;
use App\Services\AccountingService;
use App\Helpers\AccountingHelper;
use App\Http\Middleware\CheckAccountingPermissions;

class AccountingServiceProvider extends ServiceProvider
{
    /**
     * تسجيل أي خدمات التطبيق
     */
    public function register(): void
    {
        // تسجيل خدمة النظام المحاسبي
        $this->app->singleton(AccountingService::class, function ($app) {
            return new AccountingService();
        });

        // تسجيل المساعد المحاسبي
        $this->app->singleton('accounting.helper', function ($app) {
            return new AccountingHelper();
        });

        // ربط الواجهات بالتطبيقات
        $this->app->bind('AccountingServiceInterface', AccountingService::class);
    }

    /**
     * تشغيل أي خدمات التطبيق
     */
    public function boot(): void
    {
        // تحميل ملف التكوين
        $this->mergeConfigFrom(
            __DIR__.'/../../config/accounting.php', 'accounting'
        );

        // نشر ملف التكوين
        if ($this->app->runningInConsole()) {
            $this->publishes([
                __DIR__.'/../../config/accounting.php' => config_path('accounting.php'),
            ], 'accounting-config');
        }

        // تسجيل middleware المحاسبة
        $this->registerMiddleware();

        // تسجيل Blade directives
        $this->registerBladeDirectives();

        // تسجيل View composers
        $this->registerViewComposers();

        // تسجيل الماكرو
        $this->registerMacros();

        // تحميل المسارات المحاسبية
        $this->loadAccountingRoutes();
    }

    /**
     * تسجيل middleware المحاسبة
     */
    protected function registerMiddleware(): void
    {
        $router = $this->app['router'];
        
        $router->aliasMiddleware('accounting.permissions', CheckAccountingPermissions::class);
        
        // تطبيق middleware على مجموعة المسارات المحاسبية
        $router->middleware(['web', 'auth', 'accounting.permissions'])
               ->prefix('accounting')
               ->name('accounting.')
               ->group(function () {
                   // سيتم تحميل المسارات هنا
               });
    }

    /**
     * تسجيل Blade directives
     */
    protected function registerBladeDirectives(): void
    {
        // توجيه لتنسيق المبالغ
        Blade::directive('money', function ($expression) {
            return "<?php echo App\Helpers\AccountingHelper::formatMoney($expression); ?>";
        });

        // توجيه لعرض رمز الحساب
        Blade::directive('accountCode', function ($expression) {
            return "<?php echo App\Helpers\AccountingHelper::formatAccountCode($expression); ?>";
        });

        // توجيه للتحقق من الصلاحيات المحاسبية
        Blade::directive('canAccounting', function ($expression) {
            return "<?php if(App\Http\Middleware\CheckAccountingPermissions::hasAccountingPermission(auth()->user(), $expression)): ?>";
        });

        Blade::directive('endcanAccounting', function () {
            return "<?php endif; ?>";
        });

        // توجيه لعرض نوع الحساب
        Blade::directive('accountType', function ($expression) {
            $accountTypes = config('accounting.chart_of_accounts.account_types');
            return "<?php echo \$accountTypes[$expression]['name_ar'] ?? $expression; ?>";
        });

        // توجيه لعرض لون نوع الحساب
        Blade::directive('accountTypeColor', function ($expression) {
            return "<?php echo App\Helpers\AccountingHelper::getAccountTypeColor($expression); ?>";
        });

        // توجيه لعرض أيقونة نوع الحساب
        Blade::directive('accountTypeIcon', function ($expression) {
            return "<?php echo App\Helpers\AccountingHelper::getAccountTypeIcon($expression); ?>";
        });

        // توجيه لحساب عمر الدين
        Blade::directive('debtAge', function ($expression) {
            return "<?php echo App\Helpers\AccountingHelper::calculateAge($expression); ?>";
        });

        // توجيه لتصنيف عمر الدين
        Blade::directive('debtAgeClass', function ($expression) {
            return "<?php 
                \$classification = App\Helpers\AccountingHelper::classifyDebtAge($expression);
                echo \$classification['class'];
            ?>";
        });
    }

    /**
     * تسجيل View composers
     */
    protected function registerViewComposers(): void
    {
        // مشاركة البيانات المحاسبية المشتركة
        View::composer('accounting.*', function ($view) {
            $view->with([
                'accountTypes' => config('accounting.chart_of_accounts.account_types'),
                'currency' => config('accounting.general.currency'),
                'currencySymbol' => config('accounting.general.currency_symbol'),
                'companyName' => config('accounting.general.company_name'),
            ]);
        });

        // مشاركة قائمة الحسابات النشطة
        View::composer(['accounting.journal-entries.*', 'accounting.cash-transactions.*'], function ($view) {
            $accounts = \App\Models\Account::where('is_active', true)
                ->where('allow_manual_entries', true)
                ->orderBy('code')
                ->get();
            
            $view->with('accounts', $accounts);
        });

        // مشاركة مراكز التكلفة النشطة
        View::composer('accounting.*', function ($view) {
            $costCenters = \App\Models\CostCenter::where('is_active', true)
                ->orderBy('code')
                ->get();
            
            $view->with('costCenters', $costCenters);
        });

        // مشاركة إعدادات المحاسبة مع لوحة التحكم
        View::composer('dashboard', function ($view) {
            if (auth()->check()) {
                $accountingEnabled = config('accounting.general.enabled', true);
                $userPermissions = CheckAccountingPermissions::getUserPermissions(auth()->user());
                
                $view->with([
                    'accountingEnabled' => $accountingEnabled,
                    'userAccountingPermissions' => $userPermissions,
                ]);
            }
        });
    }

    /**
     * تسجيل الماكرو
     */
    protected function registerMacros(): void
    {
        // ماكرو للنماذج لتنسيق المبالغ
        \Illuminate\Database\Eloquent\Builder::macro('withFormattedBalance', function () {
            return $this->selectRaw('*, CONCAT(FORMAT(current_balance, 2), " ريال") as formatted_balance');
        });

        // ماكرو للحصول على الحسابات حسب النوع
        \Illuminate\Database\Eloquent\Builder::macro('ofType', function ($type) {
            return $this->where('type', $type);
        });

        // ماكرو للحصول على الحسابات النشطة
        \Illuminate\Database\Eloquent\Builder::macro('active', function () {
            return $this->where('is_active', true);
        });

        // ماكرو للحصول على الحسابات النقدية
        \Illuminate\Database\Eloquent\Builder::macro('cashAccounts', function () {
            return $this->where(function ($query) {
                $query->where('is_cash_account', true)
                      ->orWhere('is_bank_account', true);
            });
        });
    }

    /**
     * تحميل المسارات المحاسبية
     */
    protected function loadAccountingRoutes(): void
    {
        if ($this->app->routesAreCached()) {
            return;
        }

        // يمكن إضافة مسارات إضافية هنا إذا لزم الأمر
        Route::middleware(['web', 'auth'])
             ->prefix('api/accounting')
             ->name('api.accounting.')
             ->group(function () {
                 // مسارات API للنظام المحاسبي
                 Route::get('/accounts/search', function () {
                     $query = request('q');
                     $accounts = \App\Models\Account::where('is_active', true)
                         ->where(function ($q) use ($query) {
                             $q->where('name', 'like', "%{$query}%")
                               ->orWhere('code', 'like', "%{$query}%");
                         })
                         ->limit(20)
                         ->get(['id', 'code', 'name', 'type']);
                     
                     return response()->json($accounts);
                 })->name('accounts.search');

                 Route::get('/cost-centers/search', function () {
                     $query = request('q');
                     $costCenters = \App\Models\CostCenter::where('is_active', true)
                         ->where(function ($q) use ($query) {
                             $q->where('name', 'like', "%{$query}%")
                               ->orWhere('code', 'like', "%{$query}%");
                         })
                         ->limit(20)
                         ->get(['id', 'code', 'name']);
                     
                     return response()->json($costCenters);
                 })->name('cost-centers.search');
             });
    }

    /**
     * الحصول على الخدمات المقدمة
     */
    public function provides(): array
    {
        return [
            AccountingService::class,
            'accounting.helper',
        ];
    }
}