<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class DailyZatcaStatusNotification extends Notification
{
    use Queueable;

    protected $stats;

    /**
     * Create a new notification instance.
     */
    public function __construct($stats)
    {
        $this->stats = $stats;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $subject = 'تقرير يومي - نظام ZATCA | ' . $this->stats['date'];
        
        $message = (new MailMessage)
            ->subject($subject)
            ->greeting('تقرير يومي عن نظام ZATCA')
            ->line('التاريخ: ' . $this->stats['date'])
            ->line('')
            ->line('📊 إحصائيات الفواتير:')
            ->line('• إجمالي الفواتير المُنشأة: ' . $this->stats['total_invoices'])
            ->line('• المرسلة إلى ZATCA: ' . $this->stats['sent_to_zatca'])
            ->line('• المؤكدة: ' . $this->stats['cleared_invoices'])
            ->line('• المرفوضة: ' . $this->stats['rejected_invoices'])
            ->line('')
            ->line('🔧 حالة النظام:')
            ->line('• إعدادات ZATCA النشطة: ' . $this->stats['active_settings']);

        if ($this->stats['expired_certificates'] > 0) {
            $message->line('⚠️ شهادات تنتهي خلال 30 يوم: ' . $this->stats['expired_certificates']);
        }

        // تحديد حالة النظام العامة
        $systemHealth = $this->getSystemHealthStatus();
        $message->line('')
            ->line("🎯 حالة النظام العامة: {$systemHealth['status']}")
            ->line($systemHealth['message']);

        if (!empty($systemHealth['recommendations'])) {
            $message->line('')
                ->line('💡 التوصيات:');
            foreach ($systemHealth['recommendations'] as $recommendation) {
                $message->line('• ' . $recommendation);
            }
        }

        $message->action('عرض نافذة المزامنة', url('/invoices/sync'))
            ->line('شكراً لاستخدامكم نظامنا.');

        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $systemHealth = $this->getSystemHealthStatus();
        
        return [
            'type' => 'daily_zatca_status',
            'title' => 'التقرير اليومي - نظام ZATCA',
            'message' => "الفواتير: {$this->stats['total_invoices']} | المرسلة: {$this->stats['sent_to_zatca']} | المؤكدة: {$this->stats['cleared_invoices']}",
            'stats' => $this->stats,
            'system_health' => $systemHealth,
            'icon' => 'fas fa-chart-line',
            'color' => $systemHealth['color']
        ];
    }

    /**
     * تحديد حالة النظام العامة
     */
    private function getSystemHealthStatus(): array
    {
        $recommendations = [];
        $status = 'ممتاز';
        $color = 'success';
        $message = 'النظام يعمل بكفاءة عالية';

        // التحقق من معدل الإرسال
        if ($this->stats['total_invoices'] > 0) {
            $sendRate = ($this->stats['sent_to_zatca'] / $this->stats['total_invoices']) * 100;
            
            if ($sendRate < 50) {
                $status = 'يحتاج انتباه';
                $color = 'warning';
                $message = 'معدل إرسال الفواتير أقل من المتوقع';
                $recommendations[] = 'مراجعة إعدادات ZATCA والتأكد من صحة الاتصال';
            }
        }

        // التحقق من معدل الرفض
        if ($this->stats['sent_to_zatca'] > 0) {
            $rejectionRate = ($this->stats['rejected_invoices'] / $this->stats['sent_to_zatca']) * 100;
            
            if ($rejectionRate > 10) {
                $status = 'يحتاج انتباه';
                $color = 'warning';
                $message = 'معدل رفض الفواتير مرتفع';
                $recommendations[] = 'مراجعة بيانات الفواتير والتأكد من اكتمالها';
                $recommendations[] = 'التحقق من صحة إعدادات الشهادات';
            }
        }

        // التحقق من الشهادات المنتهية الصلاحية
        if ($this->stats['expired_certificates'] > 0) {
            $status = 'خطر';
            $color = 'danger';
            $message = 'يوجد شهادات تحتاج تجديد';
            $recommendations[] = 'تجديد الشهادات المنتهية الصلاحية فوراً';
            $recommendations[] = 'التواصل مع هيئة الزكاة لتجديد الشهادات';
        }

        // التحقق من عدم وجود إعدادات نشطة
        if ($this->stats['active_settings'] == 0) {
            $status = 'خطر';
            $color = 'danger';
            $message = 'لا توجد إعدادات ZATCA نشطة';
            $recommendations[] = 'تفعيل إعدادات ZATCA للشركات';
            $recommendations[] = 'التأكد من صحة الشهادات والمفاتيح';
        }

        return [
            'status' => $status,
            'color' => $color,
            'message' => $message,
            'recommendations' => $recommendations
        ];
    }
}