<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ZatcaSettings extends Model
{
    protected $fillable = [
        'company_id',
        'connection_type',
        'environment',
        'certificate',
        'private_key',
        'public_key',
        'certificate_serial',
        'certificate_expires_at',
        'client_id',
        'client_secret',
        'username',
        'password',
        'access_token',
        'token_expires_at',
        'invoice_counter',
        'previous_invoice_hash',
        'is_active',
        'last_connection_test',
        'connection_status',
        'last_error',
        'operation_log',
    ];

    protected $casts = [
        'certificate_expires_at' => 'datetime',
        'token_expires_at' => 'datetime',
        'last_connection_test' => 'datetime',
        'is_active' => 'boolean',
        'operation_log' => 'json',
    ];

    protected $hidden = [
        'certificate',
        'private_key',
        'client_secret',
        'password',
        'access_token',
    ];

    /**
     * علاقة مع الشركة
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * التحقق من انتهاء صلاحية الشهادة
     */
    public function isCertificateExpired(): bool
    {
        return $this->certificate_expires_at && $this->certificate_expires_at->isPast();
    }

    /**
     * التحقق من انتهاء صلاحية الرمز المميز
     */
    public function isTokenExpired(): bool
    {
        return $this->token_expires_at && $this->token_expires_at->isPast();
    }

    /**
     * التحقق من صحة الإعدادات
     */
    public function isValid(): bool
    {
        return $this->is_active
            && !empty($this->certificate)
            && !empty($this->private_key)
            && !$this->isCertificateExpired()
            && !$this->isTokenExpired();
    }

    /**
     * الحصول على URL البيئة
     */
    public function getBaseUrl(): string
    {
        return $this->environment === 'production'
            ? config('zatca.production_base_url')
            : config('zatca.sandbox_base_url');
    }

    /**
     * تسجيل عملية في السجل
     */
    public function logOperation(string $operation, array $data = []): void
    {
        $log = $this->operation_log ?? [];
        $log[] = [
            'operation' => $operation,
            'data' => $data,
            'timestamp' => now(),
        ];
        
        // الاحتفاظ بآخر 100 عملية فقط
        if (count($log) > 100) {
            $log = array_slice($log, -100);
        }
        
        $this->operation_log = $log;
        $this->save();
    }

    /**
     * تحديث حالة الاتصال
     */
    public function updateConnectionStatus(string $status, string $error = null): void
    {
        $this->connection_status = $status;
        $this->last_connection_test = now();
        if ($error) {
            $this->last_error = $error;
        }
        $this->save();
    }

    /**
     * زيادة عداد الفواتير
     */
    public function incrementInvoiceCounter(): int
    {
        $this->increment('invoice_counter');
        return $this->invoice_counter;
    }

    /**
     * تحديث hash الفاتورة السابقة
     */
    public function updatePreviousInvoiceHash(string $hash): void
    {
        $this->previous_invoice_hash = $hash;
        $this->save();
    }

    /**
     * فلترة الإعدادات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * فلترة حسب البيئة
     */
    public function scopeEnvironment($query, $environment)
    {
        return $query->where('environment', $environment);
    }
}
