<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SupplierPaymentAllocation extends Model
{
    use HasFactory;

    protected $fillable = [
        'payment_voucher_id',
        'purchase_invoice_id',
        'allocated_amount',
        'notes',
    ];

    protected $casts = [
        'allocated_amount' => 'decimal:2',
    ];

    /**
     * Get the payment voucher associated with this allocation.
     */
    public function paymentVoucher(): BelongsTo
    {
        return $this->belongsTo(PaymentVoucher::class);
    }

    /**
     * Get the purchase invoice associated with this allocation.
     */
    public function purchaseInvoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    /**
     * Get the supplier through the purchase invoice.
     */
    public function getSupplierAttribute()
    {
        return $this->purchaseInvoice->supplier;
    }

    /**
     * Scope for a specific payment voucher.
     */
    public function scopeForPayment($query, $paymentId)
    {
        return $query->where('payment_voucher_id', $paymentId);
    }

    /**
     * Scope for a specific purchase invoice.
     */
    public function scopeForInvoice($query, $invoiceId)
    {
        return $query->where('purchase_invoice_id', $invoiceId);
    }

    /**
     * Get the percentage of the invoice amount allocated.
     */
    public function getAllocationPercentageAttribute(): float
    {
        if ($this->purchaseInvoice && $this->purchaseInvoice->total_amount > 0) {
            return ($this->allocated_amount / $this->purchaseInvoice->total_amount) * 100;
        }
        return 0;
    }

    /**
     * Check if this allocation fully pays the invoice.
     */
    public function getIsFullAllocationAttribute(): bool
    {
        return $this->purchaseInvoice && 
               $this->allocated_amount >= $this->purchaseInvoice->remaining_amount;
    }

    /**
     * Create allocation for a payment.
     */
    public static function allocatePayment(
        int $paymentVoucherId,
        int $purchaseInvoiceId,
        float $amount,
        string $notes = null
    ): self {
        return self::create([
            'payment_voucher_id' => $paymentVoucherId,
            'purchase_invoice_id' => $purchaseInvoiceId,
            'allocated_amount' => $amount,
            'notes' => $notes,
        ]);
    }

    /**
     * Boot method to handle model events.
     */
    protected static function boot()
    {
        parent::boot();

        static::created(function ($allocation) {
            // Update the purchase invoice when allocation is created
            $allocation->purchaseInvoice->addPayment($allocation->allocated_amount);
        });

        static::deleted(function ($allocation) {
            // Reverse the payment when allocation is deleted
            if ($allocation->purchaseInvoice) {
                $allocation->purchaseInvoice->paid_amount -= $allocation->allocated_amount;
                $allocation->purchaseInvoice->remaining_amount += $allocation->allocated_amount;
                
                // Update payment status
                if ($allocation->purchaseInvoice->remaining_amount > 0) {
                    if ($allocation->purchaseInvoice->paid_amount > 0) {
                        $allocation->purchaseInvoice->payment_status = 'partial';
                    } else {
                        $allocation->purchaseInvoice->payment_status = 'unpaid';
                    }
                }
                
                $allocation->purchaseInvoice->save();
                
                // Update supplier balance
                $allocation->purchaseInvoice->supplier->updateBalance(
                    $allocation->allocated_amount, 
                    'add'
                );
            }
        });
    }
}