<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class SupplierDebtSchedule extends Model
{
    use HasFactory;

    protected $fillable = [
        'supplier_id',
        'purchase_invoice_id',
        'schedule_date',
        'scheduled_amount',
        'status',
        'notes',
        'reminder_sent_at',
        'payment_deadline',
        'penalty_amount',
        'is_overdue',
    ];

    protected $casts = [
        'schedule_date' => 'date',
        'payment_deadline' => 'date',
        'reminder_sent_at' => 'datetime',
        'scheduled_amount' => 'decimal:2',
        'penalty_amount' => 'decimal:2',
        'is_overdue' => 'boolean',
    ];

    /**
     * Get the supplier this schedule belongs to.
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the purchase invoice this schedule is for.
     */
    public function purchaseInvoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    /**
     * Check if schedule is overdue.
     */
    public function getIsOverdueAttribute(): bool
    {
        return $this->schedule_date && $this->schedule_date->isPast() && $this->status !== 'paid';
    }

    /**
     * Get days overdue.
     */
    public function getDaysOverdueAttribute(): int
    {
        if (!$this->is_overdue) return 0;
        return $this->schedule_date->diffInDays(now());
    }

    /**
     * Get status in Arabic.
     */
    public function getStatusLabelAttribute(): string
    {
        $labels = [
            'scheduled' => 'مجدولة',
            'reminded' => 'تم التذكير',
            'overdue' => 'متأخرة',
            'paid' => 'مدفوعة',
            'cancelled' => 'ملغاة',
        ];

        return $labels[$this->status] ?? $this->status;
    }

    /**
     * Mark as paid.
     */
    public function markAsPaid(): void
    {
        $this->status = 'paid';
        $this->save();
    }

    /**
     * Mark as overdue and calculate penalty.
     */
    public function markAsOverdue(): void
    {
        $this->status = 'overdue';
        $this->is_overdue = true;
        
        // Calculate penalty based on supplier's terms
        $penaltyRate = $this->supplier->penalty_rate ?? 0.02; // 2% default
        $this->penalty_amount = $this->scheduled_amount * $penaltyRate;
        
        $this->save();
    }

    /**
     * Send reminder notification.
     */
    public function sendReminder(): void
    {
        $this->status = 'reminded';
        $this->reminder_sent_at = now();
        $this->save();

        // Here you would trigger notification logic
        // event(new DebtReminderSent($this));
    }

    /**
     * Scope for overdue schedules.
     */
    public function scopeOverdue($query)
    {
        return $query->where('schedule_date', '<', now())
                    ->where('status', '!=', 'paid');
    }

    /**
     * Scope for schedules due soon.
     */
    public function scopeDueSoon($query, $days = 7)
    {
        return $query->whereBetween('schedule_date', [now(), now()->addDays($days)])
                    ->where('status', '!=', 'paid');
    }

    /**
     * Scope for specific supplier.
     */
    public function scopeForSupplier($query, $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    /**
     * Create payment schedule for invoice.
     */
    public static function createScheduleForInvoice(PurchaseInvoice $invoice): void
    {
        $supplier = $invoice->supplier;
        $paymentTerms = $supplier->payment_terms_days ?? 30;
        
        // Create main payment schedule
        static::create([
            'supplier_id' => $supplier->id,
            'purchase_invoice_id' => $invoice->id,
            'schedule_date' => $invoice->due_date,
            'scheduled_amount' => $invoice->remaining_amount,
            'status' => 'scheduled',
            'payment_deadline' => $invoice->due_date->addDays(5), // Grace period
        ]);

        // Create installments if applicable
        if ($invoice->total_amount > $supplier->installment_threshold ?? 50000) {
            static::createInstallmentSchedule($invoice, $supplier);
        }
    }

    /**
     * Create installment schedule.
     */
    protected static function createInstallmentSchedule(PurchaseInvoice $invoice, Supplier $supplier): void
    {
        $installments = $supplier->max_installments ?? 3;
        $amountPerInstallment = $invoice->remaining_amount / $installments;
        
        for ($i = 1; $i <= $installments; $i++) {
            $scheduleDate = $invoice->invoice_date->addDays($i * 30);
            
            static::create([
                'supplier_id' => $supplier->id,
                'purchase_invoice_id' => $invoice->id,
                'schedule_date' => $scheduleDate,
                'scheduled_amount' => $amountPerInstallment,
                'status' => 'scheduled',
                'payment_deadline' => $scheduleDate->addDays(5),
                'notes' => "قسط {$i} من {$installments}",
            ]);
        }
    }
}