<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Supplier extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'commercial_registration',
        'tax_number',
        'contact_person',
        'phone',
        'mobile',
        'email',
        'address',
        'city',
        'country',
        'postal_code',
        'currency',
        'credit_limit',
        'payment_terms_days',
        'payment_method',
        'current_balance',
        'notes',
        'is_active',
    ];

    protected $casts = [
        'credit_limit' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'payment_terms_days' => 'integer',
        'penalty_rate' => 'decimal:4',
        'installment_threshold' => 'decimal:2',
        'max_installments' => 'integer',
        'delivery_days' => 'integer',
        'is_active' => 'boolean',
    ];

    /**
     * Get all purchase invoices for this supplier.
     */
    public function purchaseInvoices(): HasMany
    {
        return $this->hasMany(PurchaseInvoice::class);
    }

    /**
     * Get all purchase orders for this supplier.
     */
    public function purchaseOrders(): HasMany
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    /**
     * Get all payment vouchers for this supplier.
     */
    public function paymentVouchers(): HasMany
    {
        return $this->hasMany(PaymentVoucher::class);
    }

    /**
     * Get all debt schedules for this supplier.
     */
    public function debtSchedules(): HasMany
    {
        return $this->hasMany(SupplierDebtSchedule::class);
    }

    /**
     * Get all payment allocations for this supplier through invoices.
     */
    public function paymentAllocations()
    {
        return $this->hasManyThrough(
            SupplierPaymentAllocation::class,
            PurchaseInvoice::class,
            'supplier_id',
            'purchase_invoice_id'
        );
    }

    /**
     * Get active suppliers only.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get suppliers with outstanding balance.
     */
    public function scopeWithBalance($query)
    {
        return $query->where('current_balance', '>', 0);
    }

    /**
     * Generate next supplier code.
     */
    public static function generateNextCode(): string
    {
        $lastSupplier = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastSupplier ? (int)substr($lastSupplier->code, 3) + 1 : 1;
        return 'SUP' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get supplier's total purchases this year.
     */
    public function getTotalPurchasesThisYear(): float
    {
        return $this->purchaseInvoices()
            ->whereYear('invoice_date', date('Y'))
            ->where('status', '!=', 'cancelled')
            ->sum('total_amount');
    }

    /**
     * Get supplier's payment status.
     */
    public function getPaymentStatusAttribute(): string
    {
        if ($this->current_balance == 0) {
            return 'paid';
        } elseif ($this->current_balance > 0) {
            // Check if there are overdue invoices
            $overdueCount = $this->purchaseInvoices()
                ->where('payment_status', 'overdue')
                ->count();
            
            return $overdueCount > 0 ? 'overdue' : 'outstanding';
        }
        
        return 'credit';
    }

    /**
     * Get supplier's overdue amount.
     */
    public function getOverdueAmount(): float
    {
        return $this->purchaseInvoices()
            ->where('payment_status', 'overdue')
            ->sum('remaining_amount');
    }

    /**
     * Check if supplier exceeds credit limit.
     */
    public function exceedsCreditLimit(): bool
    {
        return $this->credit_limit > 0 && $this->current_balance > $this->credit_limit;
    }

    /**
     * Get available credit.
     */
    public function getAvailableCredit(): float
    {
        if ($this->credit_limit <= 0) {
            return 0;
        }
        
        return max(0, $this->credit_limit - $this->current_balance);
    }

    /**
     * Update supplier balance.
     */
    public function updateBalance(float $amount, string $operation = 'add'): void
    {
        if ($operation === 'add') {
            $this->current_balance += $amount;
        } elseif ($operation === 'subtract') {
            $this->current_balance -= $amount;
        } else {
            $this->current_balance = $amount;
        }
        
        $this->save();
    }

    /**
     * Get supplier's top purchased items.
     */
    public function getTopPurchasedItems(int $limit = 5)
    {
        return \DB::table('purchase_invoice_details')
            ->join('purchase_invoices', 'purchase_invoice_details.purchase_invoice_id', '=', 'purchase_invoices.id')
            ->join('items', 'purchase_invoice_details.item_id', '=', 'items.id')
            ->where('purchase_invoices.supplier_id', $this->id)
            ->where('purchase_invoices.status', '!=', 'cancelled')
            ->select(
                'items.name',
                'items.code',
                \DB::raw('SUM(purchase_invoice_details.quantity) as total_quantity'),
                \DB::raw('SUM(purchase_invoice_details.total_amount) as total_amount')
            )
            ->groupBy('items.id', 'items.name', 'items.code')
            ->orderByDesc('total_amount')
            ->limit($limit)
            ->get();
    }

    /**
     * Get credit utilization percentage.
     */
    public function getCreditUtilizationAttribute(): float
    {
        if ($this->credit_limit <= 0) return 0;
        return ($this->current_balance / $this->credit_limit) * 100;
    }

    /**
     * Get available credit amount.
     */
    public function getAvailableCreditAttribute(): float
    {
        return max(0, $this->credit_limit - $this->current_balance);
    }

    /**
     * Get risk level based on credit utilization.
     */
    public function getRiskLevelAttribute(): string
    {
        $utilization = $this->credit_utilization;
        
        if ($utilization >= 90) return 'critical';
        if ($utilization >= 75) return 'high';
        if ($utilization >= 50) return 'medium';
        return 'low';
    }

    /**
     * Check if supplier has overdue invoices.
     */
    public function hasOverdueInvoices(): bool
    {
        return $this->purchaseInvoices()
            ->where('payment_status', '!=', 'paid')
            ->where('due_date', '<', now())
            ->exists();
    }

    /**
     * Get total payment amount for this year.
     */
    public function getTotalPaymentsThisYear(): float
    {
        return $this->paymentVouchers()
            ->whereYear('payment_date', date('Y'))
            ->sum('amount');
    }

    /**
     * Get average payment days.
     */
    public function getAveragePaymentDays(): float
    {
        $paidInvoices = $this->purchaseInvoices()
            ->where('payment_status', 'paid')
            ->get();

        if ($paidInvoices->isEmpty()) return 0;

        $totalDays = 0;
        foreach ($paidInvoices as $invoice) {
            $totalDays += $invoice->invoice_date->diffInDays($invoice->updated_at);
        }

        return $totalDays / $paidInvoices->count();
    }

    /**
     * Get supplier performance metrics.
     */
    public function getPerformanceMetrics(): array
    {
        $totalInvoices = $this->purchaseInvoices()->count();
        $paidInvoices = $this->purchaseInvoices()->where('payment_status', 'paid')->count();
        $overdueInvoices = $this->purchaseInvoices()->where('payment_status', 'overdue')->count();

        return [
            'total_purchases' => $this->getTotalPurchasesThisYear(),
            'total_payments' => $this->getTotalPaymentsThisYear(),
            'total_invoices' => $totalInvoices,
            'paid_invoices' => $paidInvoices,
            'overdue_invoices' => $overdueInvoices,
            'payment_rate' => $totalInvoices > 0 ? ($paidInvoices / $totalInvoices) * 100 : 0,
            'average_payment_days' => $this->getAveragePaymentDays(),
            'credit_utilization' => $this->credit_utilization,
            'available_credit' => $this->available_credit,
            'risk_level' => $this->risk_level,
            'current_balance' => $this->current_balance,
            'overdue_amount' => $this->getOverdueAmount(),
        ];
    }

    /**
     * Boot method to handle model events.
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($supplier) {
            if (empty($supplier->code)) {
                $supplier->code = static::generateNextCode();
            }
        });
    }
}