<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class SavedReport extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'report_type',
        'description',
        'parameters',
        'filters',
        'format',
        'file_path',
        'file_size',
        'is_scheduled',
        'schedule_frequency',
        'schedule_config',
        'last_generated_at',
        'next_generation_at',
        'is_public',
        'is_active',
        'recipients',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'parameters' => 'array',
        'filters' => 'array',
        'schedule_config' => 'array',
        'recipients' => 'array',
        'is_scheduled' => 'boolean',
        'is_public' => 'boolean',
        'is_active' => 'boolean',
        'last_generated_at' => 'datetime',
        'next_generation_at' => 'datetime',
    ];

    /**
     * العلاقات
     */
    
    // منشئ التقرير
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // آخر محدث للتقرير
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scopes
     */
    
    // التقارير المجدولة
    public function scopeScheduled($query)
    {
        return $query->where('is_scheduled', true);
    }

    // التقارير النشطة
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // التقارير العامة
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    // التقارير حسب النوع
    public function scopeByType($query, $type)
    {
        return $query->where('report_type', $type);
    }

    // التقارير حسب المنشئ
    public function scopeByCreator($query, $userId)
    {
        return $query->where('created_by', $userId);
    }

    // التقارير المستحقة للتوليد
    public function scopeDueForGeneration($query)
    {
        return $query->where('is_scheduled', true)
                    ->where('is_active', true)
                    ->where('next_generation_at', '<=', now());
    }

    // التقارير حسب التنسيق
    public function scopeByFormat($query, $format)
    {
        return $query->where('format', $format);
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // وصف نوع التقرير
    public function getReportTypeDescriptionAttribute()
    {
        $descriptions = [
            'trial_balance' => 'ميزان المراجعة',
            'balance_sheet' => 'الميزانية العمومية',
            'income_statement' => 'قائمة الدخل',
            'cash_flow' => 'التدفق النقدي',
            'general_ledger' => 'دفتر الأستاذ العام',
            'account_statement' => 'كشف حساب',
            'cost_center_report' => 'تقرير مركز التكلفة',
            'budget_report' => 'تقرير الميزانية',
            'custom' => 'تقرير مخصص',
        ];

        return $descriptions[$this->report_type] ?? $this->report_type;
    }

    // وصف تكرار الجدولة
    public function getScheduleFrequencyDescriptionAttribute()
    {
        $descriptions = [
            'daily' => 'يومياً',
            'weekly' => 'أسبوعياً',
            'monthly' => 'شهرياً',
            'quarterly' => 'ربع سنوي',
            'annually' => 'سنوياً',
        ];

        return $descriptions[$this->schedule_frequency] ?? $this->schedule_frequency;
    }

    // حجم الملف المُنسق
    public function getFormattedFileSizeAttribute()
    {
        if (!$this->file_size) {
            return 'غير معروف';
        }

        $units = ['بايت', 'كيلوبايت', 'ميجابايت', 'جيجابايت'];
        $size = $this->file_size;
        $unitIndex = 0;

        while ($size >= 1024 && $unitIndex < count($units) - 1) {
            $size /= 1024;
            $unitIndex++;
        }

        return round($size, 2) . ' ' . $units[$unitIndex];
    }

    // حالة التقرير
    public function getStatusAttribute()
    {
        if (!$this->is_active) {
            return 'معطل';
        }

        if ($this->is_scheduled) {
            if ($this->next_generation_at && $this->next_generation_at->isPast()) {
                return 'مستحق للتوليد';
            }
            return 'مجدول';
        }

        return 'نشط';
    }

    // لون الحالة
    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'معطل' => 'secondary',
            'مستحق للتوليد' => 'warning',
            'مجدول' => 'info',
            'نشط' => 'success',
            default => 'primary'
        };
    }

    // التوقيت التالي للتوليد
    public function getNextGenerationDescriptionAttribute()
    {
        if (!$this->is_scheduled || !$this->next_generation_at) {
            return 'غير مجدول';
        }

        $next = $this->next_generation_at;
        $now = now();

        if ($next->isPast()) {
            return 'مستحق الآن';
        }

        $diff = $now->diffForHumans($next, true);
        return "خلال {$diff}";
    }

    /**
     * الدوال المساعدة
     */
    
    // توليد التقرير
    public function generate($parameters = null)
    {
        $parameters = $parameters ?? $this->parameters;
        
        // تحديد خدمة التقرير حسب النوع
        $reportService = $this->getReportService();
        
        if (!$reportService) {
            throw new \Exception('نوع التقرير غير مدعوم');
        }

        // توليد التقرير
        $reportData = $reportService->generate($parameters, $this->filters);
        
        // حفظ التقرير
        $filePath = $this->saveReportFile($reportData);
        
        // تحديث معلومات التقرير
        $this->update([
            'file_path' => $filePath,
            'file_size' => Storage::size($filePath),
            'last_generated_at' => now(),
            'next_generation_at' => $this->calculateNextGeneration(),
        ]);

        // إرسال للمستلمين إذا كان مجدولاً
        if ($this->is_scheduled && !empty($this->recipients)) {
            $this->sendToRecipients();
        }

        // تسجيل في سجل المراجعة
        AuditLog::logEvent('generated', $this, 'تم توليد التقرير');

        return $filePath;
    }

    // الحصول على خدمة التقرير
    private function getReportService()
    {
        return match($this->report_type) {
            'trial_balance' => app('App\Services\Reports\TrialBalanceService'),
            'balance_sheet' => app('App\Services\Reports\BalanceSheetService'),
            'income_statement' => app('App\Services\Reports\IncomeStatementService'),
            'cash_flow' => app('App\Services\Reports\CashFlowService'),
            'general_ledger' => app('App\Services\Reports\GeneralLedgerService'),
            default => null
        };
    }

    // حفظ ملف التقرير
    private function saveReportFile($reportData)
    {
        $fileName = $this->generateFileName();
        $directory = 'reports/' . $this->report_type;
        $fullPath = $directory . '/' . $fileName;

        // إنشاء المجلد إذا لم يكن موجوداً
        if (!Storage::exists($directory)) {
            Storage::makeDirectory($directory);
        }

        // حفظ الملف حسب التنسيق
        switch ($this->format) {
            case 'excel':
                Storage::put($fullPath, $reportData['excel']);
                break;
            case 'pdf':
                Storage::put($fullPath, $reportData['pdf']);
                break;
            case 'csv':
                Storage::put($fullPath, $reportData['csv']);
                break;
            case 'html':
                Storage::put($fullPath, $reportData['html']);
                break;
        }

        return $fullPath;
    }

    // توليد اسم الملف
    private function generateFileName()
    {
        $timestamp = now()->format('Y-m-d_H-i-s');
        $extension = $this->getFileExtension();
        
        return "{$this->report_type}_{$timestamp}.{$extension}";
    }

    // الحصول على امتداد الملف
    private function getFileExtension()
    {
        return match($this->format) {
            'excel' => 'xlsx',
            'pdf' => 'pdf',
            'csv' => 'csv',
            'html' => 'html',
            default => 'txt'
        };
    }

    // حساب التوقيت التالي للتوليد
    private function calculateNextGeneration()
    {
        if (!$this->is_scheduled) {
            return null;
        }

        $now = now();
        
        return match($this->schedule_frequency) {
            'daily' => $now->addDay(),
            'weekly' => $now->addWeek(),
            'monthly' => $now->addMonth(),
            'quarterly' => $now->addMonths(3),
            'annually' => $now->addYear(),
            default => null
        };
    }

    // إرسال للمستلمين
    private function sendToRecipients()
    {
        if (empty($this->recipients) || !$this->file_path) {
            return;
        }

        // يمكن تطوير هذا لإرسال الإيميلات أو الإشعارات
        foreach ($this->recipients as $recipient) {
            // إرسال إيميل أو إشعار
            // Mail::to($recipient)->send(new ReportGenerated($this));
        }
    }

    // تحديث جدولة التقرير
    public function updateSchedule($frequency, $config = [])
    {
        $this->update([
            'is_scheduled' => true,
            'schedule_frequency' => $frequency,
            'schedule_config' => $config,
            'next_generation_at' => $this->calculateNextGeneration(),
        ]);

        return true;
    }

    // إيقاف الجدولة
    public function stopSchedule()
    {
        $this->update([
            'is_scheduled' => false,
            'schedule_frequency' => null,
            'schedule_config' => null,
            'next_generation_at' => null,
        ]);

        return true;
    }

    // تحديث المستلمين
    public function updateRecipients($recipients)
    {
        $this->update(['recipients' => $recipients]);
        return true;
    }

    // تحميل التقرير
    public function download()
    {
        if (!$this->file_path || !Storage::exists($this->file_path)) {
            throw new \Exception('ملف التقرير غير موجود');
        }

        return Storage::download($this->file_path, $this->name . '.' . $this->getFileExtension());
    }

    // حذف ملف التقرير
    public function deleteFile()
    {
        if ($this->file_path && Storage::exists($this->file_path)) {
            Storage::delete($this->file_path);
            
            $this->update([
                'file_path' => null,
                'file_size' => null,
            ]);
        }

        return true;
    }

    /**
     * الدوال الثابتة
     */
    
    // توليد التقارير المستحقة
    public static function generateDueReports()
    {
        $dueReports = static::dueForGeneration()->get();
        $generated = 0;

        foreach ($dueReports as $report) {
            try {
                $report->generate();
                $generated++;
            } catch (\Exception $e) {
                // تسجيل الخطأ
                AuditLog::logCustomEvent('report_generation_failed', 
                    "فشل في توليد التقرير: {$report->name}", 
                    ['error' => $e->getMessage()],
                    ['report_error']
                );
            }
        }

        return $generated;
    }

    // تنظيف الملفات القديمة
    public static function cleanupOldFiles($daysToKeep = 30)
    {
        $cutoffDate = now()->subDays($daysToKeep);
        $oldReports = static::where('last_generated_at', '<', $cutoffDate)
                           ->whereNotNull('file_path')
                           ->get();

        $cleaned = 0;
        foreach ($oldReports as $report) {
            $report->deleteFile();
            $cleaned++;
        }

        return $cleaned;
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        // تعيين المنشئ عند الإنشاء
        static::creating(function ($report) {
            $report->created_by = auth()->id();
        });

        // تعيين المحدث عند التحديث
        static::updating(function ($report) {
            $report->updated_by = auth()->id();
        });

        // حذف الملف عند حذف التقرير
        static::deleting(function ($report) {
            $report->deleteFile();
        });

        // تسجيل العمليات
        static::created(function ($report) {
            AuditLog::logEvent('created', $report, 'تم إنشاء تقرير محفوظ');
        });

        static::updated(function ($report) {
            AuditLog::logEvent('updated', $report, 'تم تحديث تقرير محفوظ');
        });

        static::deleted(function ($report) {
            AuditLog::logEvent('deleted', $report, 'تم حذف تقرير محفوظ');
        });
    }
}