<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SalesInvoiceDetail extends Model
{
    use HasFactory;

    protected $fillable = [
        'sales_invoice_id',
        'item_id',
        'item_description',
        'quantity',
        'unit',
        'unit_price',
        'discount_percentage',
        'discount_amount',
        'net_price',
        'tax_percentage',
        'tax_amount',
        'total_amount',
        'notes',
    ];

    protected $casts = [
        'quantity' => 'decimal:3',
        'unit_price' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'net_price' => 'decimal:2',
        'tax_percentage' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
    ];

    /**
     * Get the sales invoice that owns the detail.
     */
    public function salesInvoice(): BelongsTo
    {
        return $this->belongsTo(SalesInvoice::class);
    }

    /**
     * Get the item associated with the detail.
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Calculate line totals.
     */
    public function calculateTotals(): void
    {
        // Calculate discount amount if percentage is given
        if ($this->discount_percentage > 0 && $this->discount_amount == 0) {
            $this->discount_amount = ($this->unit_price * $this->discount_percentage) / 100;
        }
        
        // Calculate net price
        $this->net_price = $this->unit_price - $this->discount_amount;
        
        // Calculate line total before tax
        $lineTotal = $this->quantity * $this->net_price;
        
        // Calculate tax amount
        $this->tax_amount = ($lineTotal * $this->tax_percentage) / 100;
        
        // Calculate total amount
        $this->total_amount = $lineTotal + $this->tax_amount;
        
        $this->save();
    }

    /**
     * Get the profit margin for this line.
     */
    public function getProfitMarginAttribute(): float
    {
        if (!$this->item || $this->item->cost_price <= 0) {
            return 0;
        }
        
        $profit = $this->net_price - $this->item->cost_price;
        return ($profit / $this->item->cost_price) * 100;
    }

    /**
     * Get the profit amount for this line.
     */
    public function getProfitAmountAttribute(): float
    {
        if (!$this->item) {
            return 0;
        }
        
        return ($this->net_price - $this->item->cost_price) * $this->quantity;
    }

    /**
     * Get the cost amount for this line.
     */
    public function getCostAmountAttribute(): float
    {
        if (!$this->item) {
            return 0;
        }
        
        return $this->item->cost_price * $this->quantity;
    }

    /**
     * Check if quantity is available in stock.
     */
    public function checkStockAvailability(): bool
    {
        if (!$this->item || !$this->item->is_stockable) {
            return true; // Non-stockable items are always available
        }
        
        return $this->item->available_stock >= $this->quantity;
    }

    /**
     * Reserve stock for this line item.
     */
    public function reserveStock(): bool
    {
        if (!$this->item || !$this->item->is_stockable) {
            return true;
        }
        
        return $this->item->reserveStock($this->quantity);
    }

    /**
     * Release reserved stock for this line item.
     */
    public function releaseReservedStock(): void
    {
        if ($this->item && $this->item->is_stockable) {
            $this->item->releaseReservedStock($this->quantity);
        }
    }

    /**
     * Boot method to handle model events.
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($model) {
            // Auto-calculate totals when saving
            $model->calculateTotals();
        });

        static::saved(function ($model) {
            // Recalculate invoice totals when detail is saved
            $model->salesInvoice->calculateTotals();
        });

        static::deleted(function ($model) {
            // Recalculate invoice totals when detail is deleted
            if ($model->salesInvoice) {
                $model->salesInvoice->calculateTotals();
            }
            
            // Release reserved stock if any
            $model->releaseReservedStock();
        });
    }
}