<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class SalesInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_number',
        'invoice_date',
        'due_date',
        'customer_id',
        'cost_center_id',
        'reference',
        'status',
        'payment_status',
        'subtotal',
        'discount_amount',
        'discount_percentage',
        'tax_amount',
        'tax_percentage',
        'shipping_amount',
        'total_amount',
        'paid_amount',
        'remaining_amount',
        'currency',
        'exchange_rate',
        'payment_method',
        'payment_terms',
        'notes',
        'terms_conditions',
        'delivery_address',
        'delivery_date',
        'delivery_method',
        'journal_entry_id',
        'additional_fields',
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date',
        'delivery_date' => 'date',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'tax_percentage' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'additional_fields' => 'array',
    ];

    /**
     * Get the customer that owns the sales invoice.
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Get the cost center associated with the sales invoice.
     */
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    /**
     * Get the journal entry associated with the sales invoice.
     */
    public function journalEntry(): BelongsTo
    {
        return $this->belongsTo(JournalEntry::class);
    }

    /**
     * Get all sales invoice details.
     */
    public function salesInvoiceDetails(): HasMany
    {
        return $this->hasMany(SalesInvoiceDetail::class);
    }

    /**
     * Get draft invoices.
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    /**
     * Get pending invoices.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Get approved invoices.
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Get paid invoices.
     */
    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    /**
     * Get unpaid invoices.
     */
    public function scopeUnpaid($query)
    {
        return $query->where('payment_status', 'unpaid');
    }

    /**
     * Get overdue invoices.
     */
    public function scopeOverdue($query)
    {
        return $query->where('payment_status', 'overdue')
                    ->orWhere(function($q) {
                        $q->where('payment_status', 'unpaid')
                          ->where('due_date', '<', now());
                    });
    }

    /**
     * Generate next invoice number.
     */
    public static function generateNextInvoiceNumber(): string
    {
        $lastInvoice = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastInvoice ? (int)substr($lastInvoice->invoice_number, 3) + 1 : 1;
        return 'INV' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Calculate totals based on details.
     */
    public function calculateTotals(): void
    {
        $details = $this->salesInvoiceDetails;
        
        $this->subtotal = $details->sum('total_amount');
        
        // Apply discount
        if ($this->discount_percentage > 0) {
            $this->discount_amount = ($this->subtotal * $this->discount_percentage) / 100;
        }
        
        $afterDiscount = $this->subtotal - $this->discount_amount;
        
        // Calculate tax
        $this->tax_amount = ($afterDiscount * $this->tax_percentage) / 100;
        
        // Calculate total
        $this->total_amount = $afterDiscount + $this->tax_amount + $this->shipping_amount;
        
        // Update remaining amount
        $this->remaining_amount = $this->total_amount - $this->paid_amount;
        
        $this->save();
    }

    /**
     * Add payment to invoice.
     */
    public function addPayment(float $amount): void
    {
        $this->paid_amount += $amount;
        $this->remaining_amount = $this->total_amount - $this->paid_amount;
        
        // Update payment status
        if ($this->remaining_amount <= 0) {
            $this->payment_status = 'paid';
        } elseif ($this->paid_amount > 0) {
            $this->payment_status = 'partial';
        }
        
        $this->save();
        
        // Update customer balance
        $this->customer->updateBalance($amount, 'subtract');
    }

    /**
     * Check if invoice is overdue.
     */
    public function getIsOverdueAttribute(): bool
    {
        return $this->payment_status !== 'paid' && 
               $this->due_date < now() && 
               $this->status !== 'cancelled';
    }

    /**
     * Get days overdue.
     */
    public function getDaysOverdueAttribute(): int
    {
        if (!$this->is_overdue) {
            return 0;
        }
        
        return now()->diffInDays($this->due_date);
    }

    /**
     * Approve the invoice.
     */
    public function approve(): void
    {
        if ($this->status === 'pending') {
            $this->status = 'approved';
            $this->save();
            
            // Create journal entry if not exists
            if (!$this->journal_entry_id) {
                $this->createJournalEntry();
            }
            
            // Update customer balance
            $this->customer->updateBalance($this->total_amount, 'add');
        }
    }

    /**
     * Cancel the invoice.
     */
    public function cancel(): void
    {
        if (in_array($this->status, ['draft', 'pending', 'approved'])) {
            $oldStatus = $this->status;
            $this->status = 'cancelled';
            $this->save();
            
            // If was approved, reverse customer balance
            if ($oldStatus === 'approved') {
                $this->customer->updateBalance($this->total_amount, 'subtract');
            }
            
            // Reverse inventory movements if any
            $this->reverseInventoryMovements();
        }
    }

    /**
     * Create journal entry for the invoice.
     */
    public function createJournalEntry(): void
    {
        // Implementation would depend on your chart of accounts structure
        // This is a simplified example
        
        $journalEntry = JournalEntry::create([
            'entry_number' => JournalEntry::generateNextEntryNumber(),
            'entry_date' => $this->invoice_date,
            'description' => "فاتورة مبيعات رقم {$this->invoice_number} - {$this->customer->name}",
            'status' => 'posted',
        ]);
        
        // Debit: Accounts Receivable
        $journalEntry->journalEntryLines()->create([
            'account_id' => 1, // Accounts Receivable account
            'cost_center_id' => $this->cost_center_id,
            'description' => "فاتورة مبيعات {$this->invoice_number}",
            'debit_amount' => $this->total_amount,
            'credit_amount' => 0,
        ]);
        
        // Credit: Sales Revenue
        $journalEntry->journalEntryLines()->create([
            'account_id' => 2, // Sales Revenue account
            'cost_center_id' => $this->cost_center_id,
            'description' => "إيرادات مبيعات {$this->invoice_number}",
            'debit_amount' => 0,
            'credit_amount' => $this->subtotal,
        ]);
        
        // Credit: Tax Payable (if any)
        if ($this->tax_amount > 0) {
            $journalEntry->journalEntryLines()->create([
                'account_id' => 3, // Tax Payable account
                'cost_center_id' => $this->cost_center_id,
                'description' => "ضريبة القيمة المضافة {$this->invoice_number}",
                'debit_amount' => 0,
                'credit_amount' => $this->tax_amount,
            ]);
        }
        
        $this->journal_entry_id = $journalEntry->id;
        $this->save();
    }

    /**
     * Create inventory movements for the invoice.
     */
    public function createInventoryMovements(): void
    {
        foreach ($this->salesInvoiceDetails as $detail) {
            if ($detail->item->is_stockable) {
                InventoryMovement::create([
                    'movement_number' => InventoryMovement::generateNextMovementNumber(),
                    'movement_date' => $this->invoice_date,
                    'item_id' => $detail->item_id,
                    'movement_type' => 'sale',
                    'quantity' => -$detail->quantity, // Negative for outgoing
                    'unit_cost' => $detail->item->cost_price,
                    'total_cost' => $detail->quantity * $detail->item->cost_price,
                    'reference_type' => 'sales_invoice',
                    'reference_id' => $this->id,
                    'cost_center_id' => $this->cost_center_id,
                    'notes' => "بيع حسب فاتورة {$this->invoice_number}",
                    'stock_before' => $detail->item->current_stock,
                    'stock_after' => $detail->item->current_stock - $detail->quantity,
                ]);
                
                // Update item stock
                $detail->item->updateStock($detail->quantity, 'subtract');
            }
        }
    }

    /**
     * Reverse inventory movements.
     */
    public function reverseInventoryMovements(): void
    {
        InventoryMovement::where('reference_type', 'sales_invoice')
                        ->where('reference_id', $this->id)
                        ->delete();
        
        // Restore item stocks
        foreach ($this->salesInvoiceDetails as $detail) {
            if ($detail->item->is_stockable) {
                $detail->item->updateStock($detail->quantity, 'add');
            }
        }
    }
}