<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class ReceiptVoucher extends Model
{
    protected $fillable = [
        'voucher_number',
        'company_id',
        'customer_id',
        'user_id',
        'receipt_date',
        'amount',
        'payment_method',
        'reference_number',
        'notes',
        'invoice_id',
        'account_code',
        'description',
        'status'
    ];

    protected $casts = [
        'receipt_date' => 'datetime',
        'amount' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($voucher) {
            if (empty($voucher->voucher_number)) {
                $voucher->voucher_number = $voucher->generateVoucherNumber();
            }
        });
    }

    /**
     * علاقة مع الشركة
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع العميل
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * علاقة مع المستخدم
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * علاقة مع الفاتورة (اختيارية)
     */
    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * توليد رقم سند القبض
     */
    public function generateVoucherNumber(): string
    {
        $year = now()->year;
        $month = now()->format('m');
        
        $count = static::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->count() + 1;
            
        return sprintf('RV-%d%s-%06d', $year, $month, $count);
    }

    /**
     * الحصول على نص طريقة الدفع بالعربية
     */
    public function getPaymentMethodText(): string
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
            'other' => 'أخرى'
        ];
        
        return $methods[$this->payment_method] ?? 'غير محدد';
    }

    /**
     * الحصول على نص الحالة بالعربية
     */
    public function getStatusText(): string
    {
        $statuses = [
            'active' => 'نشط',
            'cancelled' => 'ملغي',
            'pending' => 'معلق'
        ];
        
        return $statuses[$this->status] ?? 'غير محدد';
    }

    /**
     * الحصول على لون الحالة للعرض
     */
    public function getStatusColor(): string
    {
        $colors = [
            'active' => 'success',
            'cancelled' => 'danger',
            'pending' => 'warning'
        ];
        
        return $colors[$this->status] ?? 'secondary';
    }

    /**
     * فلترة السندات حسب الحالة
     */
    public function scopeOfStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * السندات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * السندات الملغية
     */
    public function scopeCancelled($query)
    {
        return $query->where('status', 'cancelled');
    }

    /**
     * فلترة السندات حسب طريقة الدفع
     */
    public function scopeOfPaymentMethod($query, $method)
    {
        return $query->where('payment_method', $method);
    }

    /**
     * فلترة السندات حسب الفترة الزمنية
     */
    public function scopeForPeriod($query, $startDate, $endDate)
    {
        return $query->whereBetween('receipt_date', [$startDate, $endDate]);
    }
}