<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PurchaseInvoiceDetail extends Model
{
    use HasFactory;

    protected $fillable = [
        'purchase_invoice_id',
        'item_id',
        'item_code',
        'item_name',
        'item_description',
        'quantity',
        'received_quantity',
        'unit',
        'unit_price',
        'discount_percentage',
        'discount_amount',
        'net_price',
        'tax_percentage',
        'tax_amount',
        'total_amount',
        'batch_number',
        'expiry_date',
        'warehouse_id',
        'cost_center_id',
        'notes',
    ];

    protected $casts = [
        'quantity' => 'decimal:3',
        'received_quantity' => 'decimal:3',
        'unit_price' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'net_price' => 'decimal:2',
        'tax_percentage' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'expiry_date' => 'date',
    ];

    /**
     * Boot method to handle model events.
     */
    protected static function boot()
    {
        parent::boot();
        
        static::saving(function ($detail) {
            // Calculate amounts automatically
            $detail->calculateAmounts();
        });
        
        static::saved(function ($detail) {
            // Recalculate invoice totals
            $detail->purchaseInvoice->calculateTotals();
        });
        
        static::deleted(function ($detail) {
            // Recalculate invoice totals
            if ($detail->purchaseInvoice) {
                $detail->purchaseInvoice->calculateTotals();
            }
        });
    }

    /**
     * Get the purchase invoice this detail belongs to.
     */
    public function purchaseInvoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    /**
     * Get the item associated with this detail.
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Get the warehouse associated with this detail.
     */
    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * Get the cost center associated with this detail.
     */
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    /**
     * Calculate line amounts.
     */
    public function calculateAmounts(): void
    {
        $subtotal = $this->quantity * $this->unit_price;
        
        // Calculate discount
        if ($this->discount_percentage > 0) {
            $this->discount_amount = ($subtotal * $this->discount_percentage) / 100;
        } else {
            $this->discount_amount = $this->discount_amount ?? 0;
        }
        
        // Net price after discount
        $this->net_price = $subtotal - $this->discount_amount;
        
        // Calculate tax
        if ($this->tax_percentage > 0) {
            $this->tax_amount = ($this->net_price * $this->tax_percentage) / 100;
        } else {
            $this->tax_amount = $this->tax_amount ?? 0;
        }
        
        // Total amount
        $this->total_amount = $this->net_price + $this->tax_amount;
    }

    /**
     * Get received percentage.
     */
    public function getReceivedPercentageAttribute(): float
    {
        return $this->quantity > 0 ? ($this->received_quantity / $this->quantity) * 100 : 0;
    }

    /**
     * Check if item is fully received.
     */
    public function getIsFullyReceivedAttribute(): bool
    {
        return $this->received_quantity >= $this->quantity;
    }

    /**
     * Get remaining quantity to receive.
     */
    public function getRemainingQuantityAttribute(): float
    {
        return max(0, $this->quantity - $this->received_quantity);
    }

    /**
     * Get the unit cost after all adjustments.
     */
    public function getActualUnitCostAttribute(): float
    {
        return $this->quantity > 0 ? $this->total_amount / $this->quantity : 0;
    }

    /**
     * Check if item is expired or near expiry.
     */
    public function getExpiryStatusAttribute(): string
    {
        if (!$this->expiry_date) {
            return 'no_expiry';
        }
        
        $daysToExpiry = $this->expiry_date->diffInDays(now(), false);
        
        if ($daysToExpiry < 0) {
            return 'expired';
        } elseif ($daysToExpiry <= 30) {
            return 'near_expiry';
        } elseif ($daysToExpiry <= 90) {
            return 'warning';
        } else {
            return 'good';
        }
    }

    /**
     * Get expiry status label in Arabic.
     */
    public function getExpiryStatusLabelAttribute(): string
    {
        $labels = [
            'no_expiry' => 'لا ينتهي',
            'expired' => 'منتهي الصلاحية',
            'near_expiry' => 'قريب الانتهاء',
            'warning' => 'تحذير',
            'good' => 'جيد',
        ];

        return $labels[$this->expiry_status] ?? 'غير محدد';
    }

    /**
     * Update inventory when item is received.
     */
    public function updateInventory(): void
    {
        if ($this->item && $this->received_quantity > 0) {
            // Create inventory transaction
            $this->item->addStock(
                $this->received_quantity,
                $this->actual_unit_cost,
                'purchase_invoice',
                $this->purchase_invoice_id,
                $this->warehouse_id,
                [
                    'batch_number' => $this->batch_number,
                    'expiry_date' => $this->expiry_date,
                    'supplier_id' => $this->purchaseInvoice->supplier_id,
                ]
            );
        }
    }

    /**
     * Get line summary for reporting.
     */
    public function getSummaryAttribute(): array
    {
        return [
            'item_code' => $this->item_code,
            'item_name' => $this->item_name,
            'quantity' => $this->quantity,
            'received_quantity' => $this->received_quantity,
            'unit' => $this->unit,
            'unit_price' => $this->unit_price,
            'discount_amount' => $this->discount_amount,
            'tax_amount' => $this->tax_amount,
            'total_amount' => $this->total_amount,
            'received_percentage' => $this->received_percentage,
            'remaining_quantity' => $this->remaining_quantity,
            'actual_unit_cost' => $this->actual_unit_cost,
            'expiry_status' => $this->expiry_status_label,
        ];
    }

    /**
     * Scope for items with expiry dates.
     */
    public function scopeWithExpiry($query)
    {
        return $query->whereNotNull('expiry_date');
    }

    /**
     * Scope for expired items.
     */
    public function scopeExpired($query)
    {
        return $query->whereNotNull('expiry_date')
                    ->where('expiry_date', '<', now());
    }

    /**
     * Scope for items near expiry.
     */
    public function scopeNearExpiry($query, $days = 30)
    {
        return $query->whereNotNull('expiry_date')
                    ->where('expiry_date', '>', now())
                    ->where('expiry_date', '<=', now()->addDays($days));
    }

    /**
     * Scope for fully received items.
     */
    public function scopeFullyReceived($query)
    {
        return $query->whereColumn('received_quantity', '>=', 'quantity');
    }

    /**
     * Scope for partially received items.
     */
    public function scopePartiallyReceived($query)
    {
        return $query->where('received_quantity', '>', 0)
                    ->whereColumn('received_quantity', '<', 'quantity');
    }

    /**
     * Scope for unreceived items.
     */
    public function scopeUnreceived($query)
    {
        return $query->where(function($query) {
            $query->where('received_quantity', 0)
                  ->orWhereNull('received_quantity');
        });
    }

    /**
     * Scope for items by warehouse.
     */
    public function scopeByWarehouse($query, $warehouseId)
    {
        return $query->where('warehouse_id', $warehouseId);
    }

    /**
     * Scope for items by cost center.
     */
    public function scopeByCostCenter($query, $costCenterId)
    {
        return $query->where('cost_center_id', $costCenterId);
    }

    /**
     * Create detail from item.
     */
    public static function createFromItem(Item $item, array $data = []): self
    {
        return new static(array_merge([
            'item_id' => $item->id,
            'item_code' => $item->code,
            'item_name' => $item->name,
            'unit' => $item->unit,
            'unit_price' => $item->purchase_price ?? 0,
            'tax_percentage' => $item->tax_rate ?? 15,
        ], $data));
    }
}