<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class PurchaseInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_number',
        'supplier_invoice_number',
        'supplier_id',
        'cost_center_id',
        'invoice_date',
        'due_date',
        'payment_method',
        'payment_terms',
        'delivery_address',
        'delivery_date',
        'subtotal',
        'discount_percentage',
        'discount_amount',
        'tax_percentage',
        'tax_amount',
        'shipping_amount',
        'other_charges',
        'total_amount',
        'paid_amount',
        'remaining_amount',
        'currency',
        'exchange_rate',
        'status',
        'payment_status',
        'notes',
        'terms_conditions',
        'reference',
        'received_by',
        'received_at',
        'approved_by',
        'approved_at',
        'user_id',
        'company_id',
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date',
        'delivery_date' => 'date',
        'received_at' => 'datetime',
        'approved_at' => 'datetime',
        'subtotal' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_percentage' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'shipping_amount' => 'decimal:2',
        'other_charges' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
    ];

    /**
     * Boot method to handle model events.
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($invoice) {
            if (empty($invoice->invoice_number)) {
                $invoice->invoice_number = static::generateNextInvoiceNumber();
            }
            
            if (empty($invoice->currency)) {
                $invoice->currency = 'SAR';
            }
            
            if (empty($invoice->exchange_rate)) {
                $invoice->exchange_rate = 1;
            }
            
            if (empty($invoice->status)) {
                $invoice->status = 'draft';
            }
            
            if (empty($invoice->payment_status)) {
                $invoice->payment_status = 'unpaid';
            }
        });
        
        static::updated(function ($invoice) {
            // Update supplier balance when invoice status changes
            if ($invoice->isDirty('total_amount') || $invoice->isDirty('paid_amount')) {
                $invoice->supplier->updateBalance(
                    $invoice->remaining_amount, 
                    'set'
                );
            }
        });
    }

    /**
     * Get the supplier associated with the invoice.
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the cost center associated with the invoice.
     */
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    /**
     * Get the user who created the invoice.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the company associated with the invoice.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the invoice details (line items).
     */
    public function purchaseInvoiceDetails(): HasMany
    {
        return $this->hasMany(PurchaseInvoiceDetail::class);
    }

    /**
     * Get payment allocations for this invoice.
     */
    public function paymentAllocations(): HasMany
    {
        return $this->hasMany(SupplierPaymentAllocation::class);
    }

    /**
     * Get journal entries related to this invoice.
     */
    public function journalEntries(): HasMany
    {
        return $this->hasMany(JournalEntry::class, 'reference_id')->where('reference_type', 'purchase_invoice');
    }

    /**
     * Generate next invoice number.
     */
    public static function generateNextInvoiceNumber(): string
    {
        $lastInvoice = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastInvoice ? (int)substr($lastInvoice->invoice_number, -6) + 1 : 1;
        return 'PI' . date('Y') . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Check if invoice is overdue.
     */
    public function getIsOverdueAttribute(): bool
    {
        return $this->payment_status !== 'paid' && 
               $this->due_date && 
               $this->due_date->isPast();
    }

    /**
     * Get days overdue.
     */
    public function getDaysOverdueAttribute(): int
    {
        if (!$this->is_overdue) return 0;
        return $this->due_date->diffInDays(now());
    }

    /**
     * Get status label in Arabic.
     */
    public function getStatusLabelAttribute(): string
    {
        $labels = [
            'draft' => 'مسودة',
            'pending' => 'معلقة',
            'approved' => 'معتمدة',
            'received' => 'مستلمة',
            'cancelled' => 'ملغاة',
        ];

        return $labels[$this->status] ?? $this->status;
    }

    /**
     * Get payment status label in Arabic.
     */
    public function getPaymentStatusLabelAttribute(): string
    {
        $labels = [
            'unpaid' => 'غير مدفوعة',
            'partial' => 'مدفوعة جزئياً',
            'paid' => 'مدفوعة',
            'overdue' => 'متأخرة',
        ];

        return $labels[$this->payment_status] ?? $this->payment_status;
    }

    /**
     * Get payment method label in Arabic.
     */
    public function getPaymentMethodLabelAttribute(): string
    {
        $labels = [
            'cash' => 'نقداً',
            'credit' => 'آجل',
            'bank_transfer' => 'تحويل بنكي',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
        ];

        return $labels[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Calculate and update totals.
     */
    public function calculateTotals(): void
    {
        $this->subtotal = $this->purchaseInvoiceDetails->sum('total_amount');
        
        // Apply invoice-level discount
        $this->discount_amount = ($this->subtotal * $this->discount_percentage) / 100;
        $afterDiscount = $this->subtotal - $this->discount_amount;
        
        // Apply invoice-level tax
        $this->tax_amount = ($afterDiscount * $this->tax_percentage) / 100;
        
        // Calculate total
        $this->total_amount = $afterDiscount + $this->tax_amount + $this->shipping_amount + $this->other_charges;
        
        // Update remaining amount
        $this->remaining_amount = $this->total_amount - $this->paid_amount;
        
        // Update payment status
        $this->updatePaymentStatus();
        
        $this->save();
    }

    /**
     * Update payment status based on amounts.
     */
    public function updatePaymentStatus(): void
    {
        if ($this->remaining_amount <= 0) {
            $this->payment_status = 'paid';
        } elseif ($this->paid_amount > 0) {
            $this->payment_status = 'partial';
        } else {
            $this->payment_status = $this->is_overdue ? 'overdue' : 'unpaid';
        }
    }

    /**
     * Add payment to invoice.
     */
    public function addPayment(float $amount): void
    {
        $this->paid_amount += $amount;
        $this->remaining_amount = $this->total_amount - $this->paid_amount;
        $this->updatePaymentStatus();
        $this->save();
        
        // Update supplier balance
        $this->supplier->updateBalance($amount, 'subtract');
    }

    /**
     * Approve the invoice.
     */
    public function approve(int $userId = null): bool
    {
        if ($this->status !== 'pending') {
            return false;
        }
        
        $this->status = 'approved';
        $this->approved_by = $userId ?? auth()->id();
        $this->approved_at = now();
        $this->save();
        
        // Create journal entry for the invoice
        $this->createJournalEntry();
        
        // Update supplier balance
        $this->supplier->updateBalance($this->total_amount, 'add');
        
        return true;
    }

    /**
     * Mark invoice as received.
     */
    public function markAsReceived(array $receivedItems = [], string $receivedBy = null): bool
    {
        if ($this->status !== 'approved') {
            return false;
        }
        
        $this->status = 'received';
        $this->received_by = $receivedBy ?? auth()->user()->name;
        $this->received_at = now();
        
        // Update received quantities for items
        if (!empty($receivedItems)) {
            foreach ($receivedItems as $item) {
                $detail = $this->purchaseInvoiceDetails()->find($item['id']);
                if ($detail) {
                    $detail->received_quantity = $item['received_quantity'];
                    $detail->save();
                    
                    // Update inventory
                    $detail->updateInventory();
                }
            }
        }
        
        $this->save();
        return true;
    }

    /**
     * Cancel the invoice.
     */
    public function cancel(): bool
    {
        if (!in_array($this->status, ['draft', 'pending', 'approved'])) {
            return false;
        }
        
        $oldStatus = $this->status;
        $this->status = 'cancelled';
        $this->save();
        
        // Reverse supplier balance if was approved
        if ($oldStatus === 'approved') {
            $this->supplier->updateBalance($this->total_amount, 'subtract');
        }
        
        return true;
    }

    /**
     * Create journal entry for the invoice.
     */
    protected function createJournalEntry(): void
    {
        // This would create accounting entries
        // Implementation depends on your chart of accounts structure
    }

    /**
     * Scope for invoices by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for invoices by payment status.
     */
    public function scopeByPaymentStatus($query, $paymentStatus)
    {
        return $query->where('payment_status', $paymentStatus);
    }

    /**
     * Scope for overdue invoices.
     */
    public function scopeOverdue($query)
    {
        return $query->where('payment_status', '!=', 'paid')
                    ->where('due_date', '<', now());
    }

    /**
     * Scope for invoices within date range.
     */
    public function scopeInDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('invoice_date', [$startDate, $endDate]);
    }

    /**
     * Scope for invoices by supplier.
     */
    public function scopeBySupplier($query, $supplierId)
    {
        return $query->where('supplier_id', $supplierId);
    }

    /**
     * Get invoice aging category.
     */
    public function getAgingCategoryAttribute(): string
    {
        if ($this->payment_status === 'paid') {
            return 'paid';
        }
        
        if (!$this->due_date) {
            return 'current';
        }
        
        $daysOverdue = $this->due_date->diffInDays(now(), false);
        
        if ($daysOverdue <= 0) {
            return 'current';
        } elseif ($daysOverdue <= 30) {
            return '1_30_days';
        } elseif ($daysOverdue <= 60) {
            return '31_60_days';
        } elseif ($daysOverdue <= 90) {
            return '61_90_days';
        } else {
            return 'over_90_days';
        }
    }

    /**
     * Get received percentage.
     */
    public function getReceivedPercentageAttribute(): float
    {
        $totalQuantity = $this->purchaseInvoiceDetails->sum('quantity');
        $receivedQuantity = $this->purchaseInvoiceDetails->sum('received_quantity');
        
        return $totalQuantity > 0 ? ($receivedQuantity / $totalQuantity) * 100 : 0;
    }

    /**
     * Check if invoice is fully received.
     */
    public function getIsFullyReceivedAttribute(): bool
    {
        return $this->received_percentage >= 100;
    }

    /**
     * Get invoice summary for reporting.
     */
    public function getSummaryAttribute(): array
    {
        return [
            'invoice_number' => $this->invoice_number,
            'supplier_name' => $this->supplier->name,
            'invoice_date' => $this->invoice_date->format('d/m/Y'),
            'due_date' => $this->due_date->format('d/m/Y'),
            'total_amount' => $this->total_amount,
            'paid_amount' => $this->paid_amount,
            'remaining_amount' => $this->remaining_amount,
            'status' => $this->status_label,
            'payment_status' => $this->payment_status_label,
            'days_overdue' => $this->days_overdue,
            'aging_category' => $this->aging_category,
            'received_percentage' => $this->received_percentage,
        ];
    }
}