<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Product extends Model
{
    protected $fillable = [
        'name',
        'name_en',
        'description',
        'sku',
        'barcode',
        'type',
        'category_id',
        'price',
        'price_with_tax',
        'tax_rate',
        'stock_quantity',
        'min_stock',
        'unit',
        'zatca_category_code',
        'is_taxable',
        'is_active',
        'current_stock',
        'reorder_level',
        'cost_price',
        'purchase_price',
        'max_stock',
        'reserved_stock',
        'supplier_id',
        // حقول المطعم
        'item_type',
        'ingredients',
        'allergens',
        'calories',
        'preparation_time',
        'is_spicy',
        'is_vegetarian',
        'is_vegan',
        'is_gluten_free',
        'is_available',
        'spice_level',
        'size_small_price',
        'size_medium_price',
        'size_large_price',
        'customizations',
        'side_dishes',
        'image_url',
        'sort_order',
        'is_featured',
        'is_combo',
        'combo_items',
        'chef_notes',
        'serving_instructions',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'price_with_tax' => 'decimal:2',
        'tax_rate' => 'decimal:2',
        'is_taxable' => 'boolean',
        'is_active' => 'boolean',
        'current_stock' => 'decimal:2',
        'reorder_level' => 'decimal:2',
        'cost_price' => 'decimal:2',
        'purchase_price' => 'decimal:2',
        'max_stock' => 'decimal:2',
        'reserved_stock' => 'decimal:2',
        // حقول المطعم
        'calories' => 'decimal:2',
        'preparation_time' => 'decimal:2',
        'is_spicy' => 'boolean',
        'is_vegetarian' => 'boolean',
        'is_vegan' => 'boolean',
        'is_gluten_free' => 'boolean',
        'is_available' => 'boolean',
        'size_small_price' => 'decimal:2',
        'size_medium_price' => 'decimal:2',
        'size_large_price' => 'decimal:2',
        'customizations' => 'array',
        'side_dishes' => 'array',
        'sort_order' => 'integer',
        'is_featured' => 'boolean',
        'is_combo' => 'boolean',
        'combo_items' => 'array',
    ];

    /**
     * علاقة مع عناصر الفاتورة
     */
    public function invoiceItems(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * علاقة مع عناصر أوامر الشراء
     */
    public function purchaseOrderItems(): HasMany
    {
        return $this->hasMany(PurchaseOrderItem::class);
    }

    /**
     * علاقة مع حركات المخزون
     */
    public function inventoryMovements(): HasMany
    {
        return $this->hasMany(InventoryMovement::class);
    }

    /**
     * علاقة مع عناصر تسوية المخزون
     */
    public function inventoryAdjustmentItems(): HasMany
    {
        return $this->hasMany(InventoryAdjustmentItem::class);
    }

    /**
     * علاقة مع عناصر استلام البضائع
     */
    public function goodsReceiptItems(): HasMany
    {
        return $this->hasMany(GoodsReceiptItem::class);
    }

    /**
     * علاقة مع المورد
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * علاقة مع الفئة
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * حساب السعر مع الضريبة
     */
    public function calculatePriceWithTax(): float
    {
        if (!$this->is_taxable) {
            return $this->price;
        }
        
        return $this->price * (1 + ($this->tax_rate / 100));
    }

    /**
     * حساب مبلغ الضريبة
     */
    public function calculateTaxAmount(): float
    {
        if (!$this->is_taxable) {
            return 0;
        }
        
        return $this->price * ($this->tax_rate / 100);
    }

    /**
     * فلترة المنتجات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * فلترة المنتجات حسب النوع
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * المنتجات المادية
     */
    public function scopeProducts($query)
    {
        return $query->where('type', 'product');
    }

    /**
     * الخدمات
     */
    public function scopeServices($query)
    {
        return $query->where('type', 'service');
    }

    /**
     * المنتجات الخاضعة للضريبة
     */
    public function scopeTaxable($query)
    {
        return $query->where('is_taxable', true);
    }

    /**
     * المنتجات منخفضة المخزون
     */
    public function scopeLowStock($query)
    {
        return $query->whereColumn('stock_quantity', '<=', 'min_stock');
    }

    /**
     * فلترة المنتجات حسب معرف الفئة
     */
    public function scopeOfCategoryId($query, $categoryId)
    {
        return $query->where('category_id', $categoryId);
    }

    /**
     * التحقق من انخفاض المخزون
     */
    public function isLowStock(): bool
    {
        return $this->stock_quantity <= $this->min_stock;
    }

    /**
     * التحقق من نفاد المخزون
     */
    public function isOutOfStock(): bool
    {
        return $this->stock_quantity <= 0;
    }

    /**
     * حساب متوسط التكلفة المرجح
     */
    public function getAverageCost(): float
    {
        $movements = $this->inventoryMovements()
            ->whereIn('type', ['purchase', 'adjustment_in'])
            ->where('quantity', '>', 0)
            ->get();

        if ($movements->isEmpty()) {
            return 0;
        }

        $totalCost = $movements->sum('total_cost');
        $totalQuantity = $movements->sum('quantity');

        return $totalQuantity > 0 ? $totalCost / $totalQuantity : 0;
    }

    /**
     * حساب قيمة المخزون الحالي
     */
    public function getCurrentInventoryValue(): float
    {
        return $this->stock_quantity * $this->getAverageCost();
    }

    /**
     * تحديث المخزون
     */
    public function updateStock(int $quantity, string $type, array $movementData = []): bool
    {
        $oldQuantity = $this->stock_quantity;
        
        if ($type === 'increase') {
            $this->stock_quantity += $quantity;
        } elseif ($type === 'decrease') {
            if ($this->stock_quantity < $quantity) {
                return false; // لا يوجد مخزون كافي
            }
            $this->stock_quantity -= $quantity;
        }

        $this->save();

        // إنشاء حركة مخزون
        $this->inventoryMovements()->create(array_merge([
            'user_id' => auth()->id(),
            'type' => $movementData['movement_type'] ?? $type,
            'quantity' => $quantity,
            'movement_date' => now(),
        ], $movementData));

        return true;
    }

    /**
     * فلترة المنتجات الغذائية
     */
    public function scopeFoodItems($query)
    {
        return $query->where('item_type', 'food');
    }

    /**
     * فلترة المشروبات
     */
    public function scopeBeverages($query)
    {
        return $query->where('item_type', 'beverage');
    }

    /**
     * فلترة الحلويات
     */
    public function scopeDesserts($query)
    {
        return $query->where('item_type', 'dessert');
    }

    /**
     * فلترة المقبلات
     */
    public function scopeAppetizers($query)
    {
        return $query->where('item_type', 'appetizer');
    }

    /**
     * فلترة الأطباق الرئيسية
     */
    public function scopeMainCourses($query)
    {
        return $query->where('item_type', 'main_course');
    }

    /**
     * فلترة وجبات الكومبو
     */
    public function scopeCombos($query)
    {
        return $query->where('is_combo', true);
    }

    /**
     * فلترة المنتجات المتوفرة
     */
    public function scopeAvailable($query)
    {
        return $query->where('is_available', true);
    }

    /**
     * فلترة المنتجات النباتية
     */
    public function scopeVegetarian($query)
    {
        return $query->where('is_vegetarian', true);
    }

    /**
     * فلترة المنتجات النباتية الصرفة
     */
    public function scopeVegan($query)
    {
        return $query->where('is_vegan', true);
    }

    /**
     * فلترة المنتجات الخالية من الجلوتين
     */
    public function scopeGlutenFree($query)
    {
        return $query->where('is_gluten_free', true);
    }

    /**
     * فلترة المنتجات الحارة
     */
    public function scopeSpicy($query)
    {
        return $query->where('is_spicy', true);
    }

    /**
     * فلترة المنتجات المميزة
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * ترتيب حسب ترتيب العرض
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    /**
     * فلترة حسب نوع العنصر
     */
    public function scopeOfItemType($query, $itemType)
    {
        return $query->where('item_type', $itemType);
    }

    /**
     * فلترة حسب الحالة (للتصدير)
     */
    public function scopeOfStatus($query, $status)
    {
        switch ($status) {
            case 'available':
                return $query->where('is_active', true)->where('is_available', true);
            case 'unavailable':
                return $query->where('is_active', true)->where('is_available', false);
            case 'inactive':
                return $query->where('is_active', false);
            default:
                return $query;
        }
    }

    /**
     * فلترة حسب درجة الحرارة
     */
    public function scopeOfSpiceLevel($query, $spiceLevel)
    {
        return $query->where('spice_level', $spiceLevel);
    }

    /**
     * التحقق من كون المنتج متوفر اليوم
     */
    public function isAvailableToday(): bool
    {
        return $this->is_available && $this->is_active;
    }

    /**
     * الحصول على السعر حسب الحجم
     */
    public function getPriceBySize($size = 'regular'): float
    {
        switch ($size) {
            case 'small':
                return $this->size_small_price ?? $this->price;
            case 'medium':
                return $this->size_medium_price ?? $this->price;
            case 'large':
                return $this->size_large_price ?? $this->price;
            default:
                return $this->price;
        }
    }

    /**
     * الحصول على جميع الأحجام المتاحة مع أسعارها
     */
    public function getAvailableSizes(): array
    {
        $sizes = [];
        
        if ($this->size_small_price > 0) {
            $sizes['small'] = [
                'name' => 'صغير',
                'price' => $this->size_small_price
            ];
        }
        
        if ($this->size_medium_price > 0) {
            $sizes['medium'] = [
                'name' => 'وسط',
                'price' => $this->size_medium_price
            ];
        }
        
        if ($this->size_large_price > 0) {
            $sizes['large'] = [
                'name' => 'كبير',
                'price' => $this->size_large_price
            ];
        }
        
        if (empty($sizes)) {
            $sizes['regular'] = [
                'name' => 'عادي',
                'price' => $this->price
            ];
        }
        
        return $sizes;
    }

    /**
     * الحصول على وصف نوع العنصر بالعربية
     */
    public function getItemTypeNameAttribute(): string
    {
        $types = [
            'food' => 'طعام',
            'beverage' => 'مشروب',
            'dessert' => 'حلوى',
            'appetizer' => 'مقبلات',
            'main_course' => 'طبق رئيسي',
            'combo' => 'وجبة كومبو',
            'other' => 'أخرى'
        ];
        
        return $types[$this->item_type] ?? 'غير محدد';
    }

    /**
     * الحصول على وصف درجة الحرارة بالعربية
     */
    public function getSpiceLevelNameAttribute(): string
    {
        if (!$this->spice_level) {
            return 'غير حار';
        }
        
        $levels = [
            'mild' => 'خفيف',
            'medium' => 'متوسط',
            'hot' => 'حار',
            'very_hot' => 'حار جداً'
        ];
        
        return $levels[$this->spice_level] ?? 'غير محدد';
    }

    /**
     * الحصول على الرموز التعبيرية للخصائص الغذائية
     */
    public function getDietaryBadges(): array
    {
        $badges = [];
        
        if ($this->is_vegetarian) {
            $badges[] = ['text' => 'نباتي', 'class' => 'bg-success', 'icon' => '🥗'];
        }
        
        if ($this->is_vegan) {
            $badges[] = ['text' => 'نباتي صرف', 'class' => 'bg-success', 'icon' => '🌱'];
        }
        
        if ($this->is_gluten_free) {
            $badges[] = ['text' => 'خالي من الجلوتين', 'class' => 'bg-info', 'icon' => '🚫'];
        }
        
        if ($this->is_spicy) {
            $badges[] = ['text' => 'حار', 'class' => 'bg-danger', 'icon' => '🌶️'];
        }
        
        return $badges;
    }
}
