<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class PaymentVoucher extends Model
{
    protected $fillable = [
        'voucher_number',
        'company_id',
        'supplier_name',
        'supplier_id',
        'user_id',
        'payment_date',
        'amount',
        'payment_method',
        'reference_number',
        'notes',
        'invoice_reference',
        'account_code',
        'description',
        'status',
        'beneficiary_name',
        'beneficiary_account'
    ];

    protected $casts = [
        'payment_date' => 'datetime',
        'amount' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($voucher) {
            if (empty($voucher->voucher_number)) {
                $voucher->voucher_number = $voucher->generateVoucherNumber();
            }
        });
    }

    /**
     * علاقة مع الشركة
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع المستخدم
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * علاقة مع المورد
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class, 'supplier_id');
    }

    /**
     * Get the payment allocations for this voucher.
     */
    public function paymentAllocations(): HasMany
    {
        return $this->hasMany(SupplierPaymentAllocation::class);
    }

    /**
     * Get the purchase invoices paid by this voucher.
     */
    public function paidInvoices()
    {
        return $this->hasManyThrough(
            PurchaseInvoice::class,
            SupplierPaymentAllocation::class,
            'payment_voucher_id',
            'id',
            'id',
            'purchase_invoice_id'
        );
    }

    /**
     * توليد رقم سند الصرف
     */
    public function generateVoucherNumber(): string
    {
        $year = now()->year;
        $month = now()->format('m');
        
        $count = static::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->count() + 1;
            
        return sprintf('PV-%d%s-%06d', $year, $month, $count);
    }

    /**
     * الحصول على نص طريقة الدفع بالعربية
     */
    public function getPaymentMethodText(): string
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
            'online_transfer' => 'تحويل إلكتروني',
            'other' => 'أخرى'
        ];
        
        return $methods[$this->payment_method] ?? 'غير محدد';
    }

    /**
     * الحصول على نص الحالة بالعربية
     */
    public function getStatusText(): string
    {
        $statuses = [
            'active' => 'نشط',
            'cancelled' => 'ملغي',
            'pending' => 'معلق'
        ];
        
        return $statuses[$this->status] ?? 'غير محدد';
    }

    /**
     * الحصول على لون الحالة للعرض
     */
    public function getStatusColor(): string
    {
        $colors = [
            'active' => 'success',
            'cancelled' => 'danger',
            'pending' => 'warning'
        ];
        
        return $colors[$this->status] ?? 'secondary';
    }

    /**
     * فلترة السندات حسب الحالة
     */
    public function scopeOfStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * السندات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * السندات الملغية
     */
    public function scopeCancelled($query)
    {
        return $query->where('status', 'cancelled');
    }

    /**
     * فلترة السندات حسب طريقة الدفع
     */
    public function scopeOfPaymentMethod($query, $method)
    {
        return $query->where('payment_method', $method);
    }

    /**
     * فلترة السندات حسب الفترة الزمنية
     */
    public function scopeForPeriod($query, $startDate, $endDate)
    {
        return $query->whereBetween('payment_date', [$startDate, $endDate]);
    }

    /**
     * الحصول على اسم المستفيد أو المورد
     */
    public function getBeneficiaryNameAttribute($value)
    {
        return $value ?: ($this->supplier ? $this->supplier->name : $this->supplier_name);
    }

    /**
     * Get total allocated amount.
     */
    public function getTotalAllocatedAttribute(): float
    {
        return $this->paymentAllocations->sum('allocated_amount');
    }

    /**
     * Get unallocated amount.
     */
    public function getUnallocatedAmountAttribute(): float
    {
        return $this->amount - $this->total_allocated;
    }

    /**
     * Check if payment is fully allocated.
     */
    public function getIsFullyAllocatedAttribute(): bool
    {
        return $this->unallocated_amount <= 0;
    }

    /**
     * Check if payment can be deleted (within 24 hours).
     */
    public function canBeDeleted(): bool
    {
        return $this->created_at->diffInHours(now()) <= 24;
    }

    /**
     * Allocate payment to invoices automatically (oldest first).
     */
    public function allocateAutomatically(): void
    {
        if (!$this->supplier) return;

        $remainingAmount = $this->amount;
        
        $invoices = $this->supplier->purchaseInvoices()
            ->where('payment_status', '!=', 'paid')
            ->where('status', '!=', 'cancelled')
            ->where('remaining_amount', '>', 0)
            ->orderBy('due_date')
            ->get();

        foreach ($invoices as $invoice) {
            if ($remainingAmount <= 0) break;

            $allocationAmount = min($remainingAmount, $invoice->remaining_amount);
            
            SupplierPaymentAllocation::create([
                'payment_voucher_id' => $this->id,
                'purchase_invoice_id' => $invoice->id,
                'allocated_amount' => $allocationAmount,
                'notes' => 'توزيع تلقائي',
            ]);

            $remainingAmount -= $allocationAmount;
        }
    }

    /**
     * Allocate payment manually to specific invoices.
     */
    public function allocateManually(array $allocations): void
    {
        foreach ($allocations as $allocation) {
            if (isset($allocation['invoice_id'], $allocation['amount']) && $allocation['amount'] > 0) {
                SupplierPaymentAllocation::create([
                    'payment_voucher_id' => $this->id,
                    'purchase_invoice_id' => $allocation['invoice_id'],
                    'allocated_amount' => $allocation['amount'],
                    'notes' => $allocation['notes'] ?? 'توزيع يدوي',
                ]);
            }
        }
    }
}