<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class JournalEntry extends Model
{
    use HasFactory;

    protected $fillable = [
        'entry_number',
        'entry_date',
        'reference',
        'description',
        'type',
        'status',
        'total_debit',
        'total_credit',
        'created_by',
        'posted_by',
        'posted_at',
        'reversed_by',
        'reversed_at',
        'reversal_reason',
        'attachments'
    ];

    protected $casts = [
        'entry_date' => 'date',
        'total_debit' => 'decimal:2',
        'total_credit' => 'decimal:2',
        'posted_at' => 'datetime',
        'reversed_at' => 'datetime',
        'attachments' => 'array'
    ];

    // Relations
    public function lines(): HasMany
    {
        return $this->hasMany(JournalEntryLine::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function poster(): BelongsTo
    {
        return $this->belongsTo(User::class, 'posted_by');
    }

    public function reverser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reversed_by');
    }

    public function cashTransactions(): HasMany
    {
        return $this->hasMany(CashTransaction::class);
    }

    public function bankTransactions(): HasMany
    {
        return $this->hasMany(BankTransaction::class);
    }

    // Scopes
    public function scopePosted($query)
    {
        return $query->where('status', 'posted');
    }

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeByPeriod($query, $startDate, $endDate)
    {
        return $query->whereBetween('entry_date', [$startDate, $endDate]);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    // Helper methods
    public function isBalanced()
    {
        return abs($this->total_debit - $this->total_credit) < 0.01;
    }

    public function post($userId)
    {
        if ($this->status !== 'draft') {
            throw new \Exception('Only draft entries can be posted');
        }

        if (!$this->isBalanced()) {
            throw new \Exception('Journal entry is not balanced');
        }

        // Update account balances
        foreach ($this->lines as $line) {
            if ($line->debit_amount > 0) {
                $line->account->updateBalance($line->debit_amount, 'debit');
            }
            if ($line->credit_amount > 0) {
                $line->account->updateBalance($line->credit_amount, 'credit');
            }
        }

        $this->update([
            'status' => 'posted',
            'posted_by' => $userId,
            'posted_at' => now()
        ]);
    }

    public function reverse($userId, $reason)
    {
        if ($this->status !== 'posted') {
            throw new \Exception('Only posted entries can be reversed');
        }

        // Reverse account balances
        foreach ($this->lines as $line) {
            if ($line->debit_amount > 0) {
                $line->account->updateBalance($line->debit_amount, 'credit');
            }
            if ($line->credit_amount > 0) {
                $line->account->updateBalance($line->credit_amount, 'debit');
            }
        }

        $this->update([
            'status' => 'reversed',
            'reversed_by' => $userId,
            'reversed_at' => now(),
            'reversal_reason' => $reason
        ]);
    }

    public static function generateEntryNumber()
    {
        $lastEntry = self::whereYear('created_at', date('Y'))
            ->orderBy('id', 'desc')
            ->first();

        $nextNumber = $lastEntry ? (int)substr($lastEntry->entry_number, -4) + 1 : 1;
        
        return 'JE' . date('Y') . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    public function calculateTotals()
    {
        $this->total_debit = $this->lines()->sum('debit_amount');
        $this->total_credit = $this->lines()->sum('credit_amount');
        $this->save();
    }
}