<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Item extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'name_en',
        'description',
        'barcode',
        'unit',
        'item_type',
        'category',
        'brand',
        'model',
        'cost_price',
        'selling_price',
        'minimum_stock',
        'maximum_stock',
        'current_stock',
        'reserved_stock',
        'location',
        'supplier_code',
        'weight',
        'weight_unit',
        'dimensions',
        'color',
        'size',
        'discount_rate',
        'tax_rate',
        'image_path',
        'additional_info',
        'is_active',
        'is_stockable',
        'is_saleable',
        'is_purchasable',
    ];

    protected $casts = [
        'cost_price' => 'decimal:2',
        'selling_price' => 'decimal:2',
        'minimum_stock' => 'decimal:2',
        'maximum_stock' => 'decimal:2',
        'current_stock' => 'decimal:2',
        'reserved_stock' => 'decimal:2',
        'weight' => 'decimal:3',
        'discount_rate' => 'decimal:2',
        'tax_rate' => 'decimal:2',
        'additional_info' => 'array',
        'is_active' => 'boolean',
        'is_stockable' => 'boolean',
        'is_saleable' => 'boolean',
        'is_purchasable' => 'boolean',
    ];

    /**
     * Get all sales invoice details for this item.
     */
    public function salesInvoiceDetails(): HasMany
    {
        return $this->hasMany(SalesInvoiceDetail::class);
    }

    /**
     * Get all purchase invoice details for this item.
     */
    public function purchaseInvoiceDetails(): HasMany
    {
        return $this->hasMany(PurchaseInvoiceDetail::class);
    }

    /**
     * Get all inventory movements for this item.
     */
    public function inventoryMovements(): HasMany
    {
        return $this->hasMany(InventoryMovement::class);
    }

    /**
     * Get active items only.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get stockable items only.
     */
    public function scopeStockable($query)
    {
        return $query->where('is_stockable', true);
    }

    /**
     * Get saleable items only.
     */
    public function scopeSaleable($query)
    {
        return $query->where('is_saleable', true);
    }

    /**
     * Get purchasable items only.
     */
    public function scopePurchasable($query)
    {
        return $query->where('is_purchasable', true);
    }

    /**
     * Filter by item type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('item_type', $type);
    }

    /**
     * Filter by category.
     */
    public function scopeOfCategory($query, string $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Get items with low stock.
     */
    public function scopeLowStock($query)
    {
        return $query->where('is_stockable', true)
                    ->whereColumn('current_stock', '<=', 'minimum_stock');
    }

    /**
     * Get items with overstock.
     */
    public function scopeOverstock($query)
    {
        return $query->where('is_stockable', true)
                    ->where('maximum_stock', '>', 0)
                    ->whereColumn('current_stock', '>=', 'maximum_stock');
    }

    /**
     * Generate next item code.
     */
    public static function generateNextCode(): string
    {
        $lastItem = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastItem ? (int)substr($lastItem->code, 4) + 1 : 1;
        return 'ITEM' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }

    /**
     * Get available stock (current - reserved).
     */
    public function getAvailableStockAttribute(): float
    {
        return max(0, $this->current_stock - $this->reserved_stock);
    }

    /**
     * Get stock status.
     */
    public function getStockStatusAttribute(): string
    {
        if (!$this->is_stockable) {
            return 'not_tracked';
        }

        if ($this->current_stock <= 0) {
            return 'out_of_stock';
        } elseif ($this->current_stock <= $this->minimum_stock) {
            return 'low_stock';
        } elseif ($this->maximum_stock > 0 && $this->current_stock >= $this->maximum_stock) {
            return 'overstock';
        }

        return 'in_stock';
    }

    /**
     * Get profit margin percentage.
     */
    public function getProfitMarginAttribute(): float
    {
        if ($this->cost_price <= 0) {
            return 0;
        }

        return (($this->selling_price - $this->cost_price) / $this->cost_price) * 100;
    }

    /**
     * Update stock quantity.
     */
    public function updateStock(float $quantity, string $operation = 'add'): void
    {
        $oldStock = $this->current_stock;

        if ($operation === 'add') {
            $this->current_stock += $quantity;
        } elseif ($operation === 'subtract') {
            $this->current_stock = max(0, $this->current_stock - $quantity);
        } else {
            $this->current_stock = max(0, $quantity);
        }

        $this->save();
    }

    /**
     * Reserve stock quantity.
     */
    public function reserveStock(float $quantity): bool
    {
        if ($this->available_stock < $quantity) {
            return false;
        }

        $this->reserved_stock += $quantity;
        $this->save();

        return true;
    }

    /**
     * Release reserved stock.
     */
    public function releaseReservedStock(float $quantity): void
    {
        $this->reserved_stock = max(0, $this->reserved_stock - $quantity);
        $this->save();
    }

    /**
     * Get total sales quantity this year.
     */
    public function getTotalSalesQuantityThisYear(): float
    {
        return $this->salesInvoiceDetails()
            ->whereHas('salesInvoice', function($query) {
                $query->whereYear('invoice_date', date('Y'))
                      ->where('status', '!=', 'cancelled');
            })
            ->sum('quantity');
    }

    /**
     * Get total sales amount this year.
     */
    public function getTotalSalesAmountThisYear(): float
    {
        return $this->salesInvoiceDetails()
            ->whereHas('salesInvoice', function($query) {
                $query->whereYear('invoice_date', date('Y'))
                      ->where('status', '!=', 'cancelled');
            })
            ->sum('total_amount');
    }

    /**
     * Get total purchase quantity this year.
     */
    public function getTotalPurchaseQuantityThisYear(): float
    {
        return $this->purchaseInvoiceDetails()
            ->whereHas('purchaseInvoice', function($query) {
                $query->whereYear('invoice_date', date('Y'))
                      ->where('status', '!=', 'cancelled');
            })
            ->sum('quantity');
    }

    /**
     * Get average cost price based on recent purchases.
     */
    public function getAverageCostPrice(): float
    {
        $recentPurchases = $this->purchaseInvoiceDetails()
            ->whereHas('purchaseInvoice', function($query) {
                $query->where('status', '!=', 'cancelled')
                      ->where('invoice_date', '>=', now()->subMonths(6));
            })
            ->get();

        if ($recentPurchases->isEmpty()) {
            return $this->cost_price;
        }

        $totalQuantity = $recentPurchases->sum('quantity');
        $totalCost = $recentPurchases->sum(function($detail) {
            return $detail->quantity * $detail->unit_price;
        });

        return $totalQuantity > 0 ? $totalCost / $totalQuantity : $this->cost_price;
    }

    /**
     * Get turnover ratio (sales / average stock).
     */
    public function getTurnoverRatio(): float
    {
        $salesAmount = $this->getTotalSalesAmountThisYear();
        $averageStock = ($this->current_stock * $this->cost_price);

        return $averageStock > 0 ? $salesAmount / $averageStock : 0;
    }

    /**
     * Check if item needs reorder.
     */
    public function needsReorder(): bool
    {
        return $this->is_stockable && 
               $this->minimum_stock > 0 && 
               $this->available_stock <= $this->minimum_stock;
    }
}