<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InvoiceItem extends Model
{
    protected $fillable = [
        'invoice_id',
        'product_id',
        'account_id',
        'name',
        'name_en',
        'description',
        'sku',
        'quantity',
        'unit',
        'unit_price',
        'line_total',
        'discount_amount',
        'discount_percentage',
        'tax_rate',
        'tax_amount',
        'total_with_tax',
        'zatca_category_code',
        'is_taxable',
    ];

    protected $casts = [
        'quantity' => 'decimal:3',
        'unit_price' => 'decimal:2',
        'line_total' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'discount_percentage' => 'decimal:2',
        'tax_rate' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_with_tax' => 'decimal:2',
        'is_taxable' => 'boolean',
    ];

    /**
     * علاقة مع الفاتورة
     */
    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * علاقة مع المنتج
     */
    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * علاقة مع الحساب (إذا كان محدد لهذا العنصر)
     */
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    /**
     * حساب إجمالي السطر بدون ضريبة
     */
    public function calculateLineTotal(): float
    {
        return $this->quantity * $this->unit_price - $this->discount_amount;
    }

    /**
     * حساب مبلغ الضريبة بناءً على طريقة حساب الحساب
     */
    public function calculateTaxAmount(): float
    {
        if (!$this->is_taxable) {
            return 0;
        }

        // التحقق من وجود حساب مرتبط وطريقة حساب الضريبة
        if ($this->account && $this->account->applies_tax) {
            $taxCalculation = $this->account->calculateTaxAmount($this->line_total, $this->tax_rate);
            return $taxCalculation['tax_amount'];
        }

        // الطريقة التقليدية (غير شامل الضريبة)
        return $this->line_total * ($this->tax_rate / 100);
    }

    /**
     * حساب الإجمالي مع الضريبة بناءً على طريقة الحساب
     */
    public function calculateTotalWithTax(): float
    {
        if ($this->account && $this->account->isTaxInclusive()) {
            // إذا كان الحساب شامل الضريبة، فالسعر المدخل يتضمن الضريبة
            return $this->line_total; // السعر شامل الضريبة
        }

        // الطريقة التقليدية (إضافة الضريبة)
        return $this->line_total + $this->tax_amount;
    }

    /**
     * حساب المبلغ بدون ضريبة من المبلغ الشامل
     */
    public function calculateExclusiveAmount(): float
    {
        if ($this->account && $this->account->isTaxInclusive()) {
            return $this->account->convertInclusiveToExclusive($this->line_total, $this->tax_rate);
        }

        return $this->line_total;
    }

    /**
     * حساب المبلغ الشامل للضريبة
     */
    public function calculateInclusiveAmount(): float
    {
        if ($this->account && $this->account->isTaxExclusive()) {
            return $this->account->convertExclusiveToInclusive($this->line_total, $this->tax_rate);
        }

        return $this->line_total;
    }

    /**
     * حساب نسبة الخصم
     */
    public function calculateDiscountPercentage(): float
    {
        $subtotal = $this->quantity * $this->unit_price;
        if ($subtotal > 0) {
            return ($this->discount_amount / $subtotal) * 100;
        }
        return 0;
    }

    /**
     * تحديث الحسابات بناءً على نوع الحساب
     */
    public function updateCalculations()
    {
        // حساب إجمالي السطر الأساسي
        $baseAmount = $this->quantity * $this->unit_price - $this->discount_amount;

        if ($this->account && $this->account->applies_tax) {
            if ($this->account->isTaxInclusive()) {
                // إذا كان الحساب شامل الضريبة
                $this->line_total = $baseAmount; // المبلغ شامل الضريبة
                $exclusiveAmount = $this->account->convertInclusiveToExclusive($baseAmount, $this->tax_rate);
                $this->tax_amount = $baseAmount - $exclusiveAmount;
                $this->total_with_tax = $baseAmount;
            } else {
                // إذا كان الحساب غير شامل الضريبة
                $this->line_total = $baseAmount; // المبلغ بدون ضريبة
                $this->tax_amount = $baseAmount * ($this->tax_rate / 100);
                $this->total_with_tax = $baseAmount + $this->tax_amount;
            }
        } else {
            // الطريقة التقليدية
            $this->line_total = $baseAmount;
            $this->tax_amount = $this->is_taxable ? ($baseAmount * ($this->tax_rate / 100)) : 0;
            $this->total_with_tax = $baseAmount + $this->tax_amount;
        }

        $this->discount_percentage = $this->calculateDiscountPercentage();
    }

    /**
     * الحصول على تفاصيل حساب الضريبة
     */
    public function getTaxCalculationDetails(): array
    {
        if (!$this->is_taxable) {
            return [
                'method' => 'none',
                'rate' => 0,
                'base_amount' => $this->line_total,
                'tax_amount' => 0,
                'total_amount' => $this->line_total
            ];
        }

        if ($this->account && $this->account->applies_tax) {
            $calculation = $this->account->calculateTaxAmount($this->line_total, $this->tax_rate);
            return [
                'method' => $this->account->tax_calculation_method,
                'rate' => $this->tax_rate,
                'base_amount' => $calculation['tax_exclusive_amount'],
                'tax_amount' => $calculation['tax_amount'],
                'total_amount' => $calculation['tax_inclusive_amount'],
                'account_name' => $this->account->name
            ];
        }

        // الطريقة التقليدية
        return [
            'method' => 'exclusive',
            'rate' => $this->tax_rate,
            'base_amount' => $this->line_total,
            'tax_amount' => $this->tax_amount,
            'total_amount' => $this->total_with_tax
        ];
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        static::saving(function ($item) {
            $item->updateCalculations();
        });
    }
}
