<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Endroid\QrCode\QrCode;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\RoundBlockSizeMode;

class Invoice extends Model
{
    protected $fillable = [
        'invoice_number',
        'type',
        'company_id',
        'customer_id',
        'user_id',
        'issue_date',
        'due_date',
        'supply_date',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'currency',
        'payment_method',
        'paid_amount',
        'remaining_amount',
        'payment_status',
        'payment_date',
        'creditor_notification_sent',
        'creditor_notification_sent_at',
        'debtor_notification_sent',
        'debtor_notification_sent_at',
        'payment_notes',
        'uuid',
        'invoice_hash',
        'previous_invoice_hash',
        'invoice_counter',
        'qr_code',
        'irn',
        'status',
        'zatca_response',
        'sent_to_zatca_at',
        'xml_file_path',
        'pdf_file_path',
        'notes',
        // ZATCA Phase 2 Business Operation Type fields
        'is_third_party',
        'is_nominal',
        'is_export',
        'is_summary',
        'is_self_billed',
        'credit_debit_reason',
    ];

    protected $casts = [
        'issue_date' => 'datetime',
        'due_date' => 'datetime',
        'supply_date' => 'datetime',
        'payment_date' => 'datetime',
        'creditor_notification_sent_at' => 'datetime',
        'debtor_notification_sent_at' => 'datetime',
        'sent_to_zatca_at' => 'datetime',
        'subtotal' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'paid_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
        'creditor_notification_sent' => 'boolean',
        'debtor_notification_sent' => 'boolean',
        'zatca_response' => 'json',
        // ZATCA Phase 2 Business Operation Type casts
        'is_third_party' => 'boolean',
        'is_nominal' => 'boolean',
        'is_export' => 'boolean',
        'is_summary' => 'boolean',
        'is_self_billed' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($invoice) {
            if (empty($invoice->uuid)) {
                $invoice->uuid = Str::uuid();
            }
        });
    }

    /**
     * علاقة مع الشركة
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع العميل
     */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * علاقة مع المستخدم
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * علاقة مع عناصر الفاتورة
     */
    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    /**
     * علاقة مع الحركات النقدية
     */
    public function cashTransactions(): HasMany
    {
        return $this->hasMany(CashTransaction::class);
    }

    /**
     * فلترة الفواتير حسب النوع
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * الفواتير المعيارية
     */
    public function scopeStandard($query)
    {
        return $query->where('type', 'standard');
    }

    /**
     * الفواتير المبسطة
     */
    public function scopeSimplified($query)
    {
        return $query->where('type', 'simplified');
    }

    /**
     * فلترة الفواتير حسب الحالة
     */
    public function scopeOfStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * الفواتير المسودة
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    /**
     * الفواتير المرسلة
     */
    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    /**
     * التحقق من كون الفاتورة معيارية
     */
    public function isStandard(): bool
    {
        return $this->type === 'standard';
    }

    /**
     * التحقق من كون الفاتورة مبسطة
     */
    public function isSimplified(): bool
    {
        return $this->type === 'simplified';
    }

    /**
     * التحقق من إرسال الفاتورة لـ ZATCA
     */
    public function isSentToZatca(): bool
    {
        return !is_null($this->sent_to_zatca_at);
    }

    /**
     * حساب المجاميع
     */
    public function calculateTotals()
    {
        $this->subtotal = $this->items->sum('line_total');
        $this->tax_amount = $this->items->sum('tax_amount');
        $this->total_amount = $this->subtotal + $this->tax_amount - $this->discount_amount;
        $this->remaining_amount = $this->total_amount - $this->paid_amount;
        $this->updatePaymentStatus();
    }

    /**
     * تحديث حالة السداد
     */
    public function updatePaymentStatus()
    {
        if ($this->paid_amount <= 0) {
            $this->payment_status = 'unpaid';
        } elseif ($this->paid_amount >= $this->total_amount) {
            $this->payment_status = 'paid';
            $this->remaining_amount = 0;
        } else {
            $this->payment_status = 'partial';
        }

        // تحديد حالة التأخير
        if ($this->payment_status !== 'paid' && $this->due_date && $this->due_date->isPast()) {
            $this->payment_status = 'overdue';
        }
    }

    /**
     * إضافة دفعة جديدة
     */
    public function addPayment(float $amount, string $method = null, string $notes = null)
    {
        $this->paid_amount += $amount;
        $this->remaining_amount = $this->total_amount - $this->paid_amount;

        if ($method) {
            $this->payment_method = $method;
        }

        if ($notes) {
            $this->payment_notes = $notes;
        }

        if (!$this->payment_date && $amount > 0) {
            $this->payment_date = now();
        }

        $this->updatePaymentStatus();
        $this->save();
    }

    /**
     * التحقق من كون الفاتورة مدفوعة بالكامل
     */
    public function isPaid(): bool
    {
        return $this->payment_status === 'paid';
    }

    /**
     * التحقق من كون الفاتورة مدفوعة جزئياً
     */
    public function isPartiallyPaid(): bool
    {
        return $this->payment_status === 'partial';
    }

    /**
     * التحقق من كون الفاتورة متأخرة
     */
    public function isOverdue(): bool
    {
        return $this->payment_status === 'overdue';
    }

    /**
     * إرسال إشعار للدائن
     */
    public function sendCreditorNotification(): bool
    {
        // هنا يمكن إضافة منطق إرسال الإشعار (إيميل، SMS، إلخ)
        $this->creditor_notification_sent = true;
        $this->creditor_notification_sent_at = now();
        return $this->save();
    }

    /**
     * إرسال إشعار للمدين
     */
    public function sendDebtorNotification(): bool
    {
        // هنا يمكن إضافة منطق إرسال الإشعار (إيميل، SMS، إلخ)
        $this->debtor_notification_sent = true;
        $this->debtor_notification_sent_at = now();
        return $this->save();
    }

    /**
     * فلترة الفواتير حسب حالة السداد
     */
    public function scopeOfPaymentStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    /**
     * الفواتير غير المدفوعة
     */
    public function scopeUnpaid($query)
    {
        return $query->where('payment_status', 'unpaid');
    }

    /**
     * الفواتير المدفوعة جزئياً
     */
    public function scopePartiallyPaid($query)
    {
        return $query->where('payment_status', 'partial');
    }

    /**
     * الفواتير المدفوعة بالكامل
     */
    public function scopePaid($query)
    {
        return $query->where('payment_status', 'paid');
    }

    /**
     * الفواتير المتأخرة
     */
    public function scopeOverdue($query)
    {
        return $query->where('payment_status', 'overdue');
    }

    /**
     * الحصول على نسبة السداد
     */
    public function getPaymentPercentage(): float
    {
        if ($this->total_amount <= 0) {
            return 0;
        }

        return round(($this->paid_amount / $this->total_amount) * 100, 2);
    }

    /**
     * الحصول على نص طريقة السداد بالعربية
     */
    public function getPaymentMethodText(): string
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
            'installment' => 'تقسيط'
        ];

        return $methods[$this->payment_method] ?? 'غير محدد';
    }

    /**
     * الحصول على نص حالة السداد بالعربية
     */
    public function getPaymentStatusText(): string
    {
        $statuses = [
            'unpaid' => 'غير مدفوعة',
            'partial' => 'مدفوعة جزئياً',
            'paid' => 'مدفوعة بالكامل',
            'overdue' => 'متأخرة'
        ];

        return $statuses[$this->payment_status] ?? 'غير محدد';
    }

    /**
     * الحصول على لون حالة السداد للعرض
     */
    public function getPaymentStatusColor(): string
    {
        $colors = [
            'unpaid' => 'danger',
            'partial' => 'warning',
            'paid' => 'success',
            'overdue' => 'dark'
        ];

        return $colors[$this->payment_status] ?? 'secondary';
    }

    /**
     * توليد رقم الفاتورة
     */
    public function generateInvoiceNumber(): string
    {
        $prefix = $this->isStandard() ? 'INV' : 'SIM';
        $year = now()->year;
        $month = now()->format('m');

        $count = static::where('type', $this->type)
            ->whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->count() + 1;

        return sprintf('%s-%d%s-%06d', $prefix, $year, $month, $count);
    }

    /**
     * توليد QR Code للفاتورة
     */
    public function generateQrCode(): string
    {
        // استخدام ZatcaService لتوليد QR Code متوافق مع ZATCA
        try {
            $zatcaSettings = \App\Models\ZatcaSettings::first();
            if ($zatcaSettings) {
                $zatcaService = new \App\Services\ZatcaService($zatcaSettings);
                $qrContent = $zatcaService->generateQRCode($this);
            } else {
                // إذا لم توجد إعدادات ZATCA، استخدم التنسيق المحلي
                $qrContent = $this->buildZatcaQrContent();
            }

            // إنشاء QR Code باستخدام الصيغة الصحيحة لـ endroid/qr-code v6
            $qrCode = new QrCode(
                data: $qrContent,
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::Low,
                size: 300,
                margin: 10,
                roundBlockSizeMode: RoundBlockSizeMode::Margin
            );

            // تحويل إلى PNG
            $writer = new PngWriter();
            $result = $writer->write($qrCode);

            // تحويل إلى base64
            return base64_encode($result->getString());
        } catch (\Exception $e) {
            // في حالة حدوث خطأ، استخدم التنسيق الاحتياطي
            \Log::error('QR Code generation failed: ' . $e->getMessage());
            return $this->generateFallbackQrCode();
        }
    }

    /**
     * بناء محتوى QR Code حسب متطلبات ZATCA Phase 2
     */
    private function buildZatcaQrContent(): string
    {
        // تطبيق معايير ZATCA Phase 2 لـ QR Code بتنسيق TLV (Tag-Length-Value)
        $qrData = [];

        // Tag 01: اسم البائع
        $qrData[1] = $this->company->name;

        // Tag 02: الرقم الضريبي للبائع
        $qrData[2] = $this->company->vat_number;

        // Tag 03: التاريخ والوقت (ISO 8601)
        $qrData[3] = $this->issue_date->format('Y-m-d\TH:i:s\Z');

        // Tag 04: إجمالي الفاتورة
        $qrData[4] = number_format($this->total_amount, 2, '.', '');

        // Tag 05: مبلغ الضريبة
        $qrData[5] = number_format($this->tax_amount, 2, '.', '');

        // Phase 2: إضافة الحقول المطلوبة للفواتير المرسلة لـ ZATCA
        if ($this->sent_to_zatca_at && $this->invoice_hash) {
            // Tag 06: Invoice Hash (PIH - Previous Invoice Hash أو Current Invoice Hash)
            $qrData[6] = $this->invoice_hash;

            // Tag 07: ECDSA Signature (التوقيع الرقمي)
            if (isset($this->zatca_response['digitalSignature'])) {
                $qrData[7] = $this->zatca_response['digitalSignature'];
            } elseif (isset($this->zatca_response['signature'])) {
                $qrData[7] = $this->zatca_response['signature'];
            }

            // Tag 08: ECDSA Public Key (المفتاح العام)
            if (isset($this->zatca_response['publicKey'])) {
                $qrData[8] = $this->zatca_response['publicKey'];
            }

            // Tag 09: Cryptographic Stamp (الطابع التشفيري)
            if (isset($this->zatca_response['cryptographicStamp'])) {
                $qrData[9] = $this->zatca_response['cryptographicStamp'];
            } elseif (isset($this->zatca_response['clearedInvoice']['uuid'])) {
                // استخدام UUID المطهر كطابع تشفيري بديل
                $qrData[9] = $this->zatca_response['clearedInvoice']['uuid'];
            }
        } else {
            // Phase 1: للفواتير غير المرسلة، استخدام قيم افتراضية أو حذف الحقول
            // يمكن إضافة hash محلي إذا كان متوفراً
            if ($this->invoice_hash) {
                $qrData[6] = $this->invoice_hash;
            }
        }

        // بناء TLV Data
        $tlvData = '';
        foreach ($qrData as $tag => $value) {
            if (!empty($value)) {
                $tlvData .= $this->formatQrField($tag, $value);
            }
        }

        return base64_encode($tlvData);
    }

    /**
     * تنسيق حقل QR Code حسب معيار ZATCA
     */
    private function formatQrField(int $tag, string $value): string
    {
        // استخدام طول البايت بدلاً من طول السلسلة لدعم UTF-8
        $valueLength = strlen(utf8_encode($value));

        // التأكد من أن الطول لا يتجاوز 255 بايت
        if ($valueLength > 255) {
            $value = substr($value, 0, 255);
            $valueLength = 255;
        }

        return chr($tag) . chr($valueLength) . $value;
    }

    /**
     * حفظ QR Code في قاعدة البيانات
     */
    public function saveQrCode(): void
    {
        $this->qr_code = $this->generateQrCode();
        $this->save();
    }

    /**
     * الحصول على QR Code كـ data URL
     */
    public function getQrCodeDataUrl(): ?string
    {
        if (!$this->qr_code) {
            return null;
        }

        if($this->sent_to_zatca_at){
            $qrCode = new QrCode(
                data: $this->qr_code,
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::Low,
                size: 300,
                margin: 10,
                roundBlockSizeMode: RoundBlockSizeMode::Margin
            );

            // تحويل إلى PNG
            $writer = new PngWriter();
            $result = $writer->write($qrCode);

            // تحويل إلى base64
            return 'data:image/png;base64,' . base64_encode($result->getString());
        }

        return 'data:image/png;base64,' . $this->qr_code;
    }

    /**
     * توليد QR Code احتياطي في حالة فشل الطريقة الأساسية
     */
    private function generateFallbackQrCode(): string
    {
        try {
            $qrContent = $this->buildZatcaQrContent();

            $qrCode = new QrCode(
                data: $qrContent,
                encoding: new Encoding('UTF-8'),
                errorCorrectionLevel: ErrorCorrectionLevel::Low,
                size: 300,
                margin: 10,
                roundBlockSizeMode: RoundBlockSizeMode::Margin
            );

            $writer = new PngWriter();
            $result = $writer->write($qrCode);

            return base64_encode($result->getString());
        } catch (\Exception $e) {
            // في حالة فشل كل شيء، إرجاع قيمة فارغة أو رابط بسيط
            \Log::error('Fallback QR Code generation also failed: ' . $e->getMessage());
            return base64_encode('Error generating QR code');
        }
    }

    /**
     * توليد رابط للتحقق من الفاتورة
     */
    public function getVerificationUrl(): string
    {
        return route('invoices.verify', ['uuid' => $this->uuid]);
    }

    /**
     * تحديث QR Code للتوافق مع Phase 2
     */
    public function updateQrCodeForPhase2(): bool
    {
        try {
            // إعادة توليد QR Code بالتنسيق المحدث
            $this->qr_code = $this->generateQrCode();
            $this->save();

            \Log::info("QR Code updated for Phase 2 compliance", [
                'invoice_id' => $this->id,
                'invoice_number' => $this->invoice_number,
                'sent_to_zatca' => $this->sent_to_zatca_at ? true : false
            ]);

            return true;
        } catch (\Exception $e) {
            \Log::error("Failed to update QR Code for Phase 2", [
                'invoice_id' => $this->id,
                'error' => $e->getMessage()
            ]);

            return false;
        }
    }

    /**
     * التحقق من توافق QR Code مع Phase 2
     */
    public function isQrCodePhase2Compatible(): bool
    {
        // إذا كانت الفاتورة مرسلة لـ ZATCA وتحتوي على البيانات المطلوبة
        return $this->sent_to_zatca_at
            && $this->invoice_hash
            && $this->zatca_response
            && (
                isset($this->zatca_response['digitalSignature'])
                || isset($this->zatca_response['signature'])
            );
    }

    /**
     * الحصول على معلومات QR Code
     */
    public function getQrCodeInfo(): array
    {
        return [
            'phase' => $this->isQrCodePhase2Compatible() ? 2 : 1,
            'contains_signature' => $this->zatca_response && (
                isset($this->zatca_response['digitalSignature']) ||
                isset($this->zatca_response['signature'])
            ),
            'contains_public_key' => $this->zatca_response && isset($this->zatca_response['publicKey']),
            'contains_cryptographic_stamp' => $this->zatca_response && (
                isset($this->zatca_response['cryptographicStamp']) ||
                isset($this->zatca_response['clearedInvoice']['uuid'])
            ),
            'sent_to_zatca' => $this->sent_to_zatca_at ? true : false,
            'has_hash' => !empty($this->invoice_hash),
        ];
    }

    /**
     * الحصول على إجمالي المقبوضات النقدية
     */
    public function getTotalCashReceived(): float
    {
        return $this->cashTransactions()
            ->where('type', 'receipt')
            ->where('payment_method', 'cash')
            ->sum('amount');
    }

    /**
     * الحصول على إجمالي المقبوضات من جميع الطرق
     */
    public function getTotalReceived(): float
    {
        return $this->cashTransactions()
            ->where('type', 'receipt')
            ->sum('amount');
    }

    /**
     * التحقق من وجود حركات نقدية مرتبطة
     */
    public function hasCashTransactions(): bool
    {
        return $this->cashTransactions()->exists();
    }

    /**
     * الحصول على آخر دفعة نقدية
     */
    public function getLastCashTransaction()
    {
        return $this->cashTransactions()
            ->where('type', 'receipt')
            ->orderBy('created_at', 'desc')
            ->first();
    }

    /**
     * scope للفواتير التي لها حركات نقدية
     */
    public function scopeWithCashTransactions($query)
    {
        return $query->whereHas('cashTransactions');
    }

    /**
     * scope للفواتير المدفوعة نقداً
     */
    public function scopePaidInCash($query)
    {
        return $query->whereHas('cashTransactions', function($q) {
            $q->where('payment_method', 'cash');
        });
    }

    /**
     * تحليل الدفع حسب الطريقة
     */
    public function getPaymentBreakdown(): array
    {
        $transactions = $this->cashTransactions()
            ->where('type', 'receipt')
            ->get()
            ->groupBy('payment_method');

        $breakdown = [];
        foreach ($transactions as $method => $methodTransactions) {
            $breakdown[$method] = [
                'count' => $methodTransactions->count(),
                'total' => $methodTransactions->sum('amount'),
                'percentage' => $this->total_amount > 0
                    ? round(($methodTransactions->sum('amount') / $this->total_amount) * 100, 2)
                    : 0
            ];
        }

        return $breakdown;
    }

    /**
     * إضافة طرق مساعدة لتحليل التدفق النقدي
     */
    public function getCashFlowSummary(): array
    {
        $cashTransactions = $this->cashTransactions;
        
        return [
            'total_transactions' => $cashTransactions->count(),
            'receipts_count' => $cashTransactions->where('type', 'receipt')->count(),
            'receipts_amount' => $cashTransactions->where('type', 'receipt')->sum('amount'),
            'cash_receipts' => $cashTransactions->where('type', 'receipt')
                ->where('payment_method', 'cash')->sum('amount'),
            'bank_receipts' => $cashTransactions->where('type', 'receipt')
                ->where('payment_method', 'bank_transfer')->sum('amount'),
            'card_receipts' => $cashTransactions->where('type', 'receipt')
                ->where('payment_method', 'credit_card')->sum('amount'),
            'check_receipts' => $cashTransactions->where('type', 'receipt')
                ->where('payment_method', 'check')->sum('amount'),
        ];
    }
}
