<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class InventoryMovement extends Model
{
    use HasFactory;

    protected $fillable = [
        'movement_number',
        'movement_date',
        'item_id',
        'movement_type',
        'quantity',
        'unit_cost',
        'total_cost',
        'from_location',
        'to_location',
        'reference_type',
        'reference_id',
        'cost_center_id',
        'notes',
        'batch_number',
        'expiry_date',
        'stock_before',
        'stock_after',
        'created_by',
    ];

    protected $casts = [
        'movement_date' => 'date',
        'quantity' => 'decimal:3',
        'unit_cost' => 'decimal:2',
        'total_cost' => 'decimal:2',
        'stock_before' => 'decimal:3',
        'stock_after' => 'decimal:3',
        'expiry_date' => 'date',
    ];

    /**
     * Get the item associated with the movement.
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Get the cost center associated with the movement.
     */
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    /**
     * Get the reference model (polymorphic relation).
     */
    public function reference()
    {
        if ($this->reference_type && $this->reference_id) {
            $modelClass = match ($this->reference_type) {
                'sales_invoice' => SalesInvoice::class,
                'purchase_invoice' => PurchaseInvoice::class,
                'journal_entry' => JournalEntry::class,
                default => null,
            };
            
            if ($modelClass) {
                return $modelClass::find($this->reference_id);
            }
        }
        
        return null;
    }

    /**
     * Filter by movement type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('movement_type', $type);
    }

    /**
     * Filter by item.
     */
    public function scopeForItem($query, int $itemId)
    {
        return $query->where('item_id', $itemId);
    }

    /**
     * Filter by date range.
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('movement_date', [$startDate, $endDate]);
    }

    /**
     * Get inbound movements (positive quantity).
     */
    public function scopeInbound($query)
    {
        return $query->where('quantity', '>', 0);
    }

    /**
     * Get outbound movements (negative quantity).
     */
    public function scopeOutbound($query)
    {
        return $query->where('quantity', '<', 0);
    }

    /**
     * Generate next movement number.
     */
    public static function generateNextMovementNumber(): string
    {
        $lastMovement = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastMovement ? (int)substr($lastMovement->movement_number, 3) + 1 : 1;
        return 'MOV' . str_pad($nextNumber, 7, '0', STR_PAD_LEFT);
    }

    /**
     * Get movement direction.
     */
    public function getDirectionAttribute(): string
    {
        return $this->quantity > 0 ? 'in' : 'out';
    }

    /**
     * Get absolute quantity.
     */
    public function getAbsoluteQuantityAttribute(): float
    {
        return abs($this->quantity);
    }

    /**
     * Get movement type display name.
     */
    public function getMovementTypeDisplayAttribute(): string
    {
        return match ($this->movement_type) {
            'purchase' => 'مشتريات',
            'sale' => 'مبيعات',
            'adjustment' => 'تسوية',
            'transfer' => 'تحويل',
            'production' => 'إنتاج',
            'consumption' => 'استهلاك',
            'return' => 'مرتجع',
            'damage' => 'تالف',
            'initial_stock' => 'رصيد افتتاحي',
            'physical_count' => 'جرد فعلي',
            default => $this->movement_type,
        };
    }

    /**
     * Check if movement is expired (for items with expiry date).
     */
    public function getIsExpiredAttribute(): bool
    {
        return $this->expiry_date && $this->expiry_date < now();
    }

    /**
     * Check if movement is near expiry.
     */
    public function getIsNearExpiryAttribute(): bool
    {
        if (!$this->expiry_date) {
            return false;
        }
        
        $daysToExpiry = now()->diffInDays($this->expiry_date, false);
        return $daysToExpiry <= 30 && $daysToExpiry >= 0;
    }

    /**
     * Get days to expiry.
     */
    public function getDaysToExpiryAttribute(): ?int
    {
        if (!$this->expiry_date) {
            return null;
        }
        
        return now()->diffInDays($this->expiry_date, false);
    }

    /**
     * Get stock change (stock_after - stock_before).
     */
    public function getStockChangeAttribute(): float
    {
        return $this->stock_after - $this->stock_before;
    }

    /**
     * Create adjustment movement.
     */
    public static function createAdjustment(
        int $itemId,
        float $newStock,
        ?int $costCenterId = null,
        ?string $notes = null
    ): self {
        $item = Item::findOrFail($itemId);
        $currentStock = $item->current_stock;
        $adjustment = $newStock - $currentStock;
        
        $movement = static::create([
            'movement_number' => static::generateNextMovementNumber(),
            'movement_date' => now(),
            'item_id' => $itemId,
            'movement_type' => 'adjustment',
            'quantity' => $adjustment,
            'unit_cost' => $item->cost_price,
            'total_cost' => $adjustment * $item->cost_price,
            'cost_center_id' => $costCenterId,
            'notes' => $notes ?: 'تسوية مخزون',
            'stock_before' => $currentStock,
            'stock_after' => $newStock,
            'created_by' => auth()->user()->name ?? 'النظام',
        ]);
        
        // Update item stock
        $item->update(['current_stock' => $newStock]);
        
        return $movement;
    }

    /**
     * Create transfer movement.
     */
    public static function createTransfer(
        int $itemId,
        float $quantity,
        string $fromLocation,
        string $toLocation,
        ?int $costCenterId = null,
        ?string $notes = null
    ): self {
        $item = Item::findOrFail($itemId);
        
        return static::create([
            'movement_number' => static::generateNextMovementNumber(),
            'movement_date' => now(),
            'item_id' => $itemId,
            'movement_type' => 'transfer',
            'quantity' => 0, // Transfer doesn't change total stock
            'unit_cost' => $item->cost_price,
            'total_cost' => 0,
            'from_location' => $fromLocation,
            'to_location' => $toLocation,
            'cost_center_id' => $costCenterId,
            'notes' => $notes ?: "تحويل من {$fromLocation} إلى {$toLocation}",
            'stock_before' => $item->current_stock,
            'stock_after' => $item->current_stock,
            'created_by' => auth()->user()->name ?? 'النظام',
        ]);
    }

    /**
     * Create physical count movement.
     */
    public static function createPhysicalCount(
        int $itemId,
        float $countedStock,
        ?int $costCenterId = null,
        ?string $notes = null
    ): self {
        $item = Item::findOrFail($itemId);
        $currentStock = $item->current_stock;
        $difference = $countedStock - $currentStock;
        
        $movement = static::create([
            'movement_number' => static::generateNextMovementNumber(),
            'movement_date' => now(),
            'item_id' => $itemId,
            'movement_type' => 'physical_count',
            'quantity' => $difference,
            'unit_cost' => $item->cost_price,
            'total_cost' => $difference * $item->cost_price,
            'cost_center_id' => $costCenterId,
            'notes' => $notes ?: "جرد فعلي - الفرق: {$difference}",
            'stock_before' => $currentStock,
            'stock_after' => $countedStock,
            'created_by' => auth()->user()->name ?? 'النظام',
        ]);
        
        // Update item stock
        $item->update(['current_stock' => $countedStock]);
        
        return $movement;
    }

    /**
     * Get movements summary for an item.
     */
    public static function getSummaryForItem(int $itemId, $startDate = null, $endDate = null): array
    {
        $query = static::where('item_id', $itemId);
        
        if ($startDate && $endDate) {
            $query->whereBetween('movement_date', [$startDate, $endDate]);
        }
        
        $movements = $query->get();
        
        return [
            'total_inbound' => $movements->where('quantity', '>', 0)->sum('quantity'),
            'total_outbound' => abs($movements->where('quantity', '<', 0)->sum('quantity')),
            'net_movement' => $movements->sum('quantity'),
            'movement_count' => $movements->count(),
            'total_cost_impact' => $movements->sum('total_cost'),
        ];
    }
}