<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class InventoryAdjustment extends Model
{
    protected $fillable = [
        'user_id',
        'adjustment_number',
        'adjustment_date',
        'type',
        'reason',
        'status',
        'notes',
    ];

    protected $casts = [
        'adjustment_date' => 'date',
    ];

    /**
     * علاقة مع المستخدم
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * علاقة مع عناصر التسوية
     */
    public function items(): HasMany
    {
        return $this->hasMany(InventoryAdjustmentItem::class);
    }

    /**
     * فلترة حسب النوع
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * فلترة حسب الحالة
     */
    public function scopeOfStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * التسويات المعلقة
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * التسويات المؤكدة
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * حساب إجمالي العناصر
     */
    public function getTotalItemsAttribute(): int
    {
        return $this->items->count();
    }

    /**
     * حساب إجمالي التأثير على القيمة
     */
    public function getTotalValueImpactAttribute(): float
    {
        return $this->items->sum(function ($item) {
            return $item->quantity_difference * $item->unit_cost;
        });
    }
}