<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Folder extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'color',
        'icon',
        'parent_id',
        'path',
        'level',
        'is_public',
        'access_level',
        'permissions',
        'created_by',
        'files_count',
        'subfolders_count',
        'total_size',
        'is_system',
        'sort_order',
    ];

    protected $casts = [
        'permissions' => 'array',
        'is_public' => 'boolean',
        'is_system' => 'boolean',
    ];

    protected $appends = [
        'formatted_total_size',
        'breadcrumb',
        'full_path',
    ];

    /**
     * العلاقة مع المجلد الأب
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Folder::class, 'parent_id');
    }

    /**
     * العلاقة مع المجلدات الفرعية
     */
    public function children(): HasMany
    {
        return $this->hasMany(Folder::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * العلاقة مع جميع المجلدات الفرعية (مع الأحفاد)
     */
    public function descendants(): HasMany
    {
        return $this->hasMany(Folder::class, 'parent_id')->with('descendants');
    }

    /**
     * العلاقة مع الملفات
     */
    public function files(): HasMany
    {
        return $this->hasMany(File::class);
    }

    /**
     * العلاقة مع منشئ المجلد
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * تنسيق الحجم الإجمالي
     */
    public function getFormattedTotalSizeAttribute(): string
    {
        $bytes = $this->total_size;
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * الحصول على مسار التنقل (Breadcrumb)
     */
    public function getBreadcrumbAttribute(): array
    {
        $breadcrumb = [];
        $current = $this;
        
        while ($current) {
            array_unshift($breadcrumb, [
                'id' => $current->id,
                'name' => $current->name,
                'slug' => $current->slug,
            ]);
            $current = $current->parent;
        }
        
        return $breadcrumb;
    }

    /**
     * المسار الكامل للمجلد
     */
    public function getFullPathAttribute(): string
    {
        return collect($this->breadcrumb)->pluck('name')->implode(' / ');
    }

    /**
     * توليد slug فريد
     */
    public static function generateUniqueSlug(string $name, ?int $parentId = null, ?int $excludeId = null): string
    {
        $baseSlug = Str::slug($name);
        $slug = $baseSlug;
        $counter = 1;
        
        while (static::where('slug', $slug)
                    ->where('parent_id', $parentId)
                    ->when($excludeId, fn($q) => $q->where('id', '!=', $excludeId))
                    ->exists()) {
            $slug = $baseSlug . '-' . $counter;
            $counter++;
        }
        
        return $slug;
    }

    /**
     * توليد المسار الكامل
     */
    public function generatePath(): string
    {
        if ($this->parent) {
            return $this->parent->path . '/' . $this->slug;
        }
        
        return $this->slug;
    }

    /**
     * تحديث إحصائيات المجلد
     */
    public function updateStats(): void
    {
        $this->update([
            'files_count' => $this->files()->count(),
            'subfolders_count' => $this->children()->count(),
            'total_size' => $this->calculateTotalSize(),
        ]);
        
        // تحديث إحصائيات المجلد الأب
        if ($this->parent) {
            $this->parent->updateStats();
        }
    }

    /**
     * حساب الحجم الإجمالي
     */
    protected function calculateTotalSize(): int
    {
        $filesSize = $this->files()->sum('size');
        $subfoldersSize = $this->children()->sum('total_size');
        
        return $filesSize + $subfoldersSize;
    }

    /**
     * نسخ المجلد مع محتوياته
     */
    public function copyTo(?Folder $destination = null, string $newName = null): Folder
    {
        $newFolder = $this->replicate();
        $newFolder->name = $newName ?? ($this->name . ' - نسخة');
        $newFolder->slug = static::generateUniqueSlug($newFolder->name, $destination?->id);
        $newFolder->parent_id = $destination?->id;
        $newFolder->path = $destination ? $destination->path . '/' . $newFolder->slug : $newFolder->slug;
        $newFolder->level = $destination ? $destination->level + 1 : 0;
        $newFolder->save();
        
        // نسخ الملفات
        foreach ($this->files as $file) {
            $newFile = $file->replicate();
            $newFile->folder_id = $newFolder->id;
            $newFile->save();
        }
        
        // نسخ المجلدات الفرعية
        foreach ($this->children as $child) {
            $child->copyTo($newFolder);
        }
        
        $newFolder->updateStats();
        
        return $newFolder;
    }

    /**
     * نقل المجلد إلى وجهة جديدة
     */
    public function moveTo(?Folder $destination = null): void
    {
        $oldPath = $this->path;
        
        $this->update([
            'parent_id' => $destination?->id,
            'path' => $destination ? $destination->path . '/' . $this->slug : $this->slug,
            'level' => $destination ? $destination->level + 1 : 0,
        ]);
        
        // تحديث مسارات المجلدات الفرعية
        $this->updateChildrenPaths($oldPath);
        
        // تحديث إحصائيات المجلدات المتأثرة
        $this->updateStats();
        if ($destination) {
            $destination->updateStats();
        }
    }

    /**
     * تحديث مسارات المجلدات الفرعية
     */
    protected function updateChildrenPaths(string $oldPath): void
    {
        foreach ($this->descendants as $descendant) {
            $newPath = str_replace($oldPath, $this->path, $descendant->path);
            $descendant->update(['path' => $newPath]);
        }
    }

    /**
     * حذف المجلد مع محتوياته
     */
    public function deleteWithContents(): bool
    {
        // حذف جميع الملفات
        foreach ($this->files as $file) {
            // حذف الملف الفعلي من التخزين
            if (\Storage::exists($file->path)) {
                \Storage::delete($file->path);
            }
            $file->delete();
        }
        
        // حذف المجلدات الفرعية
        foreach ($this->children as $child) {
            $child->deleteWithContents();
        }
        
        return $this->delete();
    }

    /**
     * البحث في المجلدات
     */
    public function scopeSearch($query, string $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%");
        });
    }

    /**
     * فلترة المجلدات الجذرية
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * فلترة المجلدات العامة
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * فلترة مجلدات النظام
     */
    public function scopeSystem($query)
    {
        return $query->where('is_system', true);
    }

    /**
     * فلترة مجلدات المستخدم
     */
    public function scopeUserFolders($query)
    {
        return $query->where('is_system', false);
    }

    /**
     * ترتيب حسب المستوى والترتيب
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('level')->orderBy('sort_order')->orderBy('name');
    }
}
