<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class FiscalYear extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'start_date',
        'end_date',
        'is_current',
        'is_closed',
        'closed_at',
        'closed_by',
        'opening_balance_total',
        'closing_balance_total',
        'notes',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'closed_at' => 'date',
        'is_current' => 'boolean',
        'is_closed' => 'boolean',
        'opening_balance_total' => 'decimal:2',
        'closing_balance_total' => 'decimal:2',
    ];

    /**
     * العلاقات
     */
    
    // المستخدم الذي أغلق السنة المالية
    public function closedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'closed_by');
    }

    // الميزانيات المرتبطة بهذه السنة المالية
    public function budgets(): HasMany
    {
        return $this->hasMany(Budget::class);
    }

    // أرصدة الحسابات لهذه السنة المالية
    public function accountBalances(): HasMany
    {
        return $this->hasMany(AccountBalance::class);
    }

    // القيود المحاسبية في هذه السنة المالية
    public function journalEntries(): HasMany
    {
        return $this->hasMany(JournalEntry::class);
    }

    /**
     * Scopes
     */
    
    // السنة المالية الحالية
    public function scopeCurrent($query)
    {
        return $query->where('is_current', true);
    }

    // السنوات المالية المفتوحة
    public function scopeOpen($query)
    {
        return $query->where('is_closed', false);
    }

    // السنوات المالية المغلقة
    public function scopeClosed($query)
    {
        return $query->where('is_closed', true);
    }

    // فلترة حسب التاريخ
    public function scopeForDate($query, $date)
    {
        $date = Carbon::parse($date);
        return $query->where('start_date', '<=', $date)
                    ->where('end_date', '>=', $date);
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // فترة السنة المالية بالأشهر
    public function getDurationInMonthsAttribute()
    {
        return $this->start_date->diffInMonths($this->end_date) + 1;
    }

    // فترة السنة المالية بالأيام
    public function getDurationInDaysAttribute()
    {
        return $this->start_date->diffInDays($this->end_date) + 1;
    }

    // نسبة الانتهاء من السنة المالية
    public function getCompletionPercentageAttribute()
    {
        $now = now();
        
        if ($now->lt($this->start_date)) {
            return 0;
        }
        
        if ($now->gt($this->end_date)) {
            return 100;
        }
        
        $totalDays = $this->start_date->diffInDays($this->end_date);
        $elapsedDays = $this->start_date->diffInDays($now);
        
        return round(($elapsedDays / $totalDays) * 100, 2);
    }

    // الأيام المتبقية
    public function getRemainingDaysAttribute()
    {
        return max(0, now()->diffInDays($this->end_date, false));
    }

    // حالة السنة المالية
    public function getStatusAttribute()
    {
        if ($this->is_closed) {
            return 'مغلقة';
        }
        
        if ($this->is_current) {
            return 'جارية';
        }
        
        $now = now();
        
        if ($now->lt($this->start_date)) {
            return 'مستقبلية';
        }
        
        if ($now->gt($this->end_date)) {
            return 'منتهية';
        }
        
        return 'نشطة';
    }

    /**
     * الدوال المساعدة
     */
    
    // التحقق من إمكانية الإغلاق
    public function canBeClosed()
    {
        if ($this->is_closed) {
            return false;
        }
        
        // التحقق من وجود قيود غير معتمدة
        $unapprovedEntries = $this->journalEntries()
            ->where('is_approved', false)
            ->count();
        
        if ($unapprovedEntries > 0) {
            return false;
        }
        
        // التحقق من انتهاء السنة المالية
        return now()->gte($this->end_date);
    }

    // إغلاق السنة المالية
    public function close($closedBy = null)
    {
        if (!$this->canBeClosed()) {
            throw new \Exception('لا يمكن إغلاق السنة المالية في الوقت الحالي');
        }
        
        // حساب الرصيد الختامي
        $totalAssets = Account::where('type', 'asset')->sum('current_balance');
        $totalLiabilities = abs(Account::where('type', 'liability')->sum('current_balance'));
        $totalEquity = abs(Account::where('type', 'equity')->sum('current_balance'));
        
        $this->update([
            'is_closed' => true,
            'closed_at' => now(),
            'closed_by' => $closedBy ?? auth()->id(),
            'closing_balance_total' => $totalAssets,
        ]);
        
        // تعيين السنة التالية كحالية إذا وجدت
        $nextYear = static::where('start_date', '>', $this->end_date)
            ->orderBy('start_date')
            ->first();
        
        if ($nextYear) {
            $nextYear->update(['is_current' => true]);
        }
        
        return true;
    }

    // إعادة فتح السنة المالية
    public function reopen()
    {
        if (!$this->is_closed) {
            throw new \Exception('السنة المالية غير مغلقة');
        }
        
        $this->update([
            'is_closed' => false,
            'closed_at' => null,
            'closed_by' => null,
        ]);
        
        return true;
    }

    // تعيين كسنة مالية حالية
    public function setCurrent()
    {
        // إلغاء تحديد السنة الحالية
        static::where('is_current', true)->update(['is_current' => false]);
        
        // تعيين هذه السنة كحالية
        $this->update(['is_current' => true]);
        
        return true;
    }

    // إنشاء السنة المالية التالية
    public function createNext()
    {
        $nextStartDate = $this->end_date->copy()->addDay();
        $nextEndDate = $nextStartDate->copy()->addYear()->subDay();
        
        return static::create([
            'name' => 'السنة المالية ' . $nextStartDate->year,
            'start_date' => $nextStartDate,
            'end_date' => $nextEndDate,
            'is_current' => false,
            'is_closed' => false,
            'opening_balance_total' => $this->closing_balance_total,
        ]);
    }

    // الحصول على السنة المالية الحالية
    public static function getCurrentFiscalYear()
    {
        return static::current()->first() ?? static::forDate(now())->first();
    }

    // الحصول على السنة المالية للتاريخ المحدد
    public static function getFiscalYearForDate($date)
    {
        return static::forDate($date)->first();
    }

    // إنشاء السنة المالية الافتراضية
    public static function createDefault()
    {
        $startDate = now()->startOfYear();
        $endDate = now()->endOfYear();
        
        return static::create([
            'name' => 'السنة المالية ' . $startDate->year,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'is_current' => true,
            'is_closed' => false,
            'opening_balance_total' => 0,
        ]);
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        // عند إنشاء سنة مالية جديدة، التأكد من عدم تداخل التواريخ
        static::creating(function ($fiscalYear) {
            $overlapping = static::where(function ($query) use ($fiscalYear) {
                $query->whereBetween('start_date', [$fiscalYear->start_date, $fiscalYear->end_date])
                      ->orWhereBetween('end_date', [$fiscalYear->start_date, $fiscalYear->end_date])
                      ->orWhere(function ($q) use ($fiscalYear) {
                          $q->where('start_date', '<=', $fiscalYear->start_date)
                            ->where('end_date', '>=', $fiscalYear->end_date);
                      });
            })->exists();
            
            if ($overlapping) {
                throw new \Exception('تتداخل السنة المالية مع سنة مالية أخرى');
            }
        });
        
        // عند تعيين سنة مالية كحالية، إلغاء السنوات الأخرى
        static::updating(function ($fiscalYear) {
            if ($fiscalYear->isDirty('is_current') && $fiscalYear->is_current) {
                static::where('id', '!=', $fiscalYear->id)
                      ->where('is_current', true)
                      ->update(['is_current' => false]);
            }
        });
    }
}