<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class File extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'filename',
        'path',
        'extension',
        'mime_type',
        'size',
        'description',
        'category',
        'metadata',
        'is_public',
        'access_level',
        'uploaded_by',
        'folder_id',
        'download_count',
        'last_downloaded_at',
        'tags',
        'hash',
        'is_archived',
        'expires_at',
    ];

    protected $casts = [
        'metadata' => 'array',
        'tags' => 'array',
        'is_public' => 'boolean',
        'is_archived' => 'boolean',
        'last_downloaded_at' => 'datetime',
        'expires_at' => 'datetime',
    ];

    protected $appends = [
        'formatted_size',
        'file_url',
        'download_url',
        'is_image',
        'is_document',
        'is_video',
        'is_audio',
        'preview_url',
    ];

    /**
     * العلاقة مع المستخدم الذي رفع الملف
     */
    public function uploader(): BelongsTo
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * العلاقة مع المجلد
     */
    public function folder(): BelongsTo
    {
        return $this->belongsTo(Folder::class);
    }

    /**
     * تنسيق حجم الملف
     */
    public function getFormattedSizeAttribute(): string
    {
        $bytes = $this->size;
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * رابط الملف
     */
    public function getFileUrlAttribute(): string
    {
        if ($this->is_public) {
            return Storage::url($this->path);
        }
        
        return route('file-manager.files.download', $this->id);
    }

    /**
     * رابط تحميل الملف
     */
    public function getDownloadUrlAttribute(): string
    {
        return route('file-manager.files.download', $this->id);
    }

    /**
     * التحقق من كون الملف صورة
     */
    public function getIsImageAttribute(): bool
    {
        return Str::startsWith($this->mime_type, 'image/');
    }

    /**
     * التحقق من كون الملف مستند
     */
    public function getIsDocumentAttribute(): bool
    {
        $documentTypes = [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
            'text/plain',
        ];
        
        return in_array($this->mime_type, $documentTypes);
    }

    /**
     * التحقق من كون الملف فيديو
     */
    public function getIsVideoAttribute(): bool
    {
        return Str::startsWith($this->mime_type, 'video/');
    }

    /**
     * التحقق من كون الملف صوت
     */
    public function getIsAudioAttribute(): bool
    {
        return Str::startsWith($this->mime_type, 'audio/');
    }

    /**
     * رابط معاينة الملف
     */
    public function getPreviewUrlAttribute(): ?string
    {
        if ($this->is_image) {
            return $this->file_url;
        }
        
        if ($this->mime_type === 'application/pdf') {
            return route('file-manager.files.preview', $this->id);
        }
        
        return null;
    }

    /**
     * تحديث عداد التحميلات
     */
    public function incrementDownloadCount(): void
    {
        $this->increment('download_count');
        $this->update(['last_downloaded_at' => now()]);
    }

    /**
     * توليد hash للملف
     */
    public static function generateHash(string $filePath): string
    {
        return hash_file('sha256', $filePath);
    }

    /**
     * البحث في الملفات
     */
    public function scopeSearch($query, string $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'like', "%{$search}%")
              ->orWhere('description', 'like', "%{$search}%")
              ->orWhereJsonContains('tags', $search);
        });
    }

    /**
     * فلترة حسب النوع
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('mime_type', 'like', "{$type}/%");
    }

    /**
     * فلترة حسب التصنيف
     */
    public function scopeInCategory($query, string $category)
    {
        return $query->where('category', $category);
    }

    /**
     * فلترة الملفات العامة
     */
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    /**
     * فلترة الملفات غير المؤرشفة
     */
    public function scopeActive($query)
    {
        return $query->where('is_archived', false);
    }

    /**
     * فلترة الملفات غير المنتهية الصلاحية
     */
    public function scopeNotExpired($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expires_at')
              ->orWhere('expires_at', '>', now());
        });
    }

    /**
     * الحصول على أيقونة الملف
     */
    public function getIcon(): string
    {
        if ($this->is_image) {
            return 'fas fa-image';
        }
        
        if ($this->is_video) {
            return 'fas fa-video';
        }
        
        if ($this->is_audio) {
            return 'fas fa-music';
        }
        
        if ($this->is_document) {
            return match ($this->extension) {
                'pdf' => 'fas fa-file-pdf',
                'doc', 'docx' => 'fas fa-file-word',
                'xls', 'xlsx' => 'fas fa-file-excel',
                'ppt', 'pptx' => 'fas fa-file-powerpoint',
                default => 'fas fa-file-alt',
            };
        }
        
        return 'fas fa-file';
    }

    /**
     * الحصول على لون الملف
     */
    public function getColor(): string
    {
        if ($this->is_image) {
            return '#e74c3c';
        }
        
        if ($this->is_video) {
            return '#9b59b6';
        }
        
        if ($this->is_audio) {
            return '#f39c12';
        }
        
        if ($this->is_document) {
            return match ($this->extension) {
                'pdf' => '#e74c3c',
                'doc', 'docx' => '#3498db',
                'xls', 'xlsx' => '#27ae60',
                'ppt', 'pptx' => '#e67e22',
                default => '#95a5a6',
            };
        }
        
        return '#95a5a6';
    }

    /**
     * تنسيق حجم الملف (دالة ثابتة)
     */
    public static function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
}
