<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\Invoice;

class Customer extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'vat_number',
        'customer_type',
        'national_id',
        'cr_number',
        // Saudi National Address fields (required)
        'city',
        'district',
        'street',
        'building_number',
        'additional_number',
        'postal_code',
        'unit_number',
        // Contact information
        'email',
        'phone',
        // Legacy fields for backward compatibility
        'commercial_registration',
        'tax_number',
        'contact_person',
        'mobile',
        'address',
        'country',
        'currency',
        'credit_limit',
        'payment_terms_days',
        'payment_method',
        'current_balance',
        'registration_date',
        'notes',
        'is_active',
    ];

    protected $casts = [
        'credit_limit' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'payment_terms_days' => 'integer',
        'registration_date' => 'date',
        'is_active' => 'boolean',
        'customer_type' => 'string',
    ];

    /**
     * Get all sales invoices for this customer.
     */
    public function salesInvoices(): HasMany
    {
        return $this->hasMany(SalesInvoice::class);
    }

    /**
     * Get all invoices for this customer.
     */
    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * Get all receipt vouchers for this customer.
     */
    public function receiptVouchers(): HasMany
    {
        return $this->hasMany(ReceiptVoucher::class);
    }

    /**
     * Get active customers only.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get customers with outstanding balance.
     */
    public function scopeWithBalance($query)
    {
        return $query->where('current_balance', '>', 0);
    }

    /**
     * Filter by customer type.
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('customer_type', $type);
    }

    /**
     * Generate next customer code.
     */
    public static function generateNextCode(): string
    {
        $lastCustomer = static::orderBy('id', 'desc')->first();
        $nextNumber = $lastCustomer ? (int)substr($lastCustomer->code, 4) + 1 : 1;
        return 'CUST' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get customer's total sales this year.
     */
    public function getTotalSalesThisYear(): float
    {
        return $this->salesInvoices()
            ->whereYear('invoice_date', date('Y'))
            ->where('status', '!=', 'cancelled')
            ->sum('total_amount');
    }

    /**
     * Get customer's payment status.
     */
    public function getPaymentStatusAttribute(): string
    {
        if ($this->current_balance == 0) {
            return 'paid';
        } elseif ($this->current_balance > 0) {
            // Check if there are overdue invoices
            $overdueCount = $this->salesInvoices()
                ->where('payment_status', 'overdue')
                ->count();

            return $overdueCount > 0 ? 'overdue' : 'outstanding';
        }

        return 'credit';
    }

    /**
     * Get customer's overdue amount.
     */
    public function getOverdueAmount(): float
    {
        return $this->salesInvoices()
            ->where('payment_status', 'overdue')
            ->sum('remaining_amount');
    }

    /**
     * Check if customer exceeds credit limit.
     */
    public function exceedsCreditLimit(): bool
    {
        return $this->credit_limit > 0 && $this->current_balance > $this->credit_limit;
    }

    /**
     * Get available credit.
     */
    public function getAvailableCredit(): float
    {
        if ($this->credit_limit <= 0) {
            return 0;
        }

        return max(0, $this->credit_limit - $this->current_balance);
    }

    /**
     * Update customer balance.
     */
    public function updateBalance(float $amount, string $operation = 'add'): void
    {
        if ($operation === 'add') {
            $this->current_balance += $amount;
        } elseif ($operation === 'subtract') {
            $this->current_balance -= $amount;
        } else {
            $this->current_balance = $amount;
        }

        $this->save();
    }

    /**
     * Get customer's top purchased items.
     */
    public function getTopPurchasedItems(int $limit = 5)
    {
        return \DB::table('sales_invoice_details')
            ->join('sales_invoices', 'sales_invoice_details.sales_invoice_id', '=', 'sales_invoices.id')
            ->join('items', 'sales_invoice_details.item_id', '=', 'items.id')
            ->where('sales_invoices.customer_id', $this->id)
            ->where('sales_invoices.status', '!=', 'cancelled')
            ->select(
                'items.name',
                'items.code',
                \DB::raw('SUM(sales_invoice_details.quantity) as total_quantity'),
                \DB::raw('SUM(sales_invoice_details.total_amount) as total_amount')
            )
            ->groupBy('items.id', 'items.name', 'items.code')
            ->orderByDesc('total_amount')
            ->limit($limit)
            ->get();
    }

    /**
     * Get customer's average order value.
     */
    public function getAverageOrderValue(): float
    {
        $invoices = $this->salesInvoices()
            ->where('status', '!=', 'cancelled')
            ->get();

        if ($invoices->count() === 0) {
            return 0;
        }

        return $invoices->avg('total_amount');
    }

    /**
     * Get customer's last order date.
     */
    public function getLastOrderDate()
    {
        return $this->salesInvoices()
            ->where('status', '!=', 'cancelled')
            ->orderBy('invoice_date', 'desc')
            ->value('invoice_date');
    }

    /**
     * Get customer loyalty level based on total purchases.
     */
    public function getLoyaltyLevel(): string
    {
        $totalSales = $this->getTotalSalesThisYear();

        if ($totalSales >= 100000) {
            return 'platinum';
        } elseif ($totalSales >= 50000) {
            return 'gold';
        } elseif ($totalSales >= 20000) {
            return 'silver';
        } elseif ($totalSales >= 5000) {
            return 'bronze';
        }

        return 'standard';
    }

    /**
     * Get customer's invoice count this year.
     */
    public function getInvoiceCountThisYear(): int
    {
        return $this->salesInvoices()
            ->whereYear('invoice_date', date('Y'))
            ->where('status', '!=', 'cancelled')
            ->count();
    }

    /**
     * Get customer's full address attribute.
     */
    public function getFullAddressAttribute(): string
    {
        $addressParts = array_filter([
            $this->address,
            $this->city,
            $this->country,
            $this->postal_code
        ]);

        return implode(', ', $addressParts) ?: 'لا يوجد عنوان محدد';
    }
}