<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CostCenter extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'name_en',
        'type',
        'parent_id',
        'manager_name',
        'phone',
        'description',
        'budget_amount',
        'budget_period',
        'actual_amount',
        'variance_threshold',
        'is_active',
        'allow_budget_exceeded',
        'requires_approval',
        'track_performance',
        'manager_id'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'allow_budget_exceeded' => 'boolean',
        'requires_approval' => 'boolean',
        'track_performance' => 'boolean',
        'budget_amount' => 'decimal:2',
        'actual_amount' => 'decimal:2',
        'variance_threshold' => 'integer'
    ];

    // Relations
    public function parent(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(CostCenter::class, 'parent_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'manager_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeParents($query)
    {
        return $query->whereNull('parent_id');
    }

    // Helper methods
    public function getFullNameAttribute()
    {
        return $this->code . ' - ' . $this->name;
    }

    public function getActualExpenses($startDate, $endDate)
    {
        return JournalEntryLine::whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('entry_date', [$startDate, $endDate])
                      ->where('status', 'posted');
            })
            ->where('cost_center_id', $this->id)
            ->sum('debit_amount');
    }

    public function getBudgetVariance($startDate, $endDate)
    {
        $actualExpenses = $this->getActualExpenses($startDate, $endDate);
        $budgetAmount = $this->budget_amount ?? 0;
        
        return $budgetAmount - $actualExpenses;
    }

    public function getBudgetUtilizationPercentage($startDate, $endDate)
    {
        if (!$this->budget_amount || $this->budget_amount == 0) {
            return 0;
        }

        $actualExpenses = $this->getActualExpenses($startDate, $endDate);
        return ($actualExpenses / $this->budget_amount) * 100;
    }

    public static function getCostCentersTree()
    {
        return self::with('children')
            ->whereNull('parent_id')
            ->where('is_active', true)
            ->orderBy('code')
            ->get();
    }
}