<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Company extends Model
{
    protected $fillable = [
        'name',
        'name_en',
        'logo',
        'vat_numbers',
        'cr_number',
        'city',
        'district',
        'street',
        'building_number',
        'additional_number',
        'postal_code',
        'unit_number',
        'email',
        'phone',
        'website',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'vat_numbers' => 'array',
    ];

    /**
     * علاقة مع الفواتير
     */
    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * علاقة مع إعدادات ZATCA
     */
    public function zatcaSettings(): HasOne
    {
        return $this->hasOne(ZatcaSettings::class);
    }

    /**
     * علاقة مع سندات القبض
     */
    public function receiptVouchers(): HasMany
    {
        return $this->hasMany(ReceiptVoucher::class);
    }

    /**
     * علاقة مع سندات الصرف
     */
    public function paymentVouchers(): HasMany
    {
        return $this->hasMany(PaymentVoucher::class);
    }

    /**
     * الحصول على العنوان الكامل
     */
    public function getFullAddressAttribute(): string
    {
        return sprintf(
            '%s، %s، %s، رقم المبنى: %s، الرقم الإضافي: %s، الرمز البريدي: %s%s',
            $this->street,
            $this->district,
            $this->city,
            $this->building_number,
            $this->additional_number,
            $this->postal_code,
            $this->unit_number ? '، رقم الوحدة: ' . $this->unit_number : ''
        );
    }

    /**
     * الحصول على العنوان المختصر (للاستخدام في XML)
     */
    public function getAddressAttribute(): string
    {
        return $this->full_address;
    }

    /**
     * الحصول على السجل التجاري (alias لـ cr_number)
     */
    public function getCommercialRegistrationAttribute(): ?string
    {
        return $this->cr_number;
    }

    /**
     * الحصول على الرقم الضريبي الأول (للتوافق مع النظام القديم)
     */
    public function getVatNumberAttribute(): ?string
    {
        return $this->vat_numbers && count($this->vat_numbers) > 0 ? $this->vat_numbers[0] : null;
    }

    /**
     * الحصول على جميع الأرقام الضريبية كنص منسق
     */
    public function getFormattedVatNumbersAttribute(): string
    {
        if (!$this->vat_numbers || count($this->vat_numbers) === 0) {
            return 'غير محدد';
        }
        
        return implode(' - ', $this->vat_numbers);
    }

    /**
     * إضافة رقم ضريبي جديد
     */
    public function addVatNumber(string $vatNumber): void
    {
        $vatNumbers = $this->vat_numbers ?? [];
        
        if (!in_array($vatNumber, $vatNumbers)) {
            $vatNumbers[] = $vatNumber;
            $this->vat_numbers = $vatNumbers;
            $this->save();
        }
    }

    /**
     * إزالة رقم ضريبي
     */
    public function removeVatNumber(string $vatNumber): void
    {
        $vatNumbers = $this->vat_numbers ?? [];
        $key = array_search($vatNumber, $vatNumbers);
        
        if ($key !== false) {
            unset($vatNumbers[$key]);
            $this->vat_numbers = array_values($vatNumbers); // إعادة ترقيم المصفوفة
            $this->save();
        }
    }

    /**
     * التحقق من وجود رقم ضريبي معين
     */
    public function hasVatNumber(string $vatNumber): bool
    {
        return $this->vat_numbers && in_array($vatNumber, $this->vat_numbers);
    }

    /**
     * فلترة الشركات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * الحصول على مسار الشعار الكامل
     */
    public function getLogoUrlAttribute(): ?string
    {
        if ($this->logo) {
            return asset($this->logo);
        }
        
        return null;
    }

    /**
     * التحقق من وجود شعار
     */
    public function hasLogo(): bool
    {
        return !empty($this->logo) && file_exists(public_path($this->logo));
    }
}
