<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Category extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'name_en',
        'description',
        'code',
        'color',
        'icon',
        'image',
        'parent_id',
        'sort_order',
        'level',
        'path',
        'is_active',
        'is_featured',
        'show_on_menu',
        'slug',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'type',
        'requires_inventory',
        'default_tax_rate',
        'default_unit',
        'products_count',
        'views_count',
        'total_sales',
        'created_by',
        'updated_by',
        'last_activity_at',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_featured' => 'boolean',
        'show_on_menu' => 'boolean',
        'requires_inventory' => 'boolean',
        'meta_keywords' => 'array',
        'default_tax_rate' => 'decimal:2',
        'total_sales' => 'decimal:2',
        'last_activity_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            if (empty($category->slug)) {
                $category->slug = static::generateUniqueSlug($category->name);
            }
            if (empty($category->code)) {
                $category->code = static::generateUniqueCode($category->name);
            }
            $category->created_by = auth()->id();
            $category->last_activity_at = now();
        });

        static::updating(function ($category) {
            $category->updated_by = auth()->id();
            $category->last_activity_at = now();
        });

        static::saved(function ($category) {
            $category->updateHierarchy();
        });
    }

    // ==================== Relationships ====================

    /**
     * Get the parent category
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    /**
     * Get child categories
     */
    public function children(): HasMany
    {
        return $this->hasMany(Category::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * Get all descendant categories
     */
    public function descendants(): HasMany
    {
        return $this->hasMany(Category::class, 'parent_id')->with('descendants');
    }

    /**
     * Get products in this category
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the creator of the category
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the last updater of the category
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    // ==================== Scopes ====================

    /**
     * Scope to get active categories
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get featured categories
     */
    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    /**
     * Scope to get root categories (no parent)
     */
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope to get categories by type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Scope to get categories shown on menu
     */
    public function scopeOnMenu($query)
    {
        return $query->where('show_on_menu', true);
    }

    /**
     * Scope to get categories ordered by sort order
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    /**
     * Scope to get categories by level
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('level', $level);
    }

    // ==================== Accessors & Mutators ====================

    /**
     * Get the full name including parent hierarchy
     */
    public function getFullNameAttribute(): string
    {
        if ($this->parent_id) {
            return $this->parent->full_name . ' → ' . $this->name;
        }
        return $this->name;
    }

    /**
     * Get breadcrumb array
     */
    public function getBreadcrumbAttribute(): array
    {
        $breadcrumb = [];
        $category = $this;

        while ($category) {
            array_unshift($breadcrumb, [
                'id' => $category->id,
                'name' => $category->name,
                'slug' => $category->slug,
            ]);
            $category = $category->parent;
        }

        return $breadcrumb;
    }

    /**
     * Get the status text
     */
    public function getStatusTextAttribute(): string
    {
        return $this->is_active ? 'نشط' : 'غير نشط';
    }

    /**
     * Get the status color
     */
    public function getStatusColorAttribute(): string
    {
        return $this->is_active ? 'success' : 'secondary';
    }

    /**
     * Get the type text in Arabic
     */
    public function getTypeTextAttribute(): string
    {
        return match($this->type) {
            'product' => 'منتجات',
            'service' => 'خدمات',
            'both' => 'منتجات وخدمات',
            default => 'غير محدد'
        };
    }

    /**
     * Set slug automatically
     */
    public function setNameAttribute($value)
    {
        $this->attributes['name'] = $value;
        if (empty($this->attributes['slug'])) {
            $this->attributes['slug'] = static::generateUniqueSlug($value);
        }
    }

    // ==================== Static Methods ====================

    /**
     * Generate unique slug
     */
    public static function generateUniqueSlug(string $name, ?int $excludeId = null): string
    {
        $slug = Str::slug($name);
        $originalSlug = $slug;
        $counter = 1;

        while (static::where('slug', $slug)
            ->when($excludeId, fn($q) => $q->where('id', '!=', $excludeId))
            ->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    /**
     * Generate unique code
     */
    public static function generateUniqueCode(string $name, ?int $excludeId = null): string
    {
        $code = strtoupper(Str::limit(Str::slug($name, ''), 10, ''));
        $originalCode = $code;
        $counter = 1;

        while (static::where('code', $code)
            ->when($excludeId, fn($q) => $q->where('id', '!=', $excludeId))
            ->exists()) {
            $code = $originalCode . $counter;
            $counter++;
        }

        return $code;
    }

    /**
     * Get categories tree
     */
    public static function getCategoriesTree(): array
    {
        return static::with('children.children.children')
            ->root()
            ->active()
            ->ordered()
            ->get()
            ->toArray();
    }

    /**
     * Get flat categories list for select options
     */
    public static function getSelectOptions($excludeId = null): array
    {
        $categories = static::active()
            ->when($excludeId, fn($q) => $q->where('id', '!=', $excludeId))
            ->orderBy('level')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $options = [];
        foreach ($categories as $category) {
            $prefix = str_repeat('— ', $category->level);
            $options[$category->id] = $prefix . $category->name;
        }

        return $options;
    }

    // ==================== Instance Methods ====================

    /**
     * Update hierarchy information
     */
    public function updateHierarchy(): void
    {
        if ($this->parent_id) {
            $parent = $this->parent;
            $this->level = $parent->level + 1;
            $this->path = $parent->path ? $parent->path . '/' . $this->id : (string)$this->id;
        } else {
            $this->level = 0;
            $this->path = (string)$this->id;
        }

        $this->saveQuietly();

        // Update children hierarchy
        foreach ($this->children as $child) {
            $child->updateHierarchy();
        }
    }

    /**
     * Check if category has children
     */
    public function hasChildren(): bool
    {
        return $this->children()->exists();
    }

    /**
     * Get all ancestor categories
     */
    public function getAncestors()
    {
        $ancestors = collect();
        $category = $this->parent;

        while ($category) {
            $ancestors->prepend($category);
            $category = $category->parent;
        }

        return $ancestors;
    }

    /**
     * Check if this category is ancestor of given category
     */
    public function isAncestorOf(Category $category): bool
    {
        return $category->getAncestors()->contains('id', $this->id);
    }

    /**
     * Check if this category is descendant of given category
     */
    public function isDescendantOf(Category $category): bool
    {
        return $this->getAncestors()->contains('id', $category->id);
    }

    /**
     * Update product count
     */
    public function updateProductsCount(): void
    {
        $this->products_count = $this->products()->count();
        $this->saveQuietly();
    }

    /**
     * Increment views count
     */
    public function incrementViews(): void
    {
        $this->increment('views_count');
        $this->touch('last_activity_at');
    }

    /**
     * Update total sales
     */
    public function updateTotalSales(): void
    {
        $this->total_sales = $this->products()
            ->join('invoice_items', 'products.id', '=', 'invoice_items.product_id')
            ->join('invoices', 'invoice_items.invoice_id', '=', 'invoices.id')
            ->where('invoices.status', 'paid')
            ->sum('invoice_items.total_with_tax');
        
        $this->saveQuietly();
    }

    /**
     * Get category depth level
     */
    public function getDepth(): int
    {
        return $this->level;
    }

    /**
     * Move category to new parent
     */
    public function moveTo(?Category $newParent): void
    {
        $this->parent_id = $newParent?->id;
        $this->save();
    }

    /**
     * Duplicate category
     */
    public function duplicate(string $newName = null): Category
    {
        $newCategory = $this->replicate([
            'slug',
            'code',
            'products_count',
            'views_count',
            'total_sales',
        ]);

        $newCategory->name = $newName ?: $this->name . ' (نسخة)';
        $newCategory->is_active = false;
        $newCategory->save();

        return $newCategory;
    }
}
