<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class CashTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'transaction_number',
        'transaction_date',
        'cash_account_id',
        'type',
        'amount',
        'reference',
        'description',
        'contra_account_id',
        'cost_center_id',
        'notes',
        'is_reconciled',
        'received_from',
        'paid_to',
        'payment_method',
        'check_number',
        'check_date',
        'bank_name',
        'journal_entry_id',
        'created_by',
        'attachments',
        'invoice_id'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'check_date' => 'date',
        'amount' => 'decimal:2',
        'is_reconciled' => 'boolean',
        'attachments' => 'array'
    ];

    // Relations
    public function cashAccount(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'cash_account_id');
    }

    public function contraAccount(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'contra_account_id');
    }

    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class, 'cost_center_id');
    }

    public function journalEntry(): BelongsTo
    {
        return $this->belongsTo(JournalEntry::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class, 'invoice_id');
    }

    // Scopes
    public function scopeReceipts($query)
    {
        return $query->where('type', 'receipt');
    }

    public function scopePayments($query)
    {
        return $query->where('type', 'payment');
    }

    public function scopeByPeriod($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    public function scopeByPaymentMethod($query, $method)
    {
        return $query->where('payment_method', $method);
    }

    public function scopeLinkedToInvoices($query)
    {
        return $query->whereNotNull('invoice_id');
    }

    public function scopeUnlinkedToInvoices($query)
    {
        return $query->whereNull('invoice_id');
    }

    public function scopeCashPayments($query)
    {
        return $query->where('payment_method', 'cash');
    }

    public function scopeBankTransfers($query)
    {
        return $query->where('payment_method', 'bank_transfer');
    }

    // Helper methods
    public static function generateTransactionNumber($type)
    {
        $prefix = $type === 'receipt' ? 'CR' : 'CP';
        $lastTransaction = self::where('type', $type)
            ->whereYear('created_at', date('Y'))
            ->orderBy('id', 'desc')
            ->first();

        $nextNumber = $lastTransaction ? (int)substr($lastTransaction->transaction_number, -4) + 1 : 1;
        
        return $prefix . date('Y') . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    public function getFormattedAmountAttribute()
    {
        return number_format($this->amount, 2) . ' ريال';
    }

    public function getTypeTextAttribute()
    {
        return $this->type === 'receipt' ? 'قبض' : 'دفع';
    }

    public function createJournalEntry()
    {
        if ($this->journal_entry_id) {
            return $this->journalEntry;
        }

        $journalEntry = JournalEntry::create([
            'entry_number' => JournalEntry::generateEntryNumber(),
            'entry_date' => $this->transaction_date,
            'reference' => $this->transaction_number,
            'description' => $this->description,
            'type' => 'auto',
            'status' => 'draft',
            'total_debit' => $this->amount,
            'total_credit' => $this->amount,
            'created_by' => $this->created_by
        ]);

        if ($this->type === 'receipt') {
            // Debit Cash Account
            $journalEntry->lines()->create([
                'account_id' => $this->cash_account_id,
                'description' => $this->description,
                'debit_amount' => $this->amount,
                'credit_amount' => 0,
                'reference' => $this->reference,
                'cost_center_id' => $this->cost_center_id
            ]);

            // Credit Contra Account
            if ($this->contra_account_id) {
                $journalEntry->lines()->create([
                    'account_id' => $this->contra_account_id,
                    'description' => $this->description,
                    'debit_amount' => 0,
                    'credit_amount' => $this->amount,
                    'reference' => $this->reference,
                    'cost_center_id' => $this->cost_center_id
                ]);
            }
        } else {
            // Credit Cash Account
            $journalEntry->lines()->create([
                'account_id' => $this->cash_account_id,
                'description' => $this->description,
                'debit_amount' => 0,
                'credit_amount' => $this->amount,
                'reference' => $this->reference,
                'cost_center_id' => $this->cost_center_id
            ]);

            // Debit Contra Account
            if ($this->contra_account_id) {
                $journalEntry->lines()->create([
                    'account_id' => $this->contra_account_id,
                    'description' => $this->description,
                    'debit_amount' => $this->amount,
                    'credit_amount' => 0,
                    'reference' => $this->reference,
                    'cost_center_id' => $this->cost_center_id
                ]);
            }
        }

        $this->update(['journal_entry_id' => $journalEntry->id]);

        return $journalEntry;
    }

    /**
     * التحقق من كون الحركة النقدية مرتبطة بفاتورة
     */
    public function isLinkedToInvoice(): bool
    {
        return !is_null($this->invoice_id);
    }

    /**
     * الحصول على نص طريقة الدفع بالعربية
     */
    public function getPaymentMethodText(): string
    {
        $methods = [
            'cash' => 'نقداً',
            'bank_transfer' => 'تحويل بنكي',
            'check' => 'شيك',
            'credit_card' => 'بطاقة ائتمان',
        ];

        return $methods[$this->payment_method] ?? 'غير محدد';
    }

    /**
     * الحصول على رقم المرجع الكامل
     */
    public function getFullReferenceAttribute(): string
    {
        if ($this->isLinkedToInvoice()) {
            return $this->reference . ' (فاتورة: ' . $this->invoice->invoice_number . ')';
        }

        return $this->reference ?: 'بدون مرجع';
    }

    /**
     * scope للحركات النقدية في فترة معينة مع تجميع حسب طريقة الدفع
     */
    public static function getCashFlowByPeriodAndMethod($startDate, $endDate)
    {
        return self::byPeriod($startDate, $endDate)
            ->receipts()
            ->selectRaw('payment_method, SUM(amount) as total_amount, COUNT(*) as transactions_count')
            ->groupBy('payment_method')
            ->get();
    }

    /**
     * scope للحصول على الحركات النقدية للفواتير فقط
     */
    public static function getInvoiceRelatedTransactions($startDate = null, $endDate = null)
    {
        $query = self::linkedToInvoices()->with('invoice');

        if ($startDate && $endDate) {
            $query->byPeriod($startDate, $endDate);
        }

        return $query->get();
    }

    /**
     * احصائيات الحركات النقدية
     */
    public static function getCashTransactionStats($startDate = null, $endDate = null): array
    {
        $query = self::query();

        if ($startDate && $endDate) {
            $query->byPeriod($startDate, $endDate);
        }

        $receipts = $query->clone()->receipts();
        $payments = $query->clone()->payments();

        return [
            'total_receipts' => $receipts->sum('amount'),
            'total_payments' => $payments->sum('amount'),
            'net_cash_flow' => $receipts->sum('amount') - $payments->sum('amount'),
            'receipts_count' => $receipts->count(),
            'payments_count' => $payments->count(),
            'cash_receipts' => $receipts->clone()->cashPayments()->sum('amount'),
            'bank_receipts' => $receipts->clone()->bankTransfers()->sum('amount'),
            'invoice_linked_receipts' => $receipts->clone()->linkedToInvoices()->sum('amount'),
        ];
    }
}