<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Budget extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'code',
        'fiscal_year_id',
        'cost_center_id',
        'account_id',
        'period_type',
        'start_date',
        'end_date',
        'budgeted_amount',
        'actual_amount',
        'committed_amount',
        'variance_amount',
        'variance_percentage',
        'status',
        'description',
        'monthly_breakdown',
        'is_active',
        'auto_calculate',
        'warning_threshold',
        'critical_threshold',
        'created_by',
        'approved_by',
        'approved_at',
        'last_calculated_at',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'budgeted_amount' => 'decimal:2',
        'actual_amount' => 'decimal:2',
        'committed_amount' => 'decimal:2',
        'variance_amount' => 'decimal:2',
        'variance_percentage' => 'decimal:2',
        'monthly_breakdown' => 'array',
        'is_active' => 'boolean',
        'auto_calculate' => 'boolean',
        'warning_threshold' => 'decimal:2',
        'critical_threshold' => 'decimal:2',
        'approved_at' => 'datetime',
        'last_calculated_at' => 'datetime',
    ];

    /**
     * العلاقات
     */
    
    // السنة المالية
    public function fiscalYear(): BelongsTo
    {
        return $this->belongsTo(FiscalYear::class);
    }

    // مركز التكلفة
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    // الحساب
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    // منشئ الميزانية
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // معتمد الميزانية
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Scopes
     */
    
    // الميزانيات النشطة
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // الميزانيات المعتمدة
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    // الميزانيات حسب مركز التكلفة
    public function scopeForCostCenter($query, $costCenterId)
    {
        return $query->where('cost_center_id', $costCenterId);
    }

    // الميزانيات حسب الحساب
    public function scopeForAccount($query, $accountId)
    {
        return $query->where('account_id', $accountId);
    }

    // الميزانيات حسب الفترة
    public function scopeForPeriod($query, $startDate, $endDate)
    {
        return $query->where('start_date', '<=', $endDate)
                    ->where('end_date', '>=', $startDate);
    }

    // الميزانيات المتجاوزة للحد التحذيري
    public function scopeOverWarningThreshold($query)
    {
        return $query->whereRaw('variance_percentage >= warning_threshold');
    }

    // الميزانيات المتجاوزة للحد الحرج
    public function scopeOverCriticalThreshold($query)
    {
        return $query->whereRaw('variance_percentage >= critical_threshold');
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // المبلغ المتبقي
    public function getRemainingAmountAttribute()
    {
        return $this->budgeted_amount - $this->actual_amount;
    }

    // نسبة الاستهلاك
    public function getConsumptionPercentageAttribute()
    {
        if ($this->budgeted_amount == 0) {
            return 0;
        }
        
        return round(($this->actual_amount / $this->budgeted_amount) * 100, 2);
    }

    // حالة التنبيه
    public function getAlertStatusAttribute()
    {
        $percentage = $this->consumption_percentage;
        
        if ($percentage >= $this->critical_threshold) {
            return 'critical';
        } elseif ($percentage >= $this->warning_threshold) {
            return 'warning';
        } elseif ($percentage >= 80) {
            return 'info';
        } else {
            return 'success';
        }
    }

    // لون التنبيه
    public function getAlertColorAttribute()
    {
        return match($this->alert_status) {
            'critical' => 'danger',
            'warning' => 'warning',
            'info' => 'info',
            'success' => 'success',
            default => 'secondary'
        };
    }

    // رسالة التنبيه
    public function getAlertMessageAttribute()
    {
        $percentage = $this->consumption_percentage;
        
        return match($this->alert_status) {
            'critical' => "تم تجاوز الحد الحرج ({$this->critical_threshold}%) - الاستهلاك: {$percentage}%",
            'warning' => "تم تجاوز حد التحذير ({$this->warning_threshold}%) - الاستهلاك: {$percentage}%",
            'info' => "اقتراب من الحد المحدد - الاستهلاك: {$percentage}%",
            'success' => "ضمن الحدود المطلوبة - الاستهلاك: {$percentage}%",
            default => "حالة غير محددة"
        };
    }

    // الفترة المتبقية بالأيام
    public function getRemainingDaysAttribute()
    {
        return max(0, now()->diffInDays($this->end_date, false));
    }

    // نسبة الفترة المنقضية
    public function getPeriodElapsedPercentageAttribute()
    {
        $totalDays = $this->start_date->diffInDays($this->end_date);
        $elapsedDays = $this->start_date->diffInDays(now());
        
        if ($totalDays == 0) {
            return 100;
        }
        
        return min(100, round(($elapsedDays / $totalDays) * 100, 2));
    }

    // المعدل اليومي المتوقع
    public function getDailyBurnRateAttribute()
    {
        $totalDays = $this->start_date->diffInDays($this->end_date) + 1;
        
        if ($totalDays == 0) {
            return 0;
        }
        
        return round($this->budgeted_amount / $totalDays, 2);
    }

    // المعدل اليومي الفعلي
    public function getActualDailyBurnRateAttribute()
    {
        $elapsedDays = $this->start_date->diffInDays(now()) + 1;
        
        if ($elapsedDays == 0) {
            return 0;
        }
        
        return round($this->actual_amount / $elapsedDays, 2);
    }

    /**
     * الدوال المساعدة
     */
    
    // حساب المبلغ الفعلي
    public function calculateActualAmount()
    {
        if (!$this->auto_calculate) {
            return $this->actual_amount;
        }
        
        $actualAmount = 0;
        
        // حساب من خطوط القيود المحاسبية
        if ($this->account_id) {
            $actualAmount = JournalEntryLine::where('account_id', $this->account_id)
                ->whereHas('journalEntry', function ($query) {
                    $query->where('is_approved', true)
                          ->whereBetween('date', [$this->start_date, $this->end_date]);
                    
                    if ($this->cost_center_id) {
                        $query->where('cost_center_id', $this->cost_center_id);
                    }
                })
                ->sum('debit_amount') - JournalEntryLine::where('account_id', $this->account_id)
                ->whereHas('journalEntry', function ($query) {
                    $query->where('is_approved', true)
                          ->whereBetween('date', [$this->start_date, $this->end_date]);
                    
                    if ($this->cost_center_id) {
                        $query->where('cost_center_id', $this->cost_center_id);
                    }
                })
                ->sum('credit_amount');
        } elseif ($this->cost_center_id) {
            // حساب من مركز التكلفة
            $actualAmount = JournalEntryLine::where('cost_center_id', $this->cost_center_id)
                ->whereHas('journalEntry', function ($query) {
                    $query->where('is_approved', true)
                          ->whereBetween('date', [$this->start_date, $this->end_date]);
                })
                ->whereHas('account', function ($query) {
                    $query->where('type', 'expense');
                })
                ->sum('debit_amount');
        }
        
        // تحديث المبلغ الفعلي
        $this->actual_amount = abs($actualAmount);
        $this->calculateVariance();
        $this->last_calculated_at = now();
        $this->save();
        
        return $this->actual_amount;
    }

    // حساب الانحراف
    public function calculateVariance()
    {
        $this->variance_amount = $this->actual_amount - $this->budgeted_amount;
        
        if ($this->budgeted_amount != 0) {
            $this->variance_percentage = round(($this->variance_amount / $this->budgeted_amount) * 100, 2);
        } else {
            $this->variance_percentage = 0;
        }
    }

    // اعتماد الميزانية
    public function approve($approvedBy = null)
    {
        $this->status = 'approved';
        $this->approved_by = $approvedBy ?? auth()->id();
        $this->approved_at = now();
        $this->save();
        
        return true;
    }

    // تفعيل الميزانية
    public function activate()
    {
        if ($this->status !== 'approved') {
            throw new \Exception('يجب اعتماد الميزانية أولاً');
        }
        
        $this->status = 'active';
        $this->save();
        
        return true;
    }

    // إغلاق الميزانية
    public function close()
    {
        $this->calculateActualAmount();
        $this->status = 'closed';
        $this->save();
        
        return true;
    }

    // التحقق من إمكانية التعديل
    public function canBeModified()
    {
        return in_array($this->status, ['draft', 'approved']) && $this->is_active;
    }

    // إنشاء التفصيل الشهري
    public function generateMonthlyBreakdown()
    {
        $months = [];
        $current = $this->start_date->copy()->startOfMonth();
        $end = $this->end_date->copy()->endOfMonth();
        
        $totalMonths = $current->diffInMonths($end) + 1;
        $monthlyAmount = round($this->budgeted_amount / $totalMonths, 2);
        
        while ($current->lte($end)) {
            $months[$current->format('Y-m')] = [
                'month' => $current->format('Y-m'),
                'budgeted' => $monthlyAmount,
                'actual' => 0,
                'variance' => 0,
            ];
            
            $current->addMonth();
        }
        
        $this->monthly_breakdown = $months;
        $this->save();
        
        return $months;
    }

    // تحديث التفصيل الشهري
    public function updateMonthlyBreakdown()
    {
        if (!$this->monthly_breakdown) {
            $this->generateMonthlyBreakdown();
        }
        
        $breakdown = $this->monthly_breakdown;
        
        foreach ($breakdown as $month => &$data) {
            $monthStart = Carbon::createFromFormat('Y-m', $month)->startOfMonth();
            $monthEnd = Carbon::createFromFormat('Y-m', $month)->endOfMonth();
            
            $actualAmount = 0;
            
            if ($this->account_id) {
                $actualAmount = JournalEntryLine::where('account_id', $this->account_id)
                    ->whereHas('journalEntry', function ($query) use ($monthStart, $monthEnd) {
                        $query->where('is_approved', true)
                              ->whereBetween('date', [$monthStart, $monthEnd]);
                        
                        if ($this->cost_center_id) {
                            $query->where('cost_center_id', $this->cost_center_id);
                        }
                    })
                    ->sum('debit_amount');
            }
            
            $data['actual'] = abs($actualAmount);
            $data['variance'] = $data['actual'] - $data['budgeted'];
        }
        
        $this->monthly_breakdown = $breakdown;
        $this->save();
        
        return $breakdown;
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        // عند الإنشاء، توليد رمز تلقائي إذا لم يتم تحديده
        static::creating(function ($budget) {
            if (empty($budget->code)) {
                $budget->code = 'BUD' . now()->year . str_pad(
                    static::whereYear('created_at', now()->year)->count() + 1,
                    4,
                    '0',
                    STR_PAD_LEFT
                );
            }
        });
        
        // عند التحديث، إعادة حساب الانحراف
        static::updating(function ($budget) {
            if ($budget->isDirty(['budgeted_amount', 'actual_amount'])) {
                $budget->calculateVariance();
            }
        });
    }
}