<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BankTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'transaction_number',
        'transaction_date',
        'bank_account_id',
        'type',
        'amount',
        'reference',
        'description',
        'payee_payer',
        'check_number',
        'status',
        'cleared_date',
        'journal_entry_id',
        'created_by',
        'attachments'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'cleared_date' => 'date',
        'amount' => 'decimal:2',
        'attachments' => 'array'
    ];

    // Relations
    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function journalEntry(): BelongsTo
    {
        return $this->belongsTo(JournalEntry::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scopes
    public function scopeCleared($query)
    {
        return $query->where('status', 'cleared');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeByPeriod($query, $startDate, $endDate)
    {
        return $query->whereBetween('transaction_date', [$startDate, $endDate]);
    }

    // Helper methods
    public static function generateTransactionNumber($type)
    {
        $prefixes = [
            'deposit' => 'BD',
            'withdrawal' => 'BW',
            'transfer' => 'BT',
            'fee' => 'BF',
            'interest' => 'BI'
        ];

        $prefix = $prefixes[$type] ?? 'BT';
        $lastTransaction = self::where('type', $type)
            ->whereYear('created_at', date('Y'))
            ->orderBy('id', 'desc')
            ->first();

        $nextNumber = $lastTransaction ? (int)substr($lastTransaction->transaction_number, -4) + 1 : 1;
        
        return $prefix . date('Y') . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
    }

    public function getFormattedAmountAttribute()
    {
        return number_format($this->amount, 2) . ' ' . $this->bankAccount->currency;
    }

    public function getTypeTextAttribute()
    {
        $types = [
            'deposit' => 'إيداع',
            'withdrawal' => 'سحب',
            'transfer' => 'تحويل',
            'fee' => 'رسوم',
            'interest' => 'فوائد'
        ];

        return $types[$this->type] ?? $this->type;
    }

    public function getStatusTextAttribute()
    {
        $statuses = [
            'pending' => 'معلق',
            'cleared' => 'مسدد',
            'cancelled' => 'ملغي'
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    public function clearTransaction()
    {
        if ($this->status !== 'pending') {
            throw new \Exception('Only pending transactions can be cleared');
        }

        $this->update([
            'status' => 'cleared',
            'cleared_date' => now()
        ]);

        // Update bank account balance
        if (in_array($this->type, ['deposit', 'interest'])) {
            $this->bankAccount->updateBalance($this->amount, 'deposit');
        } else {
            $this->bankAccount->updateBalance($this->amount, 'withdrawal');
        }

        // Post journal entry if exists
        if ($this->journal_entry_id && $this->journalEntry->status === 'draft') {
            $this->journalEntry->post($this->created_by);
        }
    }

    public function createJournalEntry()
    {
        if ($this->journal_entry_id) {
            return $this->journalEntry;
        }

        $journalEntry = JournalEntry::create([
            'entry_number' => JournalEntry::generateEntryNumber(),
            'entry_date' => $this->transaction_date,
            'reference' => $this->transaction_number,
            'description' => $this->description,
            'type' => 'auto',
            'status' => 'draft',
            'total_debit' => $this->amount,
            'total_credit' => $this->amount,
            'created_by' => $this->created_by
        ]);

        if (in_array($this->type, ['deposit', 'interest'])) {
            // Debit Bank Account
            $journalEntry->lines()->create([
                'account_id' => $this->bankAccount->account_id,
                'description' => $this->description,
                'debit_amount' => $this->amount,
                'credit_amount' => 0,
                'reference' => $this->reference
            ]);
        } else {
            // Credit Bank Account
            $journalEntry->lines()->create([
                'account_id' => $this->bankAccount->account_id,
                'description' => $this->description,
                'debit_amount' => 0,
                'credit_amount' => $this->amount,
                'reference' => $this->reference
            ]);
        }

        $this->update(['journal_entry_id' => $journalEntry->id]);

        return $journalEntry;
    }
}