<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class BankAccount extends Model
{
    use HasFactory;

    protected $fillable = [
        'account_id',
        'bank_name',
        'account_number',
        'iban',
        'swift_code',
        'branch',
        'currency',
        'opening_balance',
        'current_balance',
        'is_active',
        'notes'
    ];

    protected $casts = [
        'opening_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'is_active' => 'boolean'
    ];

    // Relations
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(BankTransaction::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByCurrency($query, $currency)
    {
        return $query->where('currency', $currency);
    }

    // Helper methods
    public function getFullAccountNameAttribute()
    {
        return $this->bank_name . ' - ' . $this->account_number;
    }

    public function updateBalance($amount, $type)
    {
        if ($type === 'deposit') {
            $this->current_balance += $amount;
        } else {
            $this->current_balance -= $amount;
        }
        $this->save();

        // Also update the linked GL account
        if ($this->account) {
            $this->account->updateBalance($amount, $type === 'deposit' ? 'debit' : 'credit');
        }
    }

    public function getBalanceForPeriod($startDate, $endDate)
    {
        $deposits = $this->transactions()
            ->whereIn('type', ['deposit', 'interest'])
            ->whereBetween('transaction_date', [$startDate, $endDate])
            ->where('status', 'cleared')
            ->sum('amount');

        $withdrawals = $this->transactions()
            ->whereIn('type', ['withdrawal', 'fee', 'transfer'])
            ->whereBetween('transaction_date', [$startDate, $endDate])
            ->where('status', 'cleared')
            ->sum('amount');

        return $this->opening_balance + $deposits - $withdrawals;
    }

    public function getFormattedBalanceAttribute()
    {
        return number_format($this->current_balance, 2) . ' ' . $this->currency;
    }

    public function getClearedBalance()
    {
        $clearedTransactions = $this->transactions()->where('status', 'cleared')->get();
        
        $balance = $this->opening_balance;
        foreach ($clearedTransactions as $transaction) {
            if (in_array($transaction->type, ['deposit', 'interest'])) {
                $balance += $transaction->amount;
            } else {
                $balance -= $transaction->amount;
            }
        }
        
        return $balance;
    }

    public function getUnClearedBalance()
    {
        return $this->current_balance - $this->getClearedBalance();
    }
}