<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class AuditLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'event',
        'auditable_type',
        'auditable_id',
        'user_id',
        'user_type',
        'old_values',
        'new_values',
        'url',
        'ip_address',
        'user_agent',
        'description',
        'tags',
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
        'tags' => 'array',
        'created_at' => 'datetime',
    ];

    // إزالة updated_at لأن هذا الجدول للتسجيل فقط
    public $timestamps = false;

    protected $dates = ['created_at'];

    /**
     * العلاقات
     */
    
    // المستخدم الذي قام بالعملية
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // النموذج المُراجع (polymorphic)
    public function auditable(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Scopes
     */
    
    // تصفية حسب النموذج
    public function scopeForModel($query, $modelType, $modelId = null)
    {
        $query->where('auditable_type', $modelType);
        
        if ($modelId) {
            $query->where('auditable_id', $modelId);
        }
        
        return $query;
    }

    // تصفية حسب المستخدم
    public function scopeByUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    // تصفية حسب نوع الحدث
    public function scopeByEvent($query, $event)
    {
        return $query->where('event', $event);
    }

    // تصفية حسب الفترة الزمنية
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    // تصفية حسب العلامات
    public function scopeWithTag($query, $tag)
    {
        return $query->whereJsonContains('tags', $tag);
    }

    // الأحداث المحاسبية
    public function scopeAccountingEvents($query)
    {
        return $query->whereIn('auditable_type', [
            'App\Models\Account',
            'App\Models\JournalEntry',
            'App\Models\JournalEntryLine',
            'App\Models\CashTransaction',
            'App\Models\Budget',
            'App\Models\CostCenter',
            'App\Models\FiscalYear',
        ]);
    }

    // العمليات الحساسة
    public function scopeCriticalEvents($query)
    {
        return $query->whereIn('event', [
            'deleted',
            'approved',
            'closed',
            'reopened',
            'balance_adjusted',
        ]);
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // وصف العملية باللغة العربية
    public function getEventDescriptionAttribute()
    {
        $descriptions = [
            'created' => 'إنشاء',
            'updated' => 'تحديث',
            'deleted' => 'حذف',
            'approved' => 'اعتماد',
            'rejected' => 'رفض',
            'closed' => 'إغلاق',
            'reopened' => 'إعادة فتح',
            'activated' => 'تفعيل',
            'deactivated' => 'إلغاء تفعيل',
            'balance_adjusted' => 'تعديل رصيد',
            'exported' => 'تصدير',
            'imported' => 'استيراد',
            'calculated' => 'حساب',
            'reconciled' => 'مطابقة',
            'unreconciled' => 'إلغاء مطابقة',
        ];

        return $descriptions[$this->event] ?? $this->event;
    }

    // نوع النموذج باللغة العربية
    public function getModelTypeDescriptionAttribute()
    {
        $descriptions = [
            'App\Models\Account' => 'حساب',
            'App\Models\JournalEntry' => 'قيد محاسبي',
            'App\Models\JournalEntryLine' => 'سطر قيد',
            'App\Models\CashTransaction' => 'معاملة نقدية',
            'App\Models\BankTransaction' => 'معاملة بنكية',
            'App\Models\BankAccount' => 'حساب بنكي',
            'App\Models\CostCenter' => 'مركز تكلفة',
            'App\Models\Budget' => 'ميزانية',
            'App\Models\FiscalYear' => 'سنة مالية',
            'App\Models\User' => 'مستخدم',
        ];

        return $descriptions[$this->auditable_type] ?? $this->auditable_type;
    }

    // الحصول على الملخص
    public function getSummaryAttribute()
    {
        $summary = $this->event_description . ' ' . $this->model_type_description;
        
        if ($this->auditable) {
            $name = $this->auditable->name ?? 
                   $this->auditable->reference ?? 
                   $this->auditable->code ?? 
                   '#' . $this->auditable->id;
            $summary .= ': ' . $name;
        }
        
        return $summary;
    }

    // الحصول على التغييرات المهمة
    public function getImportantChangesAttribute()
    {
        $importantFields = [
            'name', 'amount', 'balance', 'status', 'is_approved', 
            'is_active', 'is_closed', 'total_amount'
        ];
        
        $changes = [];
        
        if ($this->old_values && $this->new_values) {
            foreach ($importantFields as $field) {
                if (isset($this->old_values[$field]) && isset($this->new_values[$field])) {
                    if ($this->old_values[$field] != $this->new_values[$field]) {
                        $changes[$field] = [
                            'old' => $this->old_values[$field],
                            'new' => $this->new_values[$field]
                        ];
                    }
                }
            }
        }
        
        return $changes;
    }

    /**
     * الدوال المساعدة
     */
    
    // تسجيل حدث مراجعة
    public static function logEvent($event, $model, $description = null, $tags = [])
    {
        $oldValues = null;
        $newValues = null;
        
        // الحصول على القيم القديمة والجديدة للتحديث
        if ($event === 'updated' && $model->isDirty()) {
            $oldValues = $model->getOriginal();
            $newValues = $model->getDirty();
        } elseif ($event === 'created') {
            $newValues = $model->getAttributes();
        }
        
        return static::create([
            'event' => $event,
            'auditable_type' => get_class($model),
            'auditable_id' => $model->id,
            'user_id' => auth()->id(),
            'user_type' => auth()->user() ? get_class(auth()->user()) : null,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'url' => request()->fullUrl(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'description' => $description,
            'tags' => $tags,
            'created_at' => now(),
        ]);
    }

    // تسجيل حدث مخصص
    public static function logCustomEvent($event, $description, $data = [], $tags = [])
    {
        return static::create([
            'event' => $event,
            'auditable_type' => 'Custom',
            'auditable_id' => null,
            'user_id' => auth()->id(),
            'user_type' => auth()->user() ? get_class(auth()->user()) : null,
            'old_values' => null,
            'new_values' => $data,
            'url' => request()->fullUrl(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'description' => $description,
            'tags' => $tags,
            'created_at' => now(),
        ]);
    }

    // الحصول على إحصائيات المراجعة
    public static function getAuditStats($startDate = null, $endDate = null)
    {
        $query = static::query();
        
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }
        
        return [
            'total_events' => $query->count(),
            'events_by_type' => $query->groupBy('event')
                ->selectRaw('event, count(*) as count')
                ->pluck('count', 'event'),
            'events_by_model' => $query->groupBy('auditable_type')
                ->selectRaw('auditable_type, count(*) as count')
                ->pluck('count', 'auditable_type'),
            'events_by_user' => $query->whereNotNull('user_id')
                ->groupBy('user_id')
                ->selectRaw('user_id, count(*) as count')
                ->with('user:id,name')
                ->get(),
            'critical_events' => $query->whereIn('event', [
                'deleted', 'approved', 'closed', 'balance_adjusted'
            ])->count(),
        ];
    }

    // تنظيف السجلات القديمة
    public static function cleanupOldLogs($daysToKeep = 365)
    {
        $cutoffDate = now()->subDays($daysToKeep);
        
        return static::where('created_at', '<', $cutoffDate)->delete();
    }
}