<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class AccountingSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'group',
        'type',
        'label',
        'description',
        'validation_rules',
        'options',
        'default_value',
        'is_public',
        'is_editable',
        'requires_restart',
        'sort_order',
    ];

    protected $casts = [
        'options' => 'array',
        'is_public' => 'boolean',
        'is_editable' => 'boolean',
        'requires_restart' => 'boolean',
    ];

    /**
     * Scopes
     */
    
    // الإعدادات حسب المجموعة
    public function scopeByGroup($query, $group)
    {
        return $query->where('group', $group);
    }

    // الإعدادات العامة
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    // الإعدادات القابلة للتعديل
    public function scopeEditable($query)
    {
        return $query->where('is_editable', true);
    }

    // ترتيب الإعدادات
    public function scopeOrdered($query)
    {
        return $query->orderBy('group')->orderBy('sort_order')->orderBy('label');
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // القيمة المُحولة حسب النوع
    public function getTypedValueAttribute()
    {
        return $this->castValue($this->value);
    }

    // القيمة الافتراضية المُحولة
    public function getTypedDefaultValueAttribute()
    {
        return $this->castValue($this->default_value);
    }

    // قواعد التحقق كمصفوفة
    public function getValidationRulesArrayAttribute()
    {
        if (empty($this->validation_rules)) {
            return [];
        }

        return explode('|', $this->validation_rules);
    }

    /**
     * الدوال المساعدة
     */
    
    // تحويل القيمة حسب النوع
    private function castValue($value)
    {
        if ($value === null) {
            return null;
        }

        return match($this->type) {
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'integer' => (int) $value,
            'float', 'decimal' => (float) $value,
            'array', 'json' => is_string($value) ? json_decode($value, true) : $value,
            'date' => \Carbon\Carbon::parse($value),
            'datetime' => \Carbon\Carbon::parse($value),
            default => (string) $value
        };
    }

    // تحديث القيمة
    public function updateValue($newValue)
    {
        // التحقق من إمكانية التعديل
        if (!$this->is_editable) {
            throw new \Exception('هذا الإعداد غير قابل للتعديل');
        }

        // التحقق من صحة القيمة
        if (!$this->validateValue($newValue)) {
            throw new \Exception('القيمة المدخلة غير صحيحة');
        }

        // تحويل القيمة للنوع المناسب للحفظ
        $valueToSave = $this->prepareValueForStorage($newValue);

        $this->update(['value' => $valueToSave]);

        // مسح الكاش
        $this->clearCache();

        // تسجيل التغيير
        AuditLog::logEvent('updated', $this, "تم تغيير إعداد {$this->label}");

        return true;
    }

    // التحقق من صحة القيمة
    private function validateValue($value)
    {
        $rules = $this->validation_rules_array;
        
        if (empty($rules)) {
            return true;
        }

        $validator = \Validator::make(
            ['value' => $value],
            ['value' => $rules]
        );

        return $validator->passes();
    }

    // تحضير القيمة للحفظ
    private function prepareValueForStorage($value)
    {
        return match($this->type) {
            'array', 'json' => is_array($value) ? json_encode($value) : $value,
            'boolean' => $value ? '1' : '0',
            'date', 'datetime' => $value instanceof \Carbon\Carbon ? $value->toDateTimeString() : $value,
            default => (string) $value
        };
    }

    // مسح الكاش
    private function clearCache()
    {
        Cache::forget("accounting_setting_{$this->key}");
        Cache::forget("accounting_settings_group_{$this->group}");
        Cache::forget('accounting_settings_all');
    }

    /**
     * الدوال الثابتة
     */
    
    // الحصول على قيمة إعداد
    public static function get($key, $default = null)
    {
        return Cache::remember("accounting_setting_{$key}", 3600, function () use ($key, $default) {
            $setting = static::where('key', $key)->first();
            
            if (!$setting) {
                return $default;
            }

            return $setting->typed_value ?? $setting->typed_default_value ?? $default;
        });
    }

    // تحديث قيمة إعداد
    public static function set($key, $value)
    {
        $setting = static::where('key', $key)->first();
        
        if (!$setting) {
            throw new \Exception("الإعداد {$key} غير موجود");
        }

        return $setting->updateValue($value);
    }

    // الحصول على جميع إعدادات مجموعة
    public static function getGroup($group)
    {
        return Cache::remember("accounting_settings_group_{$group}", 3600, function () use ($group) {
            return static::byGroup($group)
                ->ordered()
                ->get()
                ->mapWithKeys(function ($setting) {
                    return [$setting->key => $setting->typed_value ?? $setting->typed_default_value];
                });
        });
    }

    // الحصول على جميع الإعدادات
    public static function getAll()
    {
        return Cache::remember('accounting_settings_all', 3600, function () {
            return static::all()
                ->mapWithKeys(function ($setting) {
                    return [$setting->key => $setting->typed_value ?? $setting->typed_default_value];
                });
        });
    }

    // إنشاء إعداد جديد
    public static function create($data)
    {
        // التحقق من عدم وجود المفتاح
        if (static::where('key', $data['key'])->exists()) {
            throw new \Exception("الإعداد {$data['key']} موجود مسبقاً");
        }

        $setting = parent::create($data);
        
        // مسح الكاش
        Cache::forget('accounting_settings_all');
        Cache::forget("accounting_settings_group_{$setting->group}");

        return $setting;
    }

    // تحميل الإعدادات الافتراضية
    public static function loadDefaults()
    {
        $defaults = [
            // إعدادات عامة
            [
                'key' => 'company_name',
                'value' => 'شركة برايما للتقنية',
                'group' => 'general',
                'type' => 'text',
                'label' => 'اسم الشركة',
                'description' => 'الاسم الرسمي للشركة',
                'is_public' => true,
                'is_editable' => true,
                'sort_order' => 1,
            ],
            [
                'key' => 'currency',
                'value' => 'SAR',
                'group' => 'general',
                'type' => 'text',
                'label' => 'العملة',
                'description' => 'العملة الافتراضية للنظام',
                'options' => ['SAR', 'USD', 'EUR'],
                'is_public' => true,
                'is_editable' => true,
                'sort_order' => 2,
            ],
            [
                'key' => 'decimal_places',
                'value' => '2',
                'group' => 'general',
                'type' => 'integer',
                'label' => 'عدد الخانات العشرية',
                'description' => 'عدد الخانات العشرية للمبالغ',
                'validation_rules' => 'required|integer|min:0|max:4',
                'default_value' => '2',
                'is_public' => true,
                'is_editable' => true,
                'sort_order' => 3,
            ],
            
            // إعدادات القيود
            [
                'key' => 'require_approval',
                'value' => '1',
                'group' => 'journal_entries',
                'type' => 'boolean',
                'label' => 'يتطلب اعتماد',
                'description' => 'هل تتطلب القيود المحاسبية اعتماد',
                'default_value' => '1',
                'is_public' => false,
                'is_editable' => true,
                'sort_order' => 1,
            ],
            [
                'key' => 'auto_generate_reference',
                'value' => '1',
                'group' => 'journal_entries',
                'type' => 'boolean',
                'label' => 'توليد المرجع تلقائياً',
                'description' => 'توليد رقم المرجع تلقائياً للقيود',
                'default_value' => '1',
                'is_public' => false,
                'is_editable' => true,
                'sort_order' => 2,
            ],
            
            // إعدادات التقارير
            [
                'key' => 'default_date_range',
                'value' => 'current_month',
                'group' => 'reports',
                'type' => 'text',
                'label' => 'الفترة الافتراضية',
                'description' => 'الفترة الزمنية الافتراضية للتقارير',
                'options' => ['current_month', 'current_quarter', 'current_year'],
                'default_value' => 'current_month',
                'is_public' => true,
                'is_editable' => true,
                'sort_order' => 1,
            ],
        ];

        foreach ($defaults as $defaultSetting) {
            static::updateOrCreate(
                ['key' => $defaultSetting['key']],
                $defaultSetting
            );
        }
    }

    // مسح جميع الكاش
    public static function clearAllCache()
    {
        $groups = static::distinct('group')->pluck('group');
        
        foreach ($groups as $group) {
            Cache::forget("accounting_settings_group_{$group}");
        }
        
        $keys = static::pluck('key');
        foreach ($keys as $key) {
            Cache::forget("accounting_setting_{$key}");
        }
        
        Cache::forget('accounting_settings_all');
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        // مسح الكاش عند التحديث
        static::saved(function ($setting) {
            $setting->clearCache();
        });

        static::deleted(function ($setting) {
            $setting->clearCache();
        });
    }
}