<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class AccountBalance extends Model
{
    use HasFactory;

    protected $fillable = [
        'account_id',
        'fiscal_year_id',
        'cost_center_id',
        'balance_date',
        'balance_type',
        'debit_balance',
        'credit_balance',
        'net_balance',
        'period_debits',
        'period_credits',
        'transaction_count',
        'is_calculated',
        'is_verified',
        'verified_by',
        'verified_at',
        'notes',
        'breakdown',
    ];

    protected $casts = [
        'balance_date' => 'date',
        'debit_balance' => 'decimal:2',
        'credit_balance' => 'decimal:2',
        'net_balance' => 'decimal:2',
        'period_debits' => 'decimal:2',
        'period_credits' => 'decimal:2',
        'is_calculated' => 'boolean',
        'is_verified' => 'boolean',
        'verified_at' => 'datetime',
        'breakdown' => 'array',
    ];

    /**
     * العلاقات
     */
    
    // الحساب
    public function account(): BelongsTo
    {
        return $this->belongsTo(Account::class);
    }

    // السنة المالية
    public function fiscalYear(): BelongsTo
    {
        return $this->belongsTo(FiscalYear::class);
    }

    // مركز التكلفة
    public function costCenter(): BelongsTo
    {
        return $this->belongsTo(CostCenter::class);
    }

    // المستخدم الذي تحقق من الرصيد
    public function verifier(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    /**
     * Scopes
     */
    
    // رصيد افتتاحي
    public function scopeOpening($query)
    {
        return $query->where('balance_type', 'opening');
    }

    // رصيد ختامي
    public function scopeClosing($query)
    {
        return $query->where('balance_type', 'closing');
    }

    // رصيد نهاية فترة
    public function scopePeriodEnd($query)
    {
        return $query->where('balance_type', 'period_end');
    }

    // رصيد تعديل
    public function scopeAdjustment($query)
    {
        return $query->where('balance_type', 'adjustment');
    }

    // الأرصدة المحسوبة
    public function scopeCalculated($query)
    {
        return $query->where('is_calculated', true);
    }

    // الأرصدة المتحقق منها
    public function scopeVerified($query)
    {
        return $query->where('is_verified', true);
    }

    // فلترة حسب الحساب
    public function scopeForAccount($query, $accountId)
    {
        return $query->where('account_id', $accountId);
    }

    // فلترة حسب السنة المالية
    public function scopeForFiscalYear($query, $fiscalYearId)
    {
        return $query->where('fiscal_year_id', $fiscalYearId);
    }

    // فلترة حسب مركز التكلفة
    public function scopeForCostCenter($query, $costCenterId)
    {
        return $query->where('cost_center_id', $costCenterId);
    }

    // فلترة حسب التاريخ
    public function scopeForDate($query, $date)
    {
        return $query->where('balance_date', $date);
    }

    // فلترة حسب الفترة
    public function scopeBetweenDates($query, $startDate, $endDate)
    {
        return $query->whereBetween('balance_date', [$startDate, $endDate]);
    }

    /**
     * الأكسسورز والمطافرز
     */
    
    // الرصيد المطلق
    public function getAbsoluteBalanceAttribute()
    {
        return abs($this->net_balance);
    }

    // نوع الرصيد (مدين/دائن)
    public function getBalanceNatureAttribute()
    {
        if ($this->net_balance > 0) {
            return 'مدين';
        } elseif ($this->net_balance < 0) {
            return 'دائن';
        } else {
            return 'متوازن';
        }
    }

    // حالة التحقق
    public function getVerificationStatusAttribute()
    {
        if ($this->is_verified) {
            return 'متحقق منه';
        } elseif ($this->is_calculated) {
            return 'محسوب';
        } else {
            return 'يدوي';
        }
    }

    // نسبة التغيير من الفترة السابقة
    public function getChangePercentageAttribute()
    {
        $previousBalance = static::where('account_id', $this->account_id)
            ->where('fiscal_year_id', $this->fiscal_year_id)
            ->where('balance_date', '<', $this->balance_date)
            ->orderBy('balance_date', 'desc')
            ->first();

        if (!$previousBalance || $previousBalance->net_balance == 0) {
            return 0;
        }

        return round((($this->net_balance - $previousBalance->net_balance) / abs($previousBalance->net_balance)) * 100, 2);
    }

    // اتجاه التغيير
    public function getChangeDirectionAttribute()
    {
        $change = $this->change_percentage;
        
        if ($change > 0) {
            return 'زيادة';
        } elseif ($change < 0) {
            return 'نقصان';
        } else {
            return 'ثابت';
        }
    }

    // وصف نوع الرصيد
    public function getBalanceTypeDescriptionAttribute()
    {
        $descriptions = [
            'opening' => 'رصيد افتتاحي',
            'closing' => 'رصيد ختامي',
            'period_end' => 'رصيد نهاية فترة',
            'adjustment' => 'رصيد تعديل',
        ];

        return $descriptions[$this->balance_type] ?? $this->balance_type;
    }

    /**
     * الدوال المساعدة
     */
    
    // حساب الرصيد تلقائياً
    public function calculateBalance()
    {
        $account = $this->account;
        if (!$account) {
            return false;
        }

        // حساب حركات الفترة
        $query = JournalEntryLine::where('account_id', $this->account_id)
            ->whereHas('journalEntry', function ($q) {
                $q->where('is_approved', true);
                
                // تحديد نطاق التاريخ حسب نوع الرصيد
                switch ($this->balance_type) {
                    case 'opening':
                        $q->where('date', '<', $this->balance_date);
                        break;
                    case 'closing':
                    case 'period_end':
                        $q->where('date', '<=', $this->balance_date);
                        break;
                    case 'adjustment':
                        $q->where('date', '=', $this->balance_date);
                        break;
                }
                
                // فلترة حسب مركز التكلفة إذا كان محدد
                if ($this->cost_center_id) {
                    $q->where('cost_center_id', $this->cost_center_id);
                }
            });

        $this->period_debits = (clone $query)->sum('debit_amount');
        $this->period_credits = (clone $query)->sum('credit_amount');
        $this->transaction_count = (clone $query)->count();

        // حساب الأرصدة
        $this->debit_balance = max(0, $this->period_debits - $this->period_credits);
        $this->credit_balance = max(0, $this->period_credits - $this->period_debits);
        $this->net_balance = $this->period_debits - $this->period_credits;

        // تطبيق الطبيعة المحاسبية للحساب
        if (in_array($account->type, ['liability', 'equity', 'revenue'])) {
            $this->net_balance = -$this->net_balance;
        }

        $this->is_calculated = true;
        $this->save();

        return true;
    }

    // التحقق من الرصيد
    public function verify($verifiedBy = null)
    {
        $this->is_verified = true;
        $this->verified_by = $verifiedBy ?? auth()->id();
        $this->verified_at = now();
        $this->save();

        // تسجيل في سجل المراجعة
        AuditLog::logEvent('verified', $this, 'تم التحقق من رصيد الحساب');

        return true;
    }

    // إلغاء التحقق
    public function unverify()
    {
        $this->is_verified = false;
        $this->verified_by = null;
        $this->verified_at = null;
        $this->save();

        return true;
    }

    // تحديث التفصيل
    public function updateBreakdown($breakdown)
    {
        $this->breakdown = array_merge($this->breakdown ?? [], $breakdown);
        $this->save();

        return true;
    }

    // مقارنة مع رصيد آخر
    public function compareWith(AccountBalance $otherBalance)
    {
        return [
            'net_difference' => $this->net_balance - $otherBalance->net_balance,
            'debit_difference' => $this->debit_balance - $otherBalance->debit_balance,
            'credit_difference' => $this->credit_balance - $otherBalance->credit_balance,
            'transaction_difference' => $this->transaction_count - $otherBalance->transaction_count,
            'percentage_change' => $otherBalance->net_balance != 0 
                ? round((($this->net_balance - $otherBalance->net_balance) / abs($otherBalance->net_balance)) * 100, 2)
                : 0,
        ];
    }

    /**
     * الدوال الثابتة
     */
    
    // إنشاء رصيد افتتاحي
    public static function createOpeningBalance($accountId, $fiscalYearId, $amount, $costCenterId = null)
    {
        $fiscalYear = FiscalYear::find($fiscalYearId);
        if (!$fiscalYear) {
            throw new \Exception('السنة المالية غير موجودة');
        }

        return static::create([
            'account_id' => $accountId,
            'fiscal_year_id' => $fiscalYearId,
            'cost_center_id' => $costCenterId,
            'balance_date' => $fiscalYear->start_date,
            'balance_type' => 'opening',
            'debit_balance' => max(0, $amount),
            'credit_balance' => max(0, -$amount),
            'net_balance' => $amount,
            'period_debits' => 0,
            'period_credits' => 0,
            'transaction_count' => 0,
            'is_calculated' => false,
            'is_verified' => false,
        ]);
    }

    // إنشاء رصيد ختامي
    public static function createClosingBalance($accountId, $fiscalYearId, $costCenterId = null)
    {
        $fiscalYear = FiscalYear::find($fiscalYearId);
        if (!$fiscalYear) {
            throw new \Exception('السنة المالية غير موجودة');
        }

        $balance = static::create([
            'account_id' => $accountId,
            'fiscal_year_id' => $fiscalYearId,
            'cost_center_id' => $costCenterId,
            'balance_date' => $fiscalYear->end_date,
            'balance_type' => 'closing',
            'is_calculated' => true,
        ]);

        $balance->calculateBalance();
        return $balance;
    }

    // الحصول على آخر رصيد للحساب
    public static function getLatestBalance($accountId, $fiscalYearId = null, $costCenterId = null)
    {
        $query = static::where('account_id', $accountId);
        
        if ($fiscalYearId) {
            $query->where('fiscal_year_id', $fiscalYearId);
        }
        
        if ($costCenterId) {
            $query->where('cost_center_id', $costCenterId);
        }
        
        return $query->orderBy('balance_date', 'desc')->first();
    }

    // حساب جميع الأرصدة للسنة المالية
    public static function calculateAllBalances($fiscalYearId, $balanceType = 'period_end')
    {
        $fiscalYear = FiscalYear::find($fiscalYearId);
        if (!$fiscalYear) {
            throw new \Exception('السنة المالية غير موجودة');
        }

        $accounts = Account::where('is_active', true)->get();
        $calculated = 0;

        foreach ($accounts as $account) {
            $balance = static::updateOrCreate([
                'account_id' => $account->id,
                'fiscal_year_id' => $fiscalYearId,
                'balance_date' => $fiscalYear->end_date,
                'balance_type' => $balanceType,
            ]);

            $balance->calculateBalance();
            $calculated++;
        }

        return $calculated;
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();
        
        // عند الإنشاء، التحقق من عدم وجود رصيد مكرر
        static::creating(function ($balance) {
            $existing = static::where('account_id', $balance->account_id)
                ->where('fiscal_year_id', $balance->fiscal_year_id)
                ->where('balance_date', $balance->balance_date)
                ->where('balance_type', $balance->balance_type)
                ->where('cost_center_id', $balance->cost_center_id)
                ->first();

            if ($existing) {
                throw new \Exception('يوجد رصيد مسجل مسبقاً لنفس الحساب والفترة');
            }
        });

        // تسجيل العمليات في سجل المراجعة
        static::created(function ($balance) {
            AuditLog::logEvent('created', $balance, 'تم إنشاء رصيد حساب');
        });

        static::updated(function ($balance) {
            AuditLog::logEvent('updated', $balance, 'تم تحديث رصيد حساب');
        });

        static::deleted(function ($balance) {
            AuditLog::logEvent('deleted', $balance, 'تم حذف رصيد حساب');
        });
    }
}