<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Account extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'name_en',
        'type',
        'subtype',
        'parent_id',
        'level',
        'path',
        'debit_credit',
        'opening_balance',
        'current_balance',
        'is_active',
        'is_cash_account',
        'is_bank_account',
        'is_control_account',
        'tax_calculation_method',
        'applies_tax',
        'default_tax_rate',
        'description'
    ];

    protected $casts = [
        'opening_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'default_tax_rate' => 'decimal:2',
        'is_active' => 'boolean',
        'is_cash_account' => 'boolean',
        'is_bank_account' => 'boolean',
        'is_control_account' => 'boolean',
        'applies_tax' => 'boolean',
    ];

    // Relations
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(Account::class, 'parent_id');
    }

    public function journalEntryLines(): HasMany
    {
        return $this->hasMany(JournalEntryLine::class);
    }

    public function bankAccount(): HasMany
    {
        return $this->hasMany(BankAccount::class);
    }

    public function cashTransactions(): HasMany
    {
        return $this->hasMany(CashTransaction::class, 'cash_account_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeCashAccounts($query)
    {
        return $query->where('is_cash_account', true);
    }

    public function scopeBankAccounts($query)
    {
        return $query->where('is_bank_account', true);
    }

    // Helper methods
    public function getFullNameAttribute()
    {
        return $this->code . ' - ' . $this->name;
    }

    public function updateBalance($amount, $type = 'debit')
    {
        if ($this->debit_credit === 'debit') {
            if ($type === 'debit') {
                $this->current_balance += $amount;
            } else {
                $this->current_balance -= $amount;
            }
        } else {
            if ($type === 'credit') {
                $this->current_balance += $amount;
            } else {
                $this->current_balance -= $amount;
            }
        }
        $this->save();
    }

    public function getBalanceForPeriod($startDate, $endDate)
    {
        $totalDebits = $this->journalEntryLines()
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('entry_date', [$startDate, $endDate])
                      ->where('status', 'posted');
            })
            ->sum('debit_amount');

        $totalCredits = $this->journalEntryLines()
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->whereBetween('entry_date', [$startDate, $endDate])
                      ->where('status', 'posted');
            })
            ->sum('credit_amount');

        if ($this->debit_credit === 'debit') {
            return $this->opening_balance + $totalDebits - $totalCredits;
        } else {
            return $this->opening_balance + $totalCredits - $totalDebits;
        }
    }

    public static function getAccountsTree()
    {
        return self::with('children')
            ->whereNull('parent_id')
            ->orderBy('code')
            ->get();
    }

    public function getHierarchyAttribute()
    {
        $path = [];
        $account = $this;
        
        while ($account) {
            array_unshift($path, $account->name);
            $account = $account->parent;
        }
        
        return implode(' > ', $path);
    }

    /**
     * التحقق من كون الحساب يطبق الضريبة بطريقة شاملة
     */
    public function isTaxInclusive(): bool
    {
        return $this->applies_tax && $this->tax_calculation_method === 'inclusive';
    }

    /**
     * التحقق من كون الحساب يطبق الضريبة بطريقة غير شاملة
     */
    public function isTaxExclusive(): bool
    {
        return $this->applies_tax && $this->tax_calculation_method === 'exclusive';
    }

    /**
     * الحصول على معدل الضريبة الافتراضي للحساب
     */
    public function getDefaultTaxRate(): float
    {
        return $this->default_tax_rate ?? 15.0; // 15% كمعدل افتراضي في السعودية
    }

    /**
     * حساب مبلغ الضريبة بناءً على طريقة الحساب
     */
    public function calculateTaxAmount(float $amount, float $taxRate = null): array
    {
        if (!$this->applies_tax) {
            return [
                'tax_exclusive_amount' => $amount,
                'tax_amount' => 0,
                'tax_inclusive_amount' => $amount
            ];
        }

        $rate = $taxRate ?? $this->getDefaultTaxRate();
        
        if ($this->isTaxInclusive()) {
            // إذا كان الحساب شامل الضريبة، نحسب المبلغ بدون ضريبة من المبلغ الشامل
            $taxExclusiveAmount = $amount / (1 + ($rate / 100));
            $taxAmount = $amount - $taxExclusiveAmount;
            $taxInclusiveAmount = $amount;
        } else {
            // إذا كان الحساب غير شامل الضريبة، نضيف الضريبة
            $taxExclusiveAmount = $amount;
            $taxAmount = $amount * ($rate / 100);
            $taxInclusiveAmount = $amount + $taxAmount;
        }

        return [
            'tax_exclusive_amount' => round($taxExclusiveAmount, 2),
            'tax_amount' => round($taxAmount, 2),
            'tax_inclusive_amount' => round($taxInclusiveAmount, 2),
            'tax_rate' => $rate,
            'calculation_method' => $this->tax_calculation_method
        ];
    }

    /**
     * تحويل مبلغ من شامل إلى غير شامل الضريبة
     */
    public function convertInclusiveToExclusive(float $inclusiveAmount, float $taxRate = null): float
    {
        if (!$this->applies_tax) {
            return $inclusiveAmount;
        }

        $rate = $taxRate ?? $this->getDefaultTaxRate();
        return round($inclusiveAmount / (1 + ($rate / 100)), 2);
    }

    /**
     * تحويل مبلغ من غير شامل إلى شامل الضريبة
     */
    public function convertExclusiveToInclusive(float $exclusiveAmount, float $taxRate = null): float
    {
        if (!$this->applies_tax) {
            return $exclusiveAmount;
        }

        $rate = $taxRate ?? $this->getDefaultTaxRate();
        return round($exclusiveAmount * (1 + ($rate / 100)), 2);
    }

    /**
     * فلترة الحسابات شاملة الضريبة
     */
    public function scopeTaxInclusive($query)
    {
        return $query->where('applies_tax', true)->where('tax_calculation_method', 'inclusive');
    }

    /**
     * فلترة الحسابات غير شاملة الضريبة
     */
    public function scopeTaxExclusive($query)
    {
        return $query->where('applies_tax', true)->where('tax_calculation_method', 'exclusive');
    }

    /**
     * فلترة الحسابات التي تطبق الضريبة
     */
    public function scopeWithTax($query)
    {
        return $query->where('applies_tax', true);
    }

    /**
     * فلترة الحسابات التي لا تطبق الضريبة
     */
    public function scopeWithoutTax($query)
    {
        return $query->where('applies_tax', false);
    }

    /**
     * الحصول على وصف طريقة حساب الضريبة بالعربية
     */
    public function getTaxCalculationMethodNameAttribute(): string
    {
        $methods = [
            'inclusive' => 'شامل الضريبة',
            'exclusive' => 'غير شامل الضريبة'
        ];

        return $methods[$this->tax_calculation_method] ?? 'غير محدد';
    }

    /**
     * التحقق من توافق الحساب مع نوع الضريبة المطلوب
     */
    public function isCompatibleWithTaxType(string $taxType): bool
    {
        if (!$this->applies_tax) {
            return $taxType === 'none';
        }

        return $this->tax_calculation_method === $taxType;
    }
}