<?php

namespace App\Jobs;

use App\Models\Invoice;
use App\Models\ZatcaSettings;
use App\Services\ZatcaService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Notifications\ZatcaSyncCompletedNotification;

class SyncZatcaInvoicesJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $zatcaSettingsId;
    protected $invoiceIds;
    protected $userId;

    /**
     * Create a new job instance.
     */
    public function __construct($zatcaSettingsId, $invoiceIds = null, $userId = null)
    {
        $this->zatcaSettingsId = $zatcaSettingsId;
        $this->invoiceIds = $invoiceIds;
        $this->userId = $userId;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {
            $zatcaSettings = ZatcaSettings::find($this->zatcaSettingsId);
            if (!$zatcaSettings) {
                Log::error('ZATCA Settings not found for sync job', ['id' => $this->zatcaSettingsId]);
                return;
            }

            $zatcaService = new ZatcaService($zatcaSettings);
            
            // تحديد الفواتير للمزامنة
            if ($this->invoiceIds) {
                $invoices = Invoice::whereIn('id', $this->invoiceIds)->get();
            } else {
                $invoices = Invoice::where('company_id', $zatcaSettings->company_id)
                    ->whereNotNull('sent_to_zatca_at')
                    ->where('updated_at', '>', now()->subDays(7))
                    ->get();
            }

            $results = [
                'total' => $invoices->count(),
                'success' => 0,
                'failed' => 0,
                'errors' => []
            ];

            foreach ($invoices as $invoice) {
                try {
                    $zatcaService->checkInvoiceStatus($invoice);
                    $results['success']++;
                    
                    Log::info('Invoice synced successfully', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number
                    ]);

                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'error' => $e->getMessage()
                    ];

                    Log::error('Invoice sync failed', [
                        'invoice_id' => $invoice->id,
                        'error' => $e->getMessage()
                    ]);
                }

                // تأخير قصير لتجنب إغراق الخادم
                usleep(500000); // 0.5 ثانية
            }

            // تسجيل النتائج
            $zatcaSettings->logOperation('background_sync', $results);

            // إرسال إشعار بالنتائج إذا كان هناك مستخدم محدد
            if ($this->userId) {
                $user = \App\Models\User::find($this->userId);
                if ($user) {
                    $user->notify(new ZatcaSyncCompletedNotification($results));
                }
            }

        } catch (\Exception $e) {
            Log::error('ZATCA sync job failed', [
                'error' => $e->getMessage(),
                'zatca_settings_id' => $this->zatcaSettingsId
            ]);

            $this->fail($e);
        }
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error('ZATCA sync job failed completely', [
            'error' => $exception->getMessage(),
            'zatca_settings_id' => $this->zatcaSettingsId
        ]);
    }
}