<?php

namespace App\Imports;

use App\Models\Product;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithProgressBar;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Validation\Rule;

class ProductsImport implements ToModel, WithHeadingRow, WithValidation, SkipsOnError, SkipsOnFailure, WithBatchInserts, WithChunkReading, WithProgressBar
{
    use Importable, SkipsErrors, SkipsFailures;

    protected $importedCount = 0;
    protected $updatedCount = 0;
    protected $skippedCount = 0;
    protected $importSession;
    protected $previewMode = false;
    protected $previewData = [];

    public function __construct($previewMode = false, $importSession = null)
    {
        $this->previewMode = $previewMode;
        $this->importSession = $importSession ?? uniqid('import_');
        
        // Initialize cache for import session
        if (!$previewMode) {
            Cache::put("import_session_{$this->importSession}", [
                'started_at' => now(),
                'status' => 'processing',
                'progress' => 0,
                'imported' => 0,
                'updated' => 0,
                'skipped' => 0,
                'errors' => []
            ], 3600); // 1 hour
        }
    }

    /**
     * Enable preview mode
     */
    public function enablePreviewMode(): self
    {
        $this->previewMode = true;
        return $this;
    }

    /**
     * Get preview data
     */
    public function getPreviewData(): array
    {
        return $this->previewData;
    }

    /**
     * Update import progress in cache
     */
    protected function updateProgress()
    {
        if ($this->importSession && !$this->previewMode) {
            Cache::put("import_session_{$this->importSession}", [
                'started_at' => Cache::get("import_session_{$this->importSession}")['started_at'] ?? now(),
                'status' => 'processing',
                'progress' => $this->importedCount + $this->updatedCount + $this->skippedCount,
                'imported' => $this->importedCount,
                'updated' => $this->updatedCount,
                'skipped' => $this->skippedCount,
                'errors' => $this->failures()
            ], 3600);
        }
    }

    /**
     * @param array $row
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        try {
            $productData = $this->prepareProductData($row);
            
            // Preview mode - just collect data for preview
            if ($this->previewMode) {
                $this->previewData[] = [
                    'data' => $productData,
                    'status' => $this->validateProductData($productData),
                    'existing' => !empty($productData['sku']) ? Product::where('sku', $productData['sku'])->exists() : false
                ];
                return null;
            }
            
            // Skip if required fields are empty
            $validation = $this->validateProductData($productData);
            if (!$validation['valid']) {
                \Log::warning('تم تخطي صف بسبب نقص البيانات المطلوبة', [
                    'row_data' => $row,
                    'prepared_data' => $productData,
                    'validation_errors' => $validation['errors']
                ]);
                $this->skippedCount++;
                $this->updateProgress();
                return null;
            }

            // Use database transaction for data integrity
            DB::beginTransaction();
            
            try {
                // Check if product already exists by SKU
                $existingProduct = Product::where('sku', $productData['sku'])->first();
                
                if ($existingProduct) {
                    // Update existing product
                    $existingProduct->update($productData);
                    $this->updatedCount++;
                    \Log::info('تم تحديث منتج موجود', [
                        'sku' => $productData['sku'],
                        'id' => $existingProduct->id,
                        'changes' => $existingProduct->getChanges()
                    ]);
                } else {
                    // Create new product
                    $newProduct = Product::create($productData);
                    $this->importedCount++;
                    \Log::info('تم إنشاء منتج جديد', [
                        'sku' => $productData['sku'],
                        'id' => $newProduct->id
                    ]);
                }
                
                DB::commit();
                $this->updateProgress();
                
            } catch (\Exception $dbException) {
                DB::rollBack();
                throw $dbException;
            }
            
            return null; // Always return null to avoid batch insert conflicts
            
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('خطأ في استيراد المنتج: ' . $e->getMessage(), [
                'row_data' => $row,
                'error_message' => $e->getMessage(),
                'error_line' => $e->getLine(),
                'error_file' => $e->getFile(),
                'stack_trace' => $e->getTraceAsString()
            ]);
            $this->skippedCount++;
            $this->updateProgress();
            return null;
        }
    }

    /**
     * Validate product data
     */
    protected function validateProductData(array $productData): array
    {
        $errors = [];
        
        if (empty($productData['name'])) {
            $errors[] = 'اسم المنتج مطلوب';
        }
        
        if (empty($productData['sku'])) {
            $errors[] = 'رمز المنتج (SKU) مطلوب';
        }
        
        if (!isset($productData['price']) || $productData['price'] < 0) {
            $errors[] = 'سعر المنتج مطلوب ويجب أن يكون أكبر من أو يساوي الصفر';
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Prepare product data from row
     */
    private function prepareProductData(array $row): array
    {
        // Log the row headers for debugging
        \Log::debug('أعمدة الملف المستورد', [
            'headers' => array_keys($row),
            'sample_data' => array_slice($row, 0, 5)
        ]);

        // Try different possible column names (Arabic and English variations)
        // Including the actual column names found in the logs
        $data = [
            'name' => $this->getColumnValue($row, ['الاسم', 'اسم المنتج', 'name', 'product_name', 'alasm_alaarby']),
            'name_en' => $this->getColumnValue($row, ['الاسم_بالإنجليزية', 'الاسم بالإنجليزية', 'name_en', 'english_name', 'alasm_alanglyzy']),
            'description' => $this->getColumnValue($row, ['الوصف', 'وصف المنتج', 'description', 'alosf']),
            'sku' => $this->getColumnValue($row, ['الرمز_sku', 'الرمز (SKU)', 'sku', 'product_code', 'alrmz_sku']),
            'barcode' => $this->getColumnValue($row, ['الباركود', 'باركود', 'barcode', 'albarkod']),
            'type' => $this->parseType($this->getColumnValue($row, ['النوع', 'نوع المنتج', 'type', 'alfy']) ?? 'منتج'),
            'item_type' => $this->parseItemType($this->getColumnValue($row, ['نوع_العنصر', 'نوع العنصر', 'item_type', 'noaa_alaansr'])),
            'price' => (float) ($this->getColumnValue($row, ['السعر', 'سعر المنتج', 'price', 'alsaar']) ?? 0),
            'tax_rate' => (float) ($this->getColumnValue($row, ['معدل_الضريبة', 'معدل الضريبة (%)', 'tax_rate']) ?? 15),
            'is_taxable' => $this->parseBoolean($this->getColumnValue($row, ['خاضع_للضريبة', 'خاضع للضريبة', 'is_taxable']) ?? 'نعم'),
            'unit' => $this->getColumnValue($row, ['الوحدة', 'وحدة القياس', 'unit']) ?? 'قطعة',
            'stock_quantity' => (int) ($this->getColumnValue($row, ['كمية_المخزون', 'كمية المخزون', 'stock_quantity', 'alkmy_almtofr']) ?? 0),
            'min_stock' => (int) ($this->getColumnValue($row, ['الحد_الأدنى_للمخزون', 'الحد الأدنى للمخزون', 'min_stock', 'alhd_aladn_llkmy']) ?? 0),
            'preparation_time' => $this->getColumnValue($row, ['وقت_التحضير_دقيقة', 'وقت التحضير (دقيقة)', 'preparation_time']) ?
                (int) $this->getColumnValue($row, ['وقت_التحضير_دقيقة', 'وقت التحضير (دقيقة)', 'preparation_time']) : null,
            'calories' => $this->getColumnValue($row, ['السعرات_الحرارية', 'السعرات الحرارية', 'calories']) ?
                (int) $this->getColumnValue($row, ['السعرات_الحرارية', 'السعرات الحرارية', 'calories']) : null,
            'is_active' => $this->parseBoolean($this->getColumnValue($row, ['نشط', 'فعال', 'is_active', 'alhal']) ?? 'نعم'),
            'is_available' => $this->parseBoolean($this->getColumnValue($row, ['متوفر', 'متاح', 'is_available', 'mtah_llbyaa']) ?? 'نعم'),
            'is_featured' => $this->parseBoolean($this->getColumnValue($row, ['مميز', 'منتج مميز', 'is_featured', 'mntg_mmyz']) ?? 'لا'),
            'is_vegetarian' => $this->parseBoolean($this->getColumnValue($row, ['نباتي', 'is_vegetarian']) ?? 'لا'),
            'is_vegan' => $this->parseBoolean($this->getColumnValue($row, ['نباتي_صرف', 'نباتي صرف', 'is_vegan']) ?? 'لا'),
            'is_gluten_free' => $this->parseBoolean($this->getColumnValue($row, ['خالي_من_الجلوتين', 'خالي من الجلوتين', 'is_gluten_free']) ?? 'لا'),
            'is_spicy' => $this->parseBoolean($this->getColumnValue($row, ['حار', 'is_spicy']) ?? 'لا'),
        ];

        // Calculate price with tax
        if ($data['is_taxable']) {
            $data['price_with_tax'] = $data['price'] * (1 + ($data['tax_rate'] / 100));
        } else {
            $data['price_with_tax'] = $data['price'];
        }

        // IMPORTANT: Remove any ID field that might cause primary key conflicts
        // Never include id, created_at, or updated_at from Excel imports
        unset($data['id'], $data['created_at'], $data['updated_at']);
        
        // Also remove the Excel serial number column if it exists
        $serialNumberColumn = $this->getColumnValue($row, ['alrkm_altslsly', 'الرقم_التسلسلي', 'serial_number', 'id']);
        if ($serialNumberColumn !== null) {
            // This is just for logging, don't include it in data
            \Log::debug('تم تجاهل الرقم التسلسلي من Excel', ['serial' => $serialNumberColumn]);
        }

        return $data;
    }

    /**
     * Get column value by trying multiple possible column names
     */
    private function getColumnValue(array $row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            if (isset($row[$name]) && $row[$name] !== null && $row[$name] !== '') {
                return $row[$name];
            }
        }
        return null;
    }

    /**
     * Parse product type
     */
    private function parseType(string $type): string
    {
        $types = [
            'منتج' => 'product',
            'خدمة' => 'service',
            'product' => 'product',
            'service' => 'service'
        ];

        return $types[strtolower($type)] ?? 'product';
    }

    /**
     * Parse item type
     */
    private function parseItemType(?string $itemType): ?string
    {
        if (!$itemType) {
            return null;
        }

        $types = [
            'طعام' => 'food',
            'مشروب' => 'beverage',
            'حلوى' => 'dessert',
            'مقبلات' => 'appetizer',
            'طبق رئيسي' => 'main_course',
            'كومبو' => 'combo',
            'food' => 'food',
            'beverage' => 'beverage',
            'dessert' => 'dessert',
            'appetizer' => 'appetizer',
            'main_course' => 'main_course',
            'combo' => 'combo'
        ];

        return $types[strtolower($itemType)] ?? null;
    }

    /**
     * Parse boolean values
     */
    private function parseBoolean(?string $value): bool
    {
        if (!$value) {
            return false;
        }

        $trueValues = ['نعم', 'yes', 'true', '1', 'صحيح'];
        return in_array(strtolower(trim($value)), $trueValues);
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        // More flexible validation rules that support different column names
        // Including the actual column names found in the logs
        return [
            // Name variations
            'الاسم' => 'nullable|string|max:255',
            'اسم المنتج' => 'nullable|string|max:255',
            'name' => 'nullable|string|max:255',
            'product_name' => 'nullable|string|max:255',
            'alasm_alaarby' => 'nullable|string|max:255',
            
            // English name variations
            'الاسم_بالإنجليزية' => 'nullable|string|max:255',
            'الاسم بالإنجليزية' => 'nullable|string|max:255',
            'name_en' => 'nullable|string|max:255',
            'english_name' => 'nullable|string|max:255',
            'alasm_alanglyzy' => 'nullable|string|max:255',
            
            // SKU variations
            'الرمز_sku' => 'nullable|string|max:255',
            'الرمز (SKU)' => 'nullable|string|max:255',
            'sku' => 'nullable|string|max:255',
            'product_code' => 'nullable|string|max:255',
            'alrmz_sku' => 'nullable|string|max:255',
            
            // Price variations
            'السعر' => 'nullable|numeric|min:0',
            'سعر المنتج' => 'nullable|numeric|min:0',
            'price' => 'nullable|numeric|min:0',
            'alsaar' => 'nullable|numeric|min:0',
            
            // Tax rate variations
            'معدل_الضريبة' => 'nullable|numeric|min:0|max:100',
            'معدل الضريبة (%)' => 'nullable|numeric|min:0|max:100',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            
            // Other fields from logs
            'الباركود' => 'nullable|string|max:255',
            'باركود' => 'nullable|string|max:255',
            'barcode' => 'nullable|string|max:255',
            'albarkod' => 'nullable|numeric|min:0',
            'الوحدة' => 'nullable|string|max:50',
            'وحدة القياس' => 'nullable|string|max:50',
            'unit' => 'nullable|string|max:50',
            'كمية_المخزون' => 'nullable|integer|min:0',
            'كمية المخزون' => 'nullable|integer|min:0',
            'stock_quantity' => 'nullable|integer|min:0',
            'alkmy_almtofr' => 'nullable|integer|min:0',
            'الحد_الأدنى_للمخزون' => 'nullable|integer|min:0',
            'الحد الأدنى للمخزون' => 'nullable|integer|min:0',
            'min_stock' => 'nullable|integer|min:0',
            'alhd_aladn_llkmy' => 'nullable|integer|min:0',
            'وقت_التحضير_دقيقة' => 'nullable|integer|min:0',
            'وقت التحضير (دقيقة)' => 'nullable|integer|min:0',
            'preparation_time' => 'nullable|integer|min:0',
            'السعرات_الحرارية' => 'nullable|integer|min:0',
            'السعرات الحرارية' => 'nullable|integer|min:0',
            'calories' => 'nullable|integer|min:0',
            
            // Additional fields from logs
            'alrkm_altslsly' => 'nullable|integer|min:0',
            'noaa_alaansr' => 'nullable|string|max:50',
            'alfy' => 'nullable|string|max:100',
            'tklf_alohd' => 'nullable|numeric|min:0',
            'alozn_kgm' => 'nullable|numeric|min:0',
            'alosf' => 'nullable|string',
            'alhal' => 'nullable|string|max:50',
            'mtah_llbyaa' => 'nullable|string|max:50',
            'mntg_mmyz' => 'nullable|string|max:50',
            'sor_almntg' => 'nullable|string',
            'tarykh_alanshaaa' => 'nullable|date',
            'tarykh_althdyth' => 'nullable|date',
        ];
    }

    /**
     * @return array
     */
    public function customValidationMessages(): array
    {
        return [
            '*.max' => 'الحقل يجب أن يكون أقل من الحد المسموح',
            '*.numeric' => 'الحقل يجب أن يكون رقم',
            '*.min' => 'الحقل يجب أن يكون أكبر من أو يساوي الحد الأدنى',
            '*.integer' => 'الحقل يجب أن يكون عدد صحيح',
            '*.string' => 'الحقل يجب أن يكون نص',
        ];
    }

    /**
     * @return int
     */
    public function batchSize(): int
    {
        return 100;
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 100;
    }

    /**
     * Get imported count
     */
    public function getImportedCount(): int
    {
        return $this->importedCount;
    }

    /**
     * Get updated count
     */
    public function getUpdatedCount(): int
    {
        return $this->updatedCount;
    }

    /**
     * Get skipped count
     */
    public function getSkippedCount(): int
    {
        return $this->skippedCount;
    }

    /**
     * Get import session ID
     */
    public function getImportSession(): string
    {
        return $this->importSession;
    }

    /**
     * Get import progress from cache
     */
    public static function getImportProgress(string $sessionId): ?array
    {
        return Cache::get("import_session_{$sessionId}");
    }

    /**
     * Mark import as completed
     */
    public function markAsCompleted(): void
    {
        if ($this->importSession && !$this->previewMode) {
            $currentData = Cache::get("import_session_{$this->importSession}", []);
            $currentData['status'] = 'completed';
            $currentData['completed_at'] = now();
            Cache::put("import_session_{$this->importSession}", $currentData, 3600);
        }
    }

    /**
     * Mark import as failed
     */
    public function markAsFailed(string $error): void
    {
        if ($this->importSession && !$this->previewMode) {
            $currentData = Cache::get("import_session_{$this->importSession}", []);
            $currentData['status'] = 'failed';
            $currentData['error'] = $error;
            $currentData['failed_at'] = now();
            Cache::put("import_session_{$this->importSession}", $currentData, 3600);
        }
    }

    /**
     * Get detailed import summary
     */
    public function getImportSummary(): array
    {
        return [
            'session_id' => $this->importSession,
            'imported' => $this->importedCount,
            'updated' => $this->updatedCount,
            'skipped' => $this->skippedCount,
            'total_processed' => $this->importedCount + $this->updatedCount + $this->skippedCount,
            'errors' => count($this->failures()),
            'success_rate' => $this->calculateSuccessRate(),
            'preview_mode' => $this->previewMode,
            'preview_data' => $this->previewMode ? $this->previewData : null
        ];
    }

    /**
     * Calculate success rate
     */
    protected function calculateSuccessRate(): float
    {
        $total = $this->importedCount + $this->updatedCount + $this->skippedCount;
        if ($total === 0) {
            return 0;
        }
        
        $successful = $this->importedCount + $this->updatedCount;
        return round(($successful / $total) * 100, 2);
    }

    /**
     * Enhanced column mapping with better Arabic support
     */
    protected function getEnhancedColumnMapping(): array
    {
        return [
            'name' => [
                'الاسم', 'اسم_المنتج', 'اسم المنتج', 'name', 'product_name',
                'alasm_alaarby', 'اسم_المنتج_بالعربية', 'المنتج'
            ],
            'name_en' => [
                'الاسم_بالإنجليزية', 'الاسم بالإنجليزية', 'name_en', 'english_name',
                'alasm_alanglyzy', 'اسم_انجليزي', 'الاسم_الانجليزي'
            ],
            'description' => [
                'الوصف', 'وصف_المنتج', 'وصف المنتج', 'description', 'alosf',
                'تفاصيل', 'الوصف_التفصيلي'
            ],
            'sku' => [
                'الرمز_sku', 'الرمز (SKU)', 'sku', 'product_code', 'alrmz_sku',
                'كود_المنتج', 'رمز_المنتج', 'الكود', 'الرمز'
            ],
            'barcode' => [
                'الباركود', 'باركود', 'barcode', 'albarkod', 'الرمز_الشريطي',
                'كود_الأعمدة', 'الرقم_الشريطي'
            ],
            'price' => [
                'السعر', 'سعر_المنتج', 'سعر المنتج', 'price', 'alsaar',
                'التكلفة', 'القيمة', 'المبلغ'
            ],
            'tax_rate' => [
                'معدل_الضريبة', 'معدل الضريبة (%)', 'tax_rate', 'نسبة_الضريبة',
                'الضريبة', 'معدل_الضريبة_المضافة'
            ],
            'stock_quantity' => [
                'كمية_المخزون', 'كمية المخزون', 'stock_quantity', 'alkmy_almtofr',
                'المخزون', 'الكمية_المتوفرة', 'العدد_المتاح'
            ],
            'min_stock' => [
                'الحد_الأدنى_للمخزون', 'الحد الأدنى للمخزون', 'min_stock', 'alhd_aladn_llkmy',
                'الحد_الأدنى', 'أقل_كمية', 'حد_التنبيه'
            ]
        ];
    }
}