<?php

namespace App\Imports;

use App\Models\Account;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Illuminate\Support\Collection;
use Illuminate\Validation\Rule;

class AccountsImport implements ToModel, WithHeadingRow, WithValidation, WithBatchInserts, WithChunkReading
{
    private $rows = 0;
    private $errors = [];

    /**
     * @param array $row
     *
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        $this->rows++;

        // التحقق من وجود الحساب الأب إذا تم تحديده
        $parentId = null;
        if (!empty($row['parent_code'])) {
            $parent = Account::where('code', $row['parent_code'])->first();
            if ($parent) {
                $parentId = $parent->id;
            } else {
                $this->errors[] = "الصف {$this->rows}: لم يتم العثور على الحساب الأب برمز {$row['parent_code']}";
                return null;
            }
        }

        // تنسيق القيم المنطقية
        $isActive = $this->parseBooleanValue($row['is_active'] ?? 'نعم');
        $isCashAccount = $this->parseBooleanValue($row['is_cash_account'] ?? 'لا');
        $isBankAccount = $this->parseBooleanValue($row['is_bank_account'] ?? 'لا');
        $isControlAccount = $this->parseBooleanValue($row['is_control_account'] ?? 'لا');
        $requiresCostCenter = $this->parseBooleanValue($row['requires_cost_center'] ?? 'لا');
        $allowManualEntries = $this->parseBooleanValue($row['allow_manual_entries'] ?? 'نعم');

        // تحديد نوع الرصيد الافتتاحي
        $openingBalanceType = 'debit';
        if (isset($row['opening_balance_type'])) {
            $balanceType = strtolower(trim($row['opening_balance_type']));
            if (in_array($balanceType, ['دائن', 'credit', 'cr'])) {
                $openingBalanceType = 'credit';
            }
        }

        // حساب الرصيد الافتتاحي بالإشارة الصحيحة
        $openingBalance = floatval($row['opening_balance'] ?? 0);
        if ($openingBalanceType === 'credit' && $openingBalance > 0) {
            $openingBalance = -$openingBalance;
        }

        try {
            return new Account([
                'code' => $row['code'],
                'name' => $row['name'],
                'type' => $this->parseAccountType($row['type']),
                'parent_id' => $parentId,
                'description' => $row['description'] ?? null,
                'is_active' => $isActive,
                'is_cash_account' => $isCashAccount,
                'is_bank_account' => $isBankAccount,
                'is_control_account' => $isControlAccount,
                'requires_cost_center' => $requiresCostCenter,
                'allow_manual_entries' => $allowManualEntries,
                'opening_balance' => $openingBalance,
                'current_balance' => $openingBalance,
                'created_by' => auth()->id(),
            ]);
        } catch (\Exception $e) {
            $this->errors[] = "الصف {$this->rows}: خطأ في إنشاء الحساب - " . $e->getMessage();
            return null;
        }
    }

    /**
     * تحويل قيمة نصية إلى قيمة منطقية
     */
    private function parseBooleanValue($value)
    {
        if (is_bool($value)) {
            return $value;
        }

        $value = strtolower(trim($value));
        return in_array($value, ['نعم', 'yes', 'true', '1', 'صحيح']);
    }

    /**
     * تحويل نوع الحساب من النص العربي إلى الإنجليزي
     */
    private function parseAccountType($type)
    {
        $type = strtolower(trim($type));
        
        $typeMapping = [
            'أصول' => 'asset',
            'اصول' => 'asset',
            'asset' => 'asset',
            'assets' => 'asset',
            
            'خصوم' => 'liability',
            'liability' => 'liability',
            'liabilities' => 'liability',
            
            'حقوق الملكية' => 'equity',
            'حقوق ملكية' => 'equity',
            'equity' => 'equity',
            
            'إيرادات' => 'revenue',
            'ايرادات' => 'revenue',
            'revenue' => 'revenue',
            'revenues' => 'revenue',
            'income' => 'revenue',
            
            'مصروفات' => 'expense',
            'مصاريف' => 'expense',
            'expense' => 'expense',
            'expenses' => 'expense',
        ];

        return $typeMapping[$type] ?? 'asset';
    }

    /**
     * قواعد التحقق من صحة البيانات
     */
    public function rules(): array
    {
        return [
            'code' => [
                'required',
                'string',
                'max:20',
                Rule::unique('accounts', 'code')
            ],
            'name' => 'required|string|max:255',
            'type' => 'required|string',
            'opening_balance' => 'nullable|numeric',
        ];
    }

    /**
     * رسائل خطأ مخصصة
     */
    public function customValidationMessages()
    {
        return [
            'code.required' => 'رمز الحساب مطلوب',
            'code.unique' => 'رمز الحساب موجود مسبقاً',
            'code.max' => 'رمز الحساب لا يجب أن يتجاوز 20 حرف',
            'name.required' => 'اسم الحساب مطلوب',
            'name.max' => 'اسم الحساب لا يجب أن يتجاوز 255 حرف',
            'type.required' => 'نوع الحساب مطلوب',
            'opening_balance.numeric' => 'الرصيد الافتتاحي يجب أن يكون رقم',
        ];
    }

    /**
     * حجم الدفعة لمعالجة البيانات
     */
    public function batchSize(): int
    {
        return 100;
    }

    /**
     * حجم القطعة لقراءة البيانات
     */
    public function chunkSize(): int
    {
        return 100;
    }

    /**
     * الحصول على عدد الصفوف المعالجة
     */
    public function getRowCount(): int
    {
        return $this->rows;
    }

    /**
     * الحصول على الأخطاء
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * إرشادات أعمدة الملف المطلوبة
     */
    public static function getRequiredColumns(): array
    {
        return [
            'code' => 'رمز الحساب (مطلوب)',
            'name' => 'اسم الحساب (مطلوب)', 
            'type' => 'نوع الحساب (أصول/خصوم/حقوق الملكية/إيرادات/مصروفات)',
            'parent_code' => 'رمز الحساب الأب (اختياري)',
            'description' => 'الوصف (اختياري)',
            'opening_balance' => 'الرصيد الافتتاحي (اختياري)',
            'opening_balance_type' => 'نوع الرصيد (مدين/دائن - افتراضي: مدين)',
            'is_active' => 'نشط (نعم/لا - افتراضي: نعم)',
            'is_cash_account' => 'حساب نقدي (نعم/لا - افتراضي: لا)',
            'is_bank_account' => 'حساب بنكي (نعم/لا - افتراضي: لا)',
            'is_control_account' => 'حساب تحكم (نعم/لا - افتراضي: لا)',
            'requires_cost_center' => 'يتطلب مركز تكلفة (نعم/لا - افتراضي: لا)',
            'allow_manual_entries' => 'السماح بالقيود اليدوية (نعم/لا - افتراضي: نعم)',
        ];
    }

    /**
     * إنشاء ملف Excel نموذجي للاستيراد
     */
    public static function generateTemplate()
    {
        $headers = array_keys(self::getRequiredColumns());
        
        $sampleData = [
            [
                'code' => '1000',
                'name' => 'الأصول',
                'type' => 'أصول',
                'parent_code' => '',
                'description' => 'حساب رئيسي للأصول',
                'opening_balance' => '0',
                'opening_balance_type' => 'مدين',
                'is_active' => 'نعم',
                'is_cash_account' => 'لا',
                'is_bank_account' => 'لا',
                'is_control_account' => 'نعم',
                'requires_cost_center' => 'لا',
                'allow_manual_entries' => 'لا',
            ],
            [
                'code' => '1001',
                'name' => 'النقدية في الصندوق',
                'type' => 'أصول',
                'parent_code' => '1000',
                'description' => 'النقدية الموجودة في الصندوق',
                'opening_balance' => '10000',
                'opening_balance_type' => 'مدين',
                'is_active' => 'نعم',
                'is_cash_account' => 'نعم',
                'is_bank_account' => 'لا',
                'is_control_account' => 'لا',
                'requires_cost_center' => 'لا',
                'allow_manual_entries' => 'نعم',
            ],
        ];

        return [
            'headers' => $headers,
            'sample_data' => $sampleData,
        ];
    }
}