<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use App\Helpers\AccountingHelper;

class StoreAccountRequest extends FormRequest
{
    /**
     * تحديد ما إذا كان المستخدم مخول لتقديم هذا الطلب
     */
    public function authorize(): bool
    {
        return auth()->check() && (
            auth()->user()->is_admin || 
            auth()->user()->hasPermission('create_accounts')
        );
    }

    /**
     * الحصول على قواعد التحقق المطبقة على الطلب
     */
    public function rules(): array
    {
        return [
            'code' => [
                'required',
                'string',
                'max:' . config('accounting.constants.max_account_code_length', 20),
                'regex:/^[0-9]+$/',
                Rule::unique('accounts', 'code'),
                function ($attribute, $value, $fail) {
                    $errors = AccountingHelper::validateAccountCode($value, $this->input('type'));
                    if (!empty($errors)) {
                        $fail(implode(', ', $errors));
                    }
                },
            ],
            'name' => [
                'required',
                'string',
                'max:' . config('accounting.constants.max_account_name_length', 255),
                function ($attribute, $value, $fail) {
                    if (!config('accounting.chart_of_accounts.allow_duplicate_names', false)) {
                        $existing = \App\Models\Account::where('name', $value)->first();
                        if ($existing) {
                            $fail('اسم الحساب موجود مسبقاً');
                        }
                    }
                },
            ],
            'type' => [
                'required',
                'string',
                Rule::in(array_keys(config('accounting.chart_of_accounts.account_types', []))),
            ],
            'parent_id' => [
                'nullable',
                'exists:accounts,id',
                function ($attribute, $value, $fail) {
                    if ($value) {
                        $parent = \App\Models\Account::find($value);
                        if ($parent && $parent->type !== $this->input('type')) {
                            $fail('نوع الحساب الأب يجب أن يطابق نوع الحساب الفرعي');
                        }
                        
                        // التحقق من عدم إنشاء دائرة مغلقة
                        if ($this->wouldCreateCircularReference($value)) {
                            $fail('لا يمكن تعيين حساب أب يؤدي إلى مرجع دائري');
                        }
                    }
                },
            ],
            'description' => [
                'nullable',
                'string',
                'max:' . config('accounting.constants.max_description_length', 1000),
            ],
            'opening_balance' => [
                'nullable',
                'numeric',
                'between:-999999999.99,999999999.99',
            ],
            'is_active' => 'boolean',
            'is_cash_account' => [
                'boolean',
                function ($attribute, $value, $fail) {
                    if ($value && $this->input('is_bank_account')) {
                        $fail('لا يمكن أن يكون الحساب نقدي وبنكي في نفس الوقت');
                    }
                    
                    if ($value && $this->input('type') !== 'asset') {
                        $fail('الحساب النقدي يجب أن يكون من نوع الأصول');
                    }
                },
            ],
            'is_bank_account' => [
                'boolean',
                function ($attribute, $value, $fail) {
                    if ($value && $this->input('is_cash_account')) {
                        $fail('لا يمكن أن يكون الحساب نقدي وبنكي في نفس الوقت');
                    }
                    
                    if ($value && $this->input('type') !== 'asset') {
                        $fail('الحساب البنكي يجب أن يكون من نوع الأصول');
                    }
                },
            ],
            'is_control_account' => [
                'boolean',
                function ($attribute, $value, $fail) {
                    if ($value && $this->input('allow_manual_entries')) {
                        $fail('حساب التحكم لا يسمح بالقيود اليدوية');
                    }
                },
            ],
            'requires_cost_center' => 'boolean',
            'allow_manual_entries' => [
                'boolean',
                function ($attribute, $value, $fail) {
                    if (!$value && $this->input('is_control_account')) {
                        $fail('حساب التحكم لا يسمح بالقيود اليدوية');
                    }
                },
            ],
        ];
    }

    /**
     * الحصول على رسائل الخطأ المخصصة
     */
    public function messages(): array
    {
        return [
            'code.required' => 'رمز الحساب مطلوب',
            'code.unique' => 'رمز الحساب موجود مسبقاً',
            'code.regex' => 'رمز الحساب يجب أن يحتوي على أرقام فقط',
            'code.max' => 'رمز الحساب لا يجب أن يتجاوز :max رقم',
            
            'name.required' => 'اسم الحساب مطلوب',
            'name.max' => 'اسم الحساب لا يجب أن يتجاوز :max حرف',
            
            'type.required' => 'نوع الحساب مطلوب',
            'type.in' => 'نوع الحساب غير صحيح',
            
            'parent_id.exists' => 'الحساب الأب غير موجود',
            
            'description.max' => 'الوصف لا يجب أن يتجاوز :max حرف',
            
            'opening_balance.numeric' => 'الرصيد الافتتاحي يجب أن يكون رقم',
            'opening_balance.between' => 'الرصيد الافتتاحي يجب أن يكون بين :min و :max',
        ];
    }

    /**
     * الحصول على أسماء الحقول المخصصة
     */
    public function attributes(): array
    {
        return [
            'code' => 'رمز الحساب',
            'name' => 'اسم الحساب',
            'type' => 'نوع الحساب',
            'parent_id' => 'الحساب الأب',
            'description' => 'الوصف',
            'opening_balance' => 'الرصيد الافتتاحي',
            'is_active' => 'حالة النشاط',
            'is_cash_account' => 'حساب نقدي',
            'is_bank_account' => 'حساب بنكي',
            'is_control_account' => 'حساب تحكم',
            'requires_cost_center' => 'يتطلب مركز تكلفة',
            'allow_manual_entries' => 'السماح بالقيود اليدوية',
        ];
    }

    /**
     * إعداد البيانات للتحقق
     */
    protected function prepareForValidation(): void
    {
        // تنسيق رمز الحساب
        if ($this->has('code')) {
            $this->merge([
                'code' => AccountingHelper::formatAccountCode($this->input('code'))
            ]);
        }

        // تحويل القيم المنطقية
        $booleanFields = [
            'is_active', 'is_cash_account', 'is_bank_account', 
            'is_control_account', 'requires_cost_center', 'allow_manual_entries'
        ];

        foreach ($booleanFields as $field) {
            if ($this->has($field)) {
                $this->merge([
                    $field => filter_var($this->input($field), FILTER_VALIDATE_BOOLEAN)
                ]);
            }
        }

        // تحويل الرصيد الافتتاحي
        if ($this->has('opening_balance')) {
            $this->merge([
                'opening_balance' => AccountingHelper::parseAmount($this->input('opening_balance'))
            ]);
        }
    }

    /**
     * التحقق من عدم إنشاء مرجع دائري
     */
    private function wouldCreateCircularReference($parentId, $depth = 0): bool
    {
        // حماية من اللولا اللانهائية
        if ($depth > 10) {
            return true;
        }

        $parent = \App\Models\Account::find($parentId);
        if (!$parent || !$parent->parent_id) {
            return false;
        }

        // إذا كان الحساب الأب الحالي هو نفس الحساب المراد إنشاؤه
        if ($this->has('id') && $parent->parent_id == $this->input('id')) {
            return true;
        }

        // التحقق من المستوى التالي
        return $this->wouldCreateCircularReference($parent->parent_id, $depth + 1);
    }

    /**
     * إجراءات إضافية بعد التحقق
     */
    public function passedValidation(): void
    {
        // تحديد القيم الافتراضية
        $this->merge([
            'is_active' => $this->input('is_active', true),
            'is_cash_account' => $this->input('is_cash_account', false),
            'is_bank_account' => $this->input('is_bank_account', false),
            'is_control_account' => $this->input('is_control_account', false),
            'requires_cost_center' => $this->input('requires_cost_center', false),
            'allow_manual_entries' => $this->input('allow_manual_entries', true),
            'opening_balance' => $this->input('opening_balance', 0),
            'current_balance' => $this->input('opening_balance', 0),
            'created_by' => auth()->id(),
        ]);

        // تطبيق قواعد إضافية بناءً على النوع
        $this->applyTypeSpecificRules();
    }

    /**
     * تطبيق قواعد خاصة بنوع الحساب
     */
    private function applyTypeSpecificRules(): void
    {
        $accountType = $this->input('type');
        
        // إذا كان حساب تحكم، منع القيود اليدوية
        if ($this->input('is_control_account')) {
            $this->merge(['allow_manual_entries' => false]);
        }

        // إذا كان من نوع المصروفات، قد يحتاج مركز تكلفة
        if ($accountType === 'expense' && config('accounting.cost_centers.require_for_expenses')) {
            $this->merge(['requires_cost_center' => true]);
        }

        // تحديد الرصيد الطبيعي للحساب
        $accountTypeInfo = AccountingHelper::getAccountTypeInfo($accountType);
        if ($accountTypeInfo) {
            $normalBalance = $accountTypeInfo['normal_balance'];
            $openingBalance = $this->input('opening_balance', 0);
            
            // إذا كان النوع دائن والمبلغ موجب، جعله سالب
            if ($normalBalance === 'credit' && $openingBalance > 0) {
                $this->merge([
                    'opening_balance' => -$openingBalance,
                    'current_balance' => -$openingBalance,
                ]);
            }
        }
    }
}