<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Cache;
use App\Models\DeviceInitializationLog;

class CheckZatcaAccess
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // التحقق من تفعيل نظام ZATCA
        if (!config('zatca.enabled', true)) {
            // إذا كان طلب AJAX، أرجع JSON
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'نظام ZATCA غير متاح حالياً'
                ], 503);
            }
            
            // إذا كان هناك route dashboard، وإلا ارجع للصفحة الرئيسية
            $redirectRoute = route_exists('dashboard') ? 'dashboard' : '/';
            return redirect($redirectRoute)
                ->with('error', 'نظام ZATCA غير متاح حالياً');
        }

        // التحقق من القائمة البيضاء للـ IP إذا كانت مفعلة
        if (config('zatca.security.ip_whitelist_enabled', false)) {
            $allowedIps = config('zatca.security.allowed_ips', []);
            
            if (!empty($allowedIps) && !in_array($request->ip(), $allowedIps)) {
                return response()->json([
                    'success' => false,
                    'message' => 'غير مصرح لك بالوصول لهذه الخدمة من عنوان IP الحالي'
                ], 403);
            }
        }

        // التحقق من معدل الطلبات للمستخدم
        if ($this->isRateLimited($request)) {
            return response()->json([
                'success' => false,
                'message' => 'تم تجاوز الحد الأقصى للطلبات. يرجى المحاولة لاحقاً'
            ], 429);
        }

        // تسجيل محاولة الوصول
        $this->logAccess($request);

        return $next($request);
    }

    /**
     * التحقق من معدل الطلبات
     */
    protected function isRateLimited(Request $request): bool
    {
        // تخطي Rate Limiting في البيئة المحلية أو إذا كان معطل
        if (app()->environment('local') || config('zatca.device_initialization.rate_limit_per_hour', 10) <= 0) {
            return false;
        }

        $maxAttempts = config('zatca.device_initialization.rate_limit_per_hour', 10);
        $decayMinutes = 60; // ساعة واحدة

        $key = 'zatca_access_' . $request->ip();
        $attempts = Cache::get($key, 0);

        if ($attempts >= $maxAttempts) {
            return true;
        }

        Cache::put($key, $attempts + 1, now()->addMinutes($decayMinutes));
        return false;
    }

    /**
     * تسجيل محاولة الوصول
     */
    protected function logAccess(Request $request): void
    {
        if (config('zatca.logging.log_access', true)) {
            \Log::info('ZATCA Access Attempt', [
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'route' => $request->route()?->getName(),
                'timestamp' => now()->toISOString(),
            ]);
        }
    }
}