<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckAccountingPermissions
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, ...$permissions): Response
    {
        // التحقق من تسجيل الدخول
        if (!auth()->check()) {
            return redirect()->route('login')
                ->with('error', 'يجب تسجيل الدخول للوصول للنظام المحاسبي');
        }

        $user = auth()->user();

        // إذا كان المستخدم مدير، السماح بالوصول لكل شيء
        if ($user->is_admin) {
            return $next($request);
        }

        // التحقق من الصلاحيات المطلوبة
        if (!empty($permissions)) {
            foreach ($permissions as $permission) {
                if (!$this->hasAccountingPermission($user, $permission)) {
                    return response()->view('errors.403', [
                        'message' => 'ليس لديك صلاحية للوصول لهذه الصفحة'
                    ], 403);
                }
            }
        }

        return $next($request);
    }

    /**
     * التحقق من صلاحية محاسبية محددة
     */
    private function hasAccountingPermission($user, $permission)
    {
        // قائمة الصلاحيات المحاسبية
        $accountingPermissions = [
            'view_accounts' => 'عرض الحسابات',
            'create_accounts' => 'إنشاء الحسابات',
            'edit_accounts' => 'تعديل الحسابات',
            'delete_accounts' => 'حذف الحسابات',
            
            'view_journal_entries' => 'عرض القيود',
            'create_journal_entries' => 'إنشاء القيود',
            'edit_journal_entries' => 'تعديل القيود',
            'delete_journal_entries' => 'حذف القيود',
            'approve_journal_entries' => 'اعتماد القيود',
            
            'view_cash_transactions' => 'عرض المعاملات النقدية',
            'create_cash_transactions' => 'إنشاء المعاملات النقدية',
            'edit_cash_transactions' => 'تعديل المعاملات النقدية',
            'delete_cash_transactions' => 'حذف المعاملات النقدية',
            
            'view_financial_reports' => 'عرض التقارير المالية',
            'export_financial_reports' => 'تصدير التقارير المالية',
            
            'view_cost_centers' => 'عرض مراكز التكلفة',
            'create_cost_centers' => 'إنشاء مراكز التكلفة',
            'edit_cost_centers' => 'تعديل مراكز التكلفة',
            'delete_cost_centers' => 'حذف مراكز التكلفة',
            
            'view_bank_accounts' => 'عرض الحسابات البنكية',
            'create_bank_accounts' => 'إنشاء الحسابات البنكية',
            'edit_bank_accounts' => 'تعديل الحسابات البنكية',
            'delete_bank_accounts' => 'حذف الحسابات البنكية',
            
            'import_data' => 'استيراد البيانات',
            'export_data' => 'تصدير البيانات',
        ];

        // إذا لم تكن الصلاحية موجودة، رفض الوصول
        if (!array_key_exists($permission, $accountingPermissions)) {
            return false;
        }

        // للتطبيق البسيط، نفترض أن المستخدمين العاديين لديهم صلاحيات العرض فقط
        $viewOnlyPermissions = [
            'view_accounts',
            'view_journal_entries',
            'view_cash_transactions',
            'view_financial_reports',
            'view_cost_centers',
            'view_bank_accounts',
        ];

        // صلاحيات المحاسبين (يمكن تطويرها لاحقاً مع نظام أدوار متقدم)
        $accountantPermissions = [
            'view_accounts',
            'create_accounts',
            'edit_accounts',
            'view_journal_entries',
            'create_journal_entries',
            'edit_journal_entries',
            'view_cash_transactions',
            'create_cash_transactions',
            'edit_cash_transactions',
            'view_financial_reports',
            'export_financial_reports',
            'view_cost_centers',
            'view_bank_accounts',
            'export_data',
        ];

        // التحقق بناءً على نوع المستخدم
        if (isset($user->role)) {
            switch ($user->role) {
                case 'accountant':
                case 'محاسب':
                    return in_array($permission, $accountantPermissions);
                    
                case 'financial_manager':
                case 'مدير_مالي':
                    return true; // المدير المالي له جميع الصلاحيات
                    
                case 'viewer':
                case 'مشاهد':
                    return in_array($permission, $viewOnlyPermissions);
                    
                default:
                    return in_array($permission, $viewOnlyPermissions);
            }
        }

        // إذا لم يكن للمستخدم دور محدد، السماح بالعرض فقط
        return in_array($permission, $viewOnlyPermissions);
    }

    /**
     * التحقق من صلاحية الوصول لمركز تكلفة محدد
     */
    public static function canAccessCostCenter($user, $costCenterId)
    {
        // المدير يمكنه الوصول لجميع مراكز التكلفة
        if ($user->is_admin) {
            return true;
        }

        // يمكن تطوير هذا لربط المستخدمين بمراكز تكلفة محددة
        // مثلاً: return $user->cost_centers()->where('id', $costCenterId)->exists();
        
        return true; // مؤقتاً، السماح للجميع
    }

    /**
     * التحقق من صلاحية اعتماد القيود
     */
    public static function canApproveJournalEntries($user)
    {
        return $user->is_admin || 
               (isset($user->role) && in_array($user->role, ['financial_manager', 'مدير_مالي']));
    }

    /**
     * التحقق من صلاحية حذف القيود المعتمدة
     */
    public static function canDeleteApprovedEntries($user)
    {
        return $user->is_admin;
    }

    /**
     * التحقق من صلاحية تعديل الأرصدة الافتتاحية
     */
    public static function canEditOpeningBalances($user)
    {
        return $user->is_admin || 
               (isset($user->role) && in_array($user->role, ['financial_manager', 'مدير_مالي']));
    }

    /**
     * التحقق من صلاحية استيراد البيانات
     */
    public static function canImportData($user)
    {
        return $user->is_admin || 
               (isset($user->role) && in_array($user->role, ['financial_manager', 'مدير_مالي', 'accountant', 'محاسب']));
    }

    /**
     * الحصول على قائمة الصلاحيات المتاحة للمستخدم
     */
    public static function getUserPermissions($user)
    {
        if ($user->is_admin) {
            return 'جميع الصلاحيات';
        }

        $permissions = [];
        $allPermissions = [
            'view_accounts', 'create_accounts', 'edit_accounts', 'delete_accounts',
            'view_journal_entries', 'create_journal_entries', 'edit_journal_entries', 'delete_journal_entries', 'approve_journal_entries',
            'view_cash_transactions', 'create_cash_transactions', 'edit_cash_transactions', 'delete_cash_transactions',
            'view_financial_reports', 'export_financial_reports',
            'view_cost_centers', 'create_cost_centers', 'edit_cost_centers', 'delete_cost_centers',
            'view_bank_accounts', 'create_bank_accounts', 'edit_bank_accounts', 'delete_bank_accounts',
            'import_data', 'export_data'
        ];

        $middleware = new self();
        foreach ($allPermissions as $permission) {
            if ($middleware->hasAccountingPermission($user, $permission)) {
                $permissions[] = $permission;
            }
        }

        return $permissions;
    }
}