<?php

namespace App\Http\Controllers;

use App\Services\ZatcaOperationsService;
use Illuminate\Http\Request;
use App\Models\ZatcaSettings;
use App\Models\Company;
use App\Models\DeviceInitializationLog;
use App\Services\ZatcaService;
use Illuminate\Support\Facades\Log;

class ZatcaSettingsController extends Controller
{
    /**
     * عرض إعدادات ZATCA
     */
    public function index()
    {
        $companies = Company::active()->get();
        $zatcaSettings = ZatcaSettings::with('company')->get();

        return view('zatca-settings.index', compact('companies', 'zatcaSettings'));
    }

    /**
     * عرض نموذج إنشاء إعدادات جديدة
     */
    public function create()
    {
        $companies = Company::active()->whereDoesntHave('zatcaSettings')->get();

        if ($companies->isEmpty()) {
            return redirect()->route('zatca-settings.index')
                ->with('error', 'جميع الشركات لديها إعدادات ZATCA');
        }

        return view('zatca-settings.create', compact('companies'));
    }

    /**
     * حفظ إعدادات جديدة
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id|unique:zatca_settings,company_id',
            'connection_type' => 'required|in:clearance,reporting',
            'environment' => 'required|in:sandbox,production',
            'client_id' => 'nullable|string',
            'client_secret' => 'nullable|string',
            'username' => 'nullable|string',
            'password' => 'nullable|string',
            'certificate_content' => 'nullable|string',
            'private_key_content' => 'nullable|string',
            'certificate_expires_at' => 'nullable|date',
        ]);

        $data = $request->all();
        $data['invoice_counter'] = 0;
        $data['is_active'] = false;
        $data['connection_status'] = 'disconnected';

        ZatcaSettings::create($data);

        return redirect()->route('zatca-settings.index')
            ->with('success', 'تم إنشاء إعدادات ZATCA بنجاح');
    }

    /**
     * عرض تفاصيل الإعدادات
     */
    public function show(ZatcaSettings $zatcaSetting)
    {
        $zatcaSetting->load('company');
        return view('zatca-settings.show', compact('zatcaSetting'));
    }

    /**
     * عرض نموذج تعديل الإعدادات
     */
    public function edit(ZatcaSettings $zatcaSetting)
    {
        $zatcaSetting->load('company');
        return view('zatca-settings.edit', compact('zatcaSetting'));
    }

    /**
     * تحديث الإعدادات
     */
    public function update(Request $request, ZatcaSettings $zatcaSetting)
    {
        $request->validate([
            'connection_type' => 'required|in:clearance,reporting',
            'environment' => 'required|in:sandbox,production',
            'client_id' => 'nullable|string',
            'client_secret' => 'nullable|string',
            'username' => 'nullable|string',
            'password' => 'nullable|string',
            'certificate_content' => 'nullable|string',
            'private_key_content' => 'nullable|string',
            'certificate_expires_at' => 'nullable|date',
            'is_active' => 'boolean',
        ]);

        // إذا تم تعطيل الإعدادات، تغيير حالة الاتصال
        if (!$request->is_active) {
            $data = $request->all();
            $data['connection_status'] = 'disconnected';
            $data['last_connection_test'] = null;
        } else {
            $data = $request->except(['is_active']);
        }

        $zatcaSetting->update($data);

        return redirect()->route('zatca-settings.index')
            ->with('success', 'تم تحديث إعدادات ZATCA بنجاح');
    }

    /**
     * حذف الإعدادات
     */
    public function destroy(ZatcaSettings $zatcaSetting)
    {
        // التحقق من عدم وجود فواتير مرتبطة
        if ($zatcaSetting->company->invoices()->count() > 0) {
            return redirect()->route('zatca-settings.index')
                ->with('error', 'لا يمكن حذف الإعدادات لوجود فواتير مرتبطة بالشركة');
        }

        $zatcaSetting->delete();

        return redirect()->route('zatca-settings.index')
            ->with('success', 'تم حذف إعدادات ZATCA بنجاح');
    }

    /**
     * اختبار الاتصال مع ZATCA
     */
    public function testConnection(Request $request)
    {
        $request->validate([
            'zatca_settings_id' => 'required|exists:zatca_settings,id',
        ]);

        try {
            $zatcaSettings = ZatcaSettings::findOrFail($request->zatca_settings_id);

            if (!$zatcaSettings->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'إعدادات ZATCA غير نشطة'
                ]);
            }

            $zatcaService = new ZatcaService($zatcaSettings);
            $result = $zatcaService->testConnection();

            if ($result['success']) {
                $zatcaSettings->update([
                    'connection_status' => 'connected',
                    'last_connection_test' => now(),
                    'last_error' => null,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'تم الاتصال بنجاح مع ZATCA',
                    'data' => $result['data'] ?? null
                ]);
            } else {
                $zatcaSettings->update([
                    'connection_status' => 'failed',
                    'last_connection_test' => now(),
                    'last_error' => $result['message'],
                ]);

                return response()->json([
                    'success' => false,
                    'message' => $result['message']
                ]);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطأ في الاتصال: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * تجديد الرمز المميز
     */
    public function refreshToken(ZatcaSettings $zatcaSetting)
    {
        try {
            $zatcaService = new ZatcaService($zatcaSetting);
            $result = $zatcaService->refreshAccessToken();

            if ($result['success']) {
                $zatcaSetting->update([
                    'access_token' => $result['access_token'],
                    'refresh_token' => $result['refresh_token'] ?? $zatcaSetting->refresh_token,
                    'token_expires_at' => $result['expires_at'],
                    'last_error' => null,
                ]);

                return back()->with('success', 'تم تجديد الرمز المميز بنجاح');
            } else {
                return back()->with('error', 'فشل في تجديد الرمز المميز: ' . $result['message']);
            }

        } catch (\Exception $e) {
            return back()->with('error', 'خطأ في تجديد الرمز المميز: ' . $e->getMessage());
        }
    }

    /**
     * إعادة تعيين العداد
     */
    public function resetCounter(ZatcaSettings $zatcaSetting)
    {
        $zatcaSetting->update(['invoice_counter' => 0]);

        return back()->with('success', 'تم إعادة تعيين العداد إلى الصفر');
    }

    /**
     * تصدير الإعدادات
     */
    public function export(ZatcaSettings $zatcaSetting)
    {
        $data = $zatcaSetting->toArray();

        // إزالة البيانات الحساسة من التصدير
        unset($data['client_secret'], $data['password'], $data['private_key_content']);

        $filename = 'zatca_settings_' . $zatcaSetting->company->name . '_' . date('Y-m-d') . '.json';

        return response()->json($data)
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * عرض لوحة تحكم ZATCA
     */
    public function dashboard()
    {
        // جمع الإحصائيات
        $stats = [
            'companies' => [
                'total' => Company::count(),
                'active' => Company::where('status', 'active')->count(),
                'with_zatca' => Company::whereHas('zatcaSettings')->count(),
            ],
            'zatca_settings' => [
                'total' => ZatcaSettings::count(),
                'active' => ZatcaSettings::where('is_active', true)->count(),
                'connected' => ZatcaSettings::where('connection_status', 'connected')->count(),
                'failed' => ZatcaSettings::where('connection_status', 'failed')->count(),
            ],
            'device_initialization' => [
                'total_attempts' => DeviceInitializationLog::count(),
                'verified' => DeviceInitializationLog::where('verification_status', 'verified')->count(),
                'pending' => DeviceInitializationLog::where('verification_status', 'pending')->count(),
                'failed' => DeviceInitializationLog::where('verification_status', 'failed')->count(),
                'today' => DeviceInitializationLog::whereDate('created_at', today())->count(),
            ],
            'certificates' => [
                'total' => ZatcaSettings::whereNotNull('certificate_content')->count(),
                'expired' => ZatcaSettings::where('certificate_expires_at', '<', now())->count(),
                'expiring_soon' => ZatcaSettings::whereBetween('certificate_expires_at', [now(), now()->addDays(30)])->count(),
                'valid' => ZatcaSettings::where('certificate_expires_at', '>', now()->addDays(30))->count(),
            ],
        ];

        // حساب معدل النجاح
        $totalAttempts = $stats['device_initialization']['total_attempts'];
        $successRate = $totalAttempts > 0
            ? round(($stats['device_initialization']['verified'] / $totalAttempts) * 100)
            : 0;

        // النشاط الأخير
        $recentActivities = $this->getRecentActivities();

        return view('zatca-settings.dashboard', compact('stats', 'successRate', 'recentActivities'));
    }

    /**
     * عرض صفحة تهيئة الجهاز الجديد
     */
    public function deviceInitialization()
    {
        $companies = Company::active()->get();
        return view('zatca-settings.device-initialization', compact('companies'));
    }

    /**
     * عرض سجلات تهيئة الأجهزة
     */
    public function initializationLogs()
    {
        $logs = DeviceInitializationLog::with('company')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // حساب الإحصائيات
        $stats = [
            'verified' => DeviceInitializationLog::where('verification_status', 'verified')->count(),
            'pending' => DeviceInitializationLog::where('verification_status', 'pending')->count(),
            'failed' => DeviceInitializationLog::where('verification_status', 'failed')->count(),
            'expired' => DeviceInitializationLog::where('verification_status', 'expired')->count(),
        ];

        return view('zatca-settings.initialization-logs', compact('logs', 'stats'));
    }

    /**
     * عرض تفاصيل سجل تهيئة محدد
     */
    public function showInitializationLog(DeviceInitializationLog $log)
    {
        $log->load('company');

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $log->id,
                'company' => [
                    'name' => $log->company->name,
                    'vat_number' => $log->company->vat_number,
                ],
                'csid_number' => $log->csid_number,
                'verification_status' => $log->verification_status,
                'otp_requested_at' => $log->otp_requested_at?->format('Y-m-d H:i:s'),
                'otp_expires_at' => $log->otp_expires_at?->format('Y-m-d H:i:s'),
                'otp_verified_at' => $log->otp_verified_at?->format('Y-m-d H:i:s'),
                'ip_address' => $log->ip_address,
                'user_agent' => $log->user_agent,
                'request_data' => $log->request_data,
                'response_data' => $log->response_data,
                'error_message' => $log->error_message,
                'created_at' => $log->created_at->format('Y-m-d H:i:s'),
            ]
        ]);
    }

    /**
     * طلب رمز التفعيل OTP للمرحلة الثانية من الفوترة الإلكترونية
     */
    public function requestOTP(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'csid_number' => 'required|string|min:10',
        ]);

        try {
            $company = Company::findOrFail($request->company_id);

            // التحقق من وجود إعدادات ZATCA للشركة
            $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();

            if (!$zatcaSettings) {
                return response()->json([
                    'success' => false,
                    'message' => 'لا توجد إعدادات ZATCA للشركة المختارة. يرجى إنشاء إعدادات أولاً.'
                ], 400);
            }

            // إنشاء سجل جديد لطلب OTP
            $initLog = DeviceInitializationLog::create([
                'company_id' => $company->id,
                'csid_number' => $request->csid_number,
                'otp_requested_at' => now(),
                'otp_expires_at' => now()->addMinutes(5),
                'verification_status' => 'pending',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'request_data' => $request->only(['company_id', 'csid_number']),
            ]);

            // محاكاة طلب OTP من ZATCA
            $otpData = $this->simulateZatcaOTPRequest($request->csid_number, $company);

            if ($otpData['success']) {
                // تحديث السجل بمعلومات النجاح
                $initLog->update([
                    'otp_token' => $otpData['otp_token'],
                    'response_data' => $otpData,
                ]);

                // حفظ معلومات طلب OTP في الجلسة
                session([
                    'zatca_otp_request' => [
                        'log_id' => $initLog->id,
                        'company_id' => $company->id,
                        'csid_number' => $request->csid_number,
                        'otp_token' => $otpData['otp_token'],
                        'expires_at' => $initLog->otp_expires_at,
                        'created_at' => now()
                    ]
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'تم إرسال رمز التفعيل بنجاح إلى رقم الجوال المسجل',
                    'data' => [
                        'expires_in' => 300, // 5 minutes
                        'log_id' => $initLog->id
                    ]
                ]);
            } else {
                // تحديث السجل بمعلومات الفشل
                $initLog->updateVerificationStatus('failed', $otpData['message']);

                return response()->json([
                    'success' => false,
                    'message' => $otpData['message']
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Error requesting OTP: ' . $e->getMessage(), [
                'company_id' => $request->company_id ?? null,
                'csid_number' => $request->csid_number ?? null,
                'user_ip' => $request->ip(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء طلب رمز التفعيل. يرجى المحاولة مرة أخرى.'
            ], 500);
        }
    }

    /**
     * التحقق من رمز OTP وإتمام تهيئة الجهاز
     */
    public function verifyOTP(Request $request)
    {
        $request->validate([
            'otp_code' => 'required|string|size:6|regex:/^[0-9]{6}$/',
        ]);

        try {
            // استرجاع بيانات طلب OTP من الجلسة
            $company = Company::findOrFail($request->company_id);

            // التحقق من وجود إعدادات ZATCA للشركة
            $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();

            if (!$zatcaSettings) {
                return response()->json([
                    'success' => false,
                    'message' => 'لا توجد إعدادات ZATCA للشركة المختارة. يرجى إنشاء إعدادات أولاً.'
                ], 400);
            }

            $initLog = DeviceInitializationLog::create([
                'company_id' => $company->id,
                'csid_number' => '',
                'otp_requested_at' => now(),
                'otp_expires_at' => now()->addMinutes(5),
                'verification_status' => 'pending',
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'request_data' => $request->only(['company_id', 'otp_code']),
            ]);

            $company = Company::findOrFail($request->company_id);
            $zatcaController = new ZatcaOperationsService();
            $jsonArray =  $zatcaController->register_device_api($company,$request->otp_code);
//            Log::error($deviceData);
//            $deviceData = stripslashes($deviceData);
//            $deviceData = trim($deviceData);
//            $deviceData = preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $deviceData); // Remove hidden characters

           // $jsonArray = json_decode($deviceData, true);
            Log::error($jsonArray);
            if (true) {
                // تحديث سجل التهيئة بالنجاح
                $initLog->update([
                    'otp_verified_at' => now(),
                    'verification_status' => 'verified',
                ]);

                // تحديث إعدادات ZATCA بالشهادة الجديدة

                $zatcaSettings = ZatcaSettings::where('company_id', $company->id)->first();

                if ($zatcaSettings) {
                    $zatcaSettings->update([
                        'certificate' => $jsonArray['signature'],
                        'private_key' => $jsonArray['private_key'],
                        'connection_status' => 'connected',
                        'is_active' => true,
                        'last_connection_test' => now(),
                        'last_error' => null,
                        'public_key' => $jsonArray['public_key'],
                        'access_token' => $jsonArray['token'],
                        'client_secret' => $jsonArray['secret'],
                        'certificate_serial' => $jsonArray['SN'],
                        'username' => $jsonArray['issuer']
                    ]);
                }

                // مسح بيانات OTP من الجلسة
                session()->forget('zatca_otp_request');

                Log::info('Device initialization completed successfully', [
                    'company_id' => $company->id,
                    'company_name' => $company->name,
                    'log_id' => $initLog->id,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'تم تسجيل الجهاز بنجاح',
                    'data' => [
                        'company_name' => $company->name,
                        'certificate_expires_at' => '',
                        'log_id' => $initLog->id
                    ]
                ]);
            } else {
                // تحديث سجل التهيئة بالفشل
                $initLog->updateVerificationStatus('failed', '');

                return response()->json([
                    'success' => false,
                    'message' => ''
                ], 400);
            }

        } catch (\Exception $e) {
            Log::error('Error verifying OTP: ' . $e, [
                'otp_code' => $request->otp_code,
                'user_ip' => $request->ip(),
                'session_data' => session('zatca_otp_request'),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التحقق من رمز التفعيل. يرجى المحاولة مرة أخرى.'
            ], 500);
        }
    }

    /**
     * محاكاة طلب OTP من ZATCA (للأغراض التوضيحية)
     * في التطبيق الحقيقي، سيتم استدعاء API الخاص بـ ZATCA
     */
    private function simulateZatcaOTPRequest($csidNumber, $company)
    {
        // محاكاة تأخير الشبكة
        usleep(500000); // 0.5 ثانية

        // التحقق من صحة رقم الشهادة (محاكاة)
        if (strlen($csidNumber) < 20) {
            return [
                'success' => false,
                'message' => 'رقم شهادة CSID غير صالح. يرجى التحقق من الرقم والمحاولة مرة أخرى.'
            ];
        }

        // محاكاة نجاح الطلب 90% من الوقت
        if (rand(1, 10) <= 9) {
            return [
                'success' => true,
                'otp_token' => 'otp_' . uniqid() . '_' . $company->id,
                'message' => 'تم إرسال رمز التفعيل بنجاح'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'فشل في إرسال رمز التفعيل. يرجى التحقق من اتصال الإنترنت والمحاولة مرة أخرى.'
            ];
        }
    }

    /**
     * محاكاة التحقق من OTP مع ZATCA (للأغراض التوضيحية)
     * في التطبيق الحقيقي، سيتم استدعاء API الخاص بـ ZATCA
     */
    private function simulateZatcaOTPVerification($otpCode, $otpToken, $csidNumber)
    {
        // محاكاة تأخير الشبكة
        usleep(800000); // 0.8 ثانية

        // كود تجريبي للاختبار
        if ($otpCode === '123456') {
            return [
                'success' => true,
                'certificate' => $this->generateMockCertificate(),
                'private_key' => $this->generateMockPrivateKey(),
                'expires_at' => now()->addYear(),
                'message' => 'تم التحقق بنجاح'
            ];
        }

        // محاكاة نجاح التحقق 80% من الوقت للرموز الأخرى
        if (rand(1, 10) <= 8) {
            return [
                'success' => true,
                'certificate' => $this->generateMockCertificate(),
                'private_key' => $this->generateMockPrivateKey(),
                'expires_at' => now()->addYear(),
                'message' => 'تم التحقق بنجاح'
            ];
        } else {
            $errors = [
                'رمز التفعيل غير صحيح',
                'انتهت صلاحية رمز التفعيل',
                'تم استخدام رمز التفعيل مسبقاً',
                'خطأ في الاتصال مع خدمات ZATCA'
            ];

            return [
                'success' => false,
                'message' => $errors[array_rand($errors)]
            ];
        }
    }

    /**
     * إنشاء شهادة تجريبية للاختبار
     */
    private function generateMockCertificate()
    {
        return "-----BEGIN CERTIFICATE-----\n" .
               "MIICljCCAX4CAQAwDQYJKoZIhvcNAQENBQAwFzEVMBMGA1UEAwwMTW9jayBDQSBD\n" .
               "ZXJ0MB4XDTI0MDEwMTAwMDAwMFoXDTI1MDEwMTAwMDAwMFowFzEVMBMGA1UEAwwM\n" .
               "TW9jayBDZXJ0aWZpY2F0ZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB\n" .
               "AL8xb9+YQHl2F3k9JH8dY2cP5x7J2kL9mN6oP8qR5sT3uV7wX9zA2bC4dE5fG6hI\n" .
               "jK7lM8nO9pQ0qR1sT2uV3wX4yZ6oP7qR8sT9uV0wX5zA6bC7dE8fG9hIjK4lM5nO\n" .
               "2pQ3qR4sT5uV6wX1yZ9oP4qR1sT6uV3wX2zA9bC1dE2fG3hIjK1lM2nO5pQ6qR7s\n" .
               "T8uV9wX5yZ2oP1qR4sT1uV0wX8zA3bC8dE5fG0hIjK8lM8nO8pQ9qR0sT9uV2wX3\n" .
               "yZ5oP8qR7sT0uV5wX6zA0bC5dE8fG5hIjK5lM5nO5pQ2qR5sT2uV5wX0yZ8oP5qR\n" .
               "0sCAwEAAaAAMA0GCSqGSIb3DQEBCwUAA4IBAQBtO2l6MxB4k1dF2hJ3p5Q7r8sT\n" .
               "9uV2wX1yZ6oP4qR5sT3uV7wX9zA2bC4dE5fG6hIjK7lM8nO9pQ0qR1sT2uV3wX4\n" .
               "-----END CERTIFICATE-----";
    }

    /**
     * إنشاء مفتاح خاص تجريبي للاختبار
     */
    private function generateMockPrivateKey()
    {
        return "-----BEGIN PRIVATE KEY-----\n" .
               "MIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQC/MW/fmEB5dhd5\n" .
               "PSR/HWNnD+ceydpC/ZjeqD/KkebE97le8F/cwNmwuHROXxuoSIyu5TPJzvaUNKkd\n" .
               "bE9rld8F+MmeqD+6kfLE/bldMF+cwOmwu3RPHxvYSIyu5TPJztqUN6keLK+blewF\n" .
               "9cmfaD+KkdbE+rld8F9swPWwtXRNnxt4SIytZTNpzuaUOqke7E/Llfcf+ckdqD9a\n" .
               "keLkdbldMF/MwN2wvHROXxtISIyvJTPJzvaUPakdLE/bldcf+cmemD+6ke7E9Lle\n" .
               "cF+swNGwunRPHxuYSIylZTOZzuaUNqkeLE9rlfcf+ckemD+6kdLA\n" .
               "-----END PRIVATE KEY-----";
    }

    /**
     * الحصول على النشاط الأخير
     */
    protected function getRecentActivities(): array
    {
        $activities = [];

        // آخر عمليات تهيئة الأجهزة
        $recentInitializations = DeviceInitializationLog::with('company')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        foreach ($recentInitializations as $log) {
            $activities[] = [
                'title' => 'تهيئة جهاز جديد',
                'description' => "الشركة: {$log->company->name}",
                'time' => $log->created_at->diffForHumans(),
                'icon' => $this->getActivityIcon($log->verification_status),
                'color' => $this->getActivityColor($log->verification_status),
            ];
        }

        // آخر اختبارات الاتصال
        $recentConnections = ZatcaSettings::with('company')
            ->whereNotNull('last_connection_test')
            ->orderBy('last_connection_test', 'desc')
            ->limit(3)
            ->get();

        foreach ($recentConnections as $setting) {
            $activities[] = [
                'title' => 'اختبار اتصال',
                'description' => "الشركة: {$setting->company->name}",
                'time' => $setting->last_connection_test->diffForHumans(),
                'icon' => $setting->connection_status === 'connected' ? 'check' : 'times',
                'color' => $setting->connection_status === 'connected' ? '#27ae60' : '#e74c3c',
            ];
        }

        // ترتيب النشاطات حسب الوقت
        usort($activities, function($a, $b) {
            return strcmp($b['time'], $a['time']);
        });

        return array_slice($activities, 0, 8); // أحدث 8 أنشطة
    }

    /**
     * الحصول على أيقونة النشاط حسب الحالة
     */
    protected function getActivityIcon(string $status): string
    {
        return match($status) {
            'verified' => 'check-circle',
            'pending' => 'clock',
            'failed' => 'times-circle',
            'expired' => 'hourglass-end',
            default => 'info-circle'
        };
    }

    /**
     * الحصول على لون النشاط حسب الحالة
     */
    protected function getActivityColor(string $status): string
    {
        return match($status) {
            'verified' => '#27ae60',
            'pending' => '#f39c12',
            'failed' => '#e74c3c',
            'expired' => '#95a5a6',
            default => '#3498db'
        };
    }
}
