<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\PurchaseInvoice;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class TaxReportController extends Controller
{
    /**
     * Display tax reports dashboard
     */
    public function index()
    {
        return view('system-reports.tax.index');
    }

    /**
     * VAT summary report
     */
    public function vatSummary(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        // Sales VAT (Output VAT)
        $salesVAT = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(subtotal) as total_before_tax,
                SUM(tax_amount) as output_vat,
                SUM(total_amount) as total_with_tax
            ')
            ->first();

        // Purchase VAT (Input VAT)
        $purchaseVAT = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->selectRaw('
                COUNT(*) as total_invoices,
                SUM(subtotal) as total_before_tax,
                SUM(tax_amount) as input_vat,
                SUM(total_amount) as total_with_tax
            ')
            ->first();

        // Net VAT calculation
        $netVAT = [
            'output_vat' => $salesVAT->output_vat ?? 0,
            'input_vat' => $purchaseVAT->input_vat ?? 0,
            'net_vat_due' => ($salesVAT->output_vat ?? 0) - ($purchaseVAT->input_vat ?? 0)
        ];

        return view('system-reports.tax.vat-summary', compact('salesVAT', 'purchaseVAT', 'netVAT', 'startDate', 'endDate'));
    }

    /**
     * VAT return preparation
     */
    public function vatReturn(Request $request)
    {
        $quarter = $request->get('quarter', ceil(now()->month / 3));
        $year = $request->get('year', now()->year);
        
        $startDate = Carbon::create($year, ($quarter - 1) * 3 + 1, 1)->startOfMonth();
        $endDate = Carbon::create($year, $quarter * 3, 1)->endOfMonth();

        // Standard rated supplies (15%) - calculate based on tax_amount/subtotal ratio
        $standardSupplies = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->whereRaw('(tax_amount / NULLIF(subtotal, 0)) * 100 BETWEEN 14 AND 16') // 15% ±1% tolerance
            ->selectRaw('
                SUM(subtotal) as total_supplies,
                SUM(tax_amount) as vat_amount
            ')
            ->first();

        // Zero rated supplies - where tax_amount is 0 but subtotal is not 0
        $zeroSupplies = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->where('tax_amount', 0)
            ->where('subtotal', '>', 0)
            ->selectRaw('
                SUM(subtotal) as total_supplies,
                SUM(tax_amount) as vat_amount
            ')
            ->first();

        // Exempt supplies - where both tax_amount and subtotal could be handled differently
        // For now, we'll consider invoices with 0 tax and 0 subtotal as exempt
        $exemptSupplies = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->where('tax_amount', 0)
            ->where('subtotal', 0)
            ->selectRaw('
                SUM(subtotal) as total_supplies
            ')
            ->first();

        // Input VAT
        $inputVAT = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->selectRaw('
                SUM(subtotal) as total_purchases,
                SUM(tax_amount) as input_vat
            ')
            ->first();

        $vatReturnData = [
            'standard_supplies' => $standardSupplies->total_supplies ?? 0,
            'standard_vat' => $standardSupplies->vat_amount ?? 0,
            'zero_supplies' => $zeroSupplies->total_supplies ?? 0,
            'exempt_supplies' => $exemptSupplies->total_supplies ?? 0,
            'total_supplies' => ($standardSupplies->total_supplies ?? 0) + ($zeroSupplies->total_supplies ?? 0) + ($exemptSupplies->total_supplies ?? 0),
            'output_vat' => $standardSupplies->vat_amount ?? 0,
            'input_vat' => $inputVAT->input_vat ?? 0,
            'net_vat' => ($standardSupplies->vat_amount ?? 0) - ($inputVAT->input_vat ?? 0)
        ];

        return view('system-reports.tax.vat-return', compact('vatReturnData', 'quarter', 'year', 'startDate', 'endDate'));
    }

    /**
     * Tax transactions report
     */
    public function transactions(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());
        $type = $request->get('type', 'all'); // all, sales, purchases

        $salesTransactions = [];
        $purchaseTransactions = [];

        if ($type === 'all' || $type === 'sales') {
            $salesTransactions = Invoice::with('customer')
                ->whereBetween('issue_date', [$startDate, $endDate])
                ->select('id', 'invoice_number', 'customer_id', 'issue_date', 'subtotal', 'tax_amount', 'total_amount')
                ->orderByDesc('issue_date')
                ->get()
                ->map(function($invoice) {
                    $taxRate = $invoice->subtotal > 0 ? ($invoice->tax_amount / $invoice->subtotal) * 100 : 0;
                    return [
                        'type' => 'مبيعات',
                        'number' => $invoice->invoice_number,
                        'party' => $invoice->customer->name ?? 'غير محدد',
                        'date' => $invoice->issue_date,
                        'subtotal' => $invoice->subtotal,
                        'vat_rate' => round($taxRate, 1),
                        'vat_amount' => $invoice->tax_amount,
                        'total' => $invoice->total_amount
                    ];
                });
        }

        if ($type === 'all' || $type === 'purchases') {
            $purchaseTransactions = PurchaseInvoice::with('supplier')
                ->whereBetween('invoice_date', [$startDate, $endDate])
                ->select('id', 'invoice_number', 'supplier_id', 'invoice_date', 'subtotal', 'tax_amount', 'total_amount')
                ->orderByDesc('invoice_date')
                ->get()
                ->map(function($invoice) {
                    $taxRate = $invoice->subtotal > 0 ? ($invoice->tax_amount / $invoice->subtotal) * 100 : 0;
                    return [
                        'type' => 'مشتريات',
                        'number' => $invoice->invoice_number,
                        'party' => $invoice->supplier->name ?? 'غير محدد',
                        'date' => $invoice->invoice_date,
                        'subtotal' => $invoice->subtotal,
                        'vat_rate' => round($taxRate, 1),
                        'vat_amount' => $invoice->tax_amount,
                        'total' => $invoice->total_amount
                    ];
                });
        }

        $transactions = collect($salesTransactions)
            ->merge($purchaseTransactions)
            ->sortByDesc('date');

        return view('system-reports.tax.transactions', compact('transactions', 'startDate', 'endDate', 'type'));
    }

    /**
     * Tax by customer report
     */
    public function byCustomer(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        $customerTax = DB::table('customers')
            ->leftJoin('invoices', 'customers.id', '=', 'invoices.customer_id')
            ->whereBetween('invoices.issue_date', [$startDate, $endDate])
            ->selectRaw('
                customers.id,
                customers.name,
                customers.vat_number,
                COUNT(invoices.id) as total_invoices,
                SUM(invoices.subtotal) as total_before_tax,
                SUM(invoices.tax_amount) as total_tax,
                SUM(invoices.total_amount) as total_with_tax
            ')
            ->groupBy('customers.id', 'customers.name', 'customers.vat_number')
            ->orderByDesc('total_tax')
            ->paginate(15);

        return view('system-reports.tax.by-customer', compact('customerTax', 'startDate', 'endDate'));
    }

    /**
     * Tax audit trail
     */
    public function auditTrail(Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth());
        $endDate = $request->get('end_date', now()->endOfMonth());

        // All tax-related changes and adjustments
        $auditData = collect();

        // Sales invoices
        $salesAudit = Invoice::whereBetween('issue_date', [$startDate, $endDate])
            ->select('id', 'invoice_number', 'issue_date', 'tax_amount', 'created_at', 'updated_at')
            ->get()
            ->map(function($invoice) {
                return [
                    'type' => 'فاتورة مبيعات',
                    'reference' => $invoice->invoice_number,
                    'date' => $invoice->issue_date,
                    'tax_amount' => $invoice->tax_amount,
                    'created_at' => $invoice->created_at,
                    'updated_at' => $invoice->updated_at
                ];
            });

        // Purchase invoices
        $purchaseAudit = PurchaseInvoice::whereBetween('invoice_date', [$startDate, $endDate])
            ->select('id', 'invoice_number', 'invoice_date', 'tax_amount', 'created_at', 'updated_at')
            ->get()
            ->map(function($invoice) {
                return [
                    'type' => 'فاتورة مشتريات',
                    'reference' => $invoice->invoice_number,
                    'date' => $invoice->invoice_date,
                    'tax_amount' => $invoice->tax_amount,
                    'created_at' => $invoice->created_at,
                    'updated_at' => $invoice->updated_at
                ];
            });

        $auditTrail = $salesAudit->merge($purchaseAudit)->sortByDesc('created_at');

        return view('system-reports.tax.audit-trail', compact('auditTrail', 'startDate', 'endDate'));
    }

    /**
     * Export tax report
     */
    public function export(Request $request)
    {
        $type = $request->get('type', 'vat-summary');
        $format = $request->get('format', 'excel');
        
        // Implementation for export functionality
        return redirect()->back()->with('success', 'تم تصدير التقرير بنجاح');
    }
}